/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/hierarchy/rc/bs/LevelElem
 */
define(["baja!"],
        function (baja) {

  "use strict";

  var subclass = baja.subclass,
      callSuper = baja.callSuper,
      objectify = baja.objectify,

      entityOrd = "entityOrd",
      childPredicate = "childPredicate",
      levelDefPath = "levelDefPath",
      hierarchyOrd = "hierarchyOrd",
      displayName = "n$3adisplayName";

  /**
   * Represents a `hierarchy:LevelElem` in BajaScript.
   *
   * @class
   * @alias module:nmodule/hierarchy/rc/bs/LevelElem
   * @extends baja.Component
   */
  var LevelElem = function LevelElem() {
    callSuper(LevelElem, this, arguments);
  };

  subclass(LevelElem, baja.Component);

  baja.registerType("hierarchy:LevelElem",
    function () {
      return LevelElem;
    });

  LevelElem.entityOrd = entityOrd;
  LevelElem.childPredicate = childPredicate;
  LevelElem.levelDefPath = levelDefPath;
  LevelElem.hierarchyOrd = hierarchyOrd;

  LevelElem.$hierarchyInit = "hierarchyInit";

  /**
   * Initialize the element with the specified hierarchy space.
   *
   * @private
   * @inner
   *
   * @param {module:nmodule/hierarchy/rc/bs/LevelElem} elem The element instance.
   * @param {module:nmodule/hierarchy/rc/bs/HierarchySpace} hierarchySpace
   * The hierarchy space for the element.
   * @param parent The parent node to use for this element.
   */
  function init(elem, hierarchySpace, parent) {
    elem.$hierarchySpace = hierarchySpace;
    parent.$addChildNode(elem);

    // Resolve the target Component and cache the result.
    var ord = elem.getContextParams().get(entityOrd, baja.Ord.DEFAULT);

    if (ord && ord !== baja.Ord.DEFAULT) {
      elem.$targetOrd = baja.Ord.make({ base: "station:", child: ord });
    }
  }

  LevelElem.prototype.$fw = function (x, a, b, c, d) {
    if (x === LevelElem.$hierarchyInit) {
      init(this, a, b);
      return;
    }
    return callSuper("$fw", LevelElem, this, arguments);
  };

  /**
   * Return the entity ord of the resolved component and the level def that identifies the location
   * within the hierarchy.
   */
  LevelElem.prototype.getNavOrd = function () {
    return this.getContextParams().get(hierarchyOrd, baja.Ord.DEFAULT);
  };

  /**
   * Remote call get get ther hierarchy nav children, not necessarily the
   *  target component's nav children.
   */
  LevelElem.prototype.getNavChildren = function (obj) {
    var that = this,
        cb = new baja.comm.Callback(objectify(obj, "ok").ok);

    if (that.$navKids) {
      cb.ok(that.$navKids);
    }

    // Decode response and add the Nav Children.
    cb.addOk(function (ok, fail, resp) {
      baja.bson.importUnknownTypes(resp, function () {
        var i,
            elem;

        that.$navKids = [];

        for (i = 0; i < resp.length; ++i) {
          if (resp[i].o) {
            elem = baja.bson.decodeValue(resp[i].o, baja.$serverDecodeContext);

            if (elem && elem instanceof LevelElem) {
              // Initialize the each level element.
              elem.$fw(LevelElem.$hierarchyInit, that.$hierarchySpace, that);
            }
          }
        }

        ok(that.$navKids);
      });
    });

    // Add a BOX request to load the hierarchy children.
    cb.addReq("hierarchy", "load", {
      path: that.getLevelDefPath(),
      contextParams: that.getContextParams().encodeToString(),
      tags: that.getElemTags().encodeToString()
    });

    cb.commit();

    return cb.promise();
  };

  LevelElem.prototype.getNavParent = function () {
    return this.$navParent || null;
  };

  LevelElem.prototype.getNavName = function () {
    return this.getElemName();
  };

  /**
   * Return n:displayName tag or elemName if not set.
   */
  LevelElem.prototype.getNavDisplayName = function () {
    return this.getElemTags().get(displayName, this.getElemName());
  };

  LevelElem.prototype.getNavIcon = function () {
    return this.getElemIcon();
  };

  /**
   * Add a child node to this element.
   *
   * @param node The node to add as a child.
   */
  LevelElem.prototype.$addChildNode = function (node) {
    var that = this;
    that.$navKids = that.$navKids || [];
    node.$navParent = that;
    that.$navKids.push(node);
    return node;
  };

  /**
   * Return a reference to the hierarchy space this element
   * is bound too.
   *
   * @returns {module:nmodule/hierarchy/rc/bs/HierarchySpace}
   */
  LevelElem.prototype.getHierarchySpace = function () {
    return this.$hierarchySpace;
  };

  /**
   * Returns the level def path or a blank string if nothing
   * can be found.
   *
   * @returns {String}
   */
  LevelElem.prototype.getLevelDefPath = function () {
    return this.getContextParams().get(levelDefPath, "");
  };

  return LevelElem;
});
