/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/hierarchy/rc/bs/HierarchySpace
 */
define(["baja!",
        "Promise",
        "nmodule/hierarchy/rc/bs/LevelElem"],
        function (
        baja,
        Promise,
        LevelElem) {

  "use strict";

  var subclass = baja.subclass,
      callSuper = baja.callSuper,
      objectify = baja.objectify;

  /**
   * Represents a `hierarchy:HierarchySpace` in BajaScript.
   *
   * @class
   * @alias module:nmodule/hierarchy/rc/bs/HierarchySpace
   * @extends baja.NavContainer
   */
  var HierarchySpace = function HierarchySpace() {
    callSuper(HierarchySpace, this, [{
      navName: "hierarchy",
      icon: "module://icons/x16/r2/tree.png",
      ord: "local:|hierarchy:"
    }]);
  };

  subclass(HierarchySpace, baja.NavContainer);

  baja.registerType("hierarchy:HierarchySpace", function () {
    return HierarchySpace;
  });

  HierarchySpace.prototype.getNavChildren = function (obj) {
    var that = this,
        cb = new baja.comm.Callback(objectify(obj, "ok").ok);

    // Decode response and add the Nav Children.
    cb.addOk(function (ok, fail, resp) {
      baja.bson.importUnknownTypes(resp, function () {
        var i,
            elem;

        that.$navKids = [];

        for (i = 0; i < resp.length; ++i) {
          if (resp[i].o) {
            elem = baja.bson.decodeValue(resp[i].o, baja.$serverDecodeContext);

            if (elem) {
              // Initialize the each level element.
              elem.$fw(LevelElem.$hierarchyInit, that, that);
            }
          }
        }

        ok(that.$navKids);
      });
    });

    // Add a BOX request for the route of the Space.
    cb.addReq("hierarchy", "load", {});
    cb.commit();

    return cb.promise();
  };

  HierarchySpace.prototype.getAbsoluteOrd = function () {
    return this.getNavOrd();
  };

  /**
  * Determine whether an ord is a hierarchy ord: scheme of the first and only query is "hierarchy"
  *
  * @param {String|baja.Ord|Object} ord
  * @returns {boolean}
  */
  HierarchySpace.isHierarchyOrd = function(ord) {
    try {
      var queries = baja.Ord.make(ord).parse();
      return queries.size() === 1 && queries.get(0).getSchemeName() === "hierarchy";
    }
    catch (ignore) {
      return false;
    }
  };

  /**
   * Resolve the hierarchy ord to a level elem on the server.  Usually, a hierarchy ord is resolved
   * to the component that it points to, such as a folder or point.  However, the children of the
   * resolved component may not be the same as the children in the hierarchy because of the queries
   * in subsequent level definitions.  Web chart utilizes this functionality.
   *
   * @param {String|baja.Ord|Object} hierarchyOrd
   * @returns {LevelElem} hierarchy level element
   */
  HierarchySpace.resolveHierarchyLevelElem = function(ord) {
    if (!ord) {
      return Promise.reject(new Error("Hierarchy ord is not defined"));
    }

    var cb = new baja.comm.Callback();

    // decode response to a LevelElem
    cb.addOk(function (ok, fail, resp) {
      baja.bson.importUnknownTypes(resp, function () {
        var elem = baja.bson.decodeValue(resp.o, baja.$serverDecodeContext);
        if (!elem.getType().is("hierarchy:LevelElem")) {
          fail(new Error("Hierarchy resolve response is not of type LevelElem"));
          return;
        }
        ok(elem);
      }, fail);
    });

    // Add a BOX request for the route of the Space.
    ord = baja.Ord.make(ord.toString()).normalize();
    cb.addReq("hierarchy", "resolve", {ord: ord.toString()});
    cb.commit();

    return cb.promise();
  };

  return HierarchySpace;
});
