/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author JJ Frankovich
 */

/**
 * API Status: **Private**
 * @module nmodule/hierarchy/rc/bs/HierarchyChartFactory
 */
define([ //'Promise',
         'baja!',
         'nmodule/webChart/rc/model/BaseModel',
         'nmodule/webChart/rc/model/BaseChartFactory',
         'nmodule/hierarchy/rc/bs/LevelElem',
         'nmodule/webChart/rc/model/ServletSeries',
         'nmodule/webChart/rc/model/seriesFactory'],
function (baja,
          BaseModel,
          BaseChartFactory,
          LevelElem,
          ServletSeries,
          seriesFactory) {

  "use strict";

  /**
   * The factory for TransformGraph which generates SeriesTransformSeries for a hierarchy LevelElem
   *
   * @class
   * @extends module:nmodule/webChart/rc/model/BaseChartFactory
   * @alias module:nmodule/hierarchy/rc/bs/HierarchyChartFactory
   */
  var HierarchyChartFactory = function() {
    BaseChartFactory.apply(this, arguments);
  };

  HierarchyChartFactory.prototype = new BaseChartFactory();
  HierarchyChartFactory.prototype.constructor = HierarchyChartFactory;

  /**
   * If the level elem has no nav children (it is a leaf), call seriesFactory.make on the entity the level elem
   * represents. Otherwise, run all of the level elem children that are also level elems through seriesFactory.make,
   * which will return to this factory to check each grandchild level elem.
   *
   * @param {module:nmodule/webChart/rc/model/BaseModel} model
   * @param {baja.Subscriber} subscriber
   * @param {Object} seriesParams the 'value' of this parameter is the LevelElem on which to operate
   * @param {Object} [params] This optional argument is a container for additional factory resolution
   * parameters. This params Object should just be passed through to any recursive factory calls,
   * including seriesFactory.make(). This parameter was added starting in Niagara 4.7.
   * @param {Array.<String>} params.currentSeriesOrds Collects existing and pending ORD strings
   * for series that are (or will be) added to the chart.  This allows for limit checking to
   * skip looking for additional series when the chart's series limit is reached.
   * @returns {*}
   */
  HierarchyChartFactory.prototype.factory = function(model, subscriber, seriesParams, params)
  {
    var levelElem = seriesParams.value;

    return levelElem.getNavChildren()
      .then(function(children) {
        var newSeriesParams = [],
            i,
            ord;

        if (children.length <= 0) {
          // if the level element is a leaf and it has an entity ord, run that ord through the series factories.
          ord = levelElem.getContextParams().get(LevelElem.entityOrd, baja.Ord.DEFAULT);
          if (ord && ord !== baja.Ord.DEFAULT) {
            newSeriesParams.push({ord: ord});
          }
        } else {
          // run seriesFactory against each child; for potential performance improvement, see NCCB-19789
          for (i = 0; i < children.length; i++) {
            newSeriesParams.push({ord: children[i].getNavOrd(), value: children[i]});
          }
        }

        if (newSeriesParams.length <= 0) {
          // nothing to do
          return [];
        }

        return seriesFactory.make(model, subscriber, newSeriesParams, params);
      })
      .then(function(results) {
        if (!results) {
          return [];
        }
        return results;
      });
  };

  return HierarchyChartFactory;
});
