/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Vikram Nagulan
 */

/**
 * API Status: **Private**
 * @module nmodule/kitIo/rc/baja/TabularConversion
 */
define([
  'baja!',
  'bajaScript/baja/obj/objUtil',
  'nmodule/driver/rc/baja/XYPoint',
  'underscore'], function
  (baja,
   objUtil,
   XYPoint,
   _) {

  'use strict';

  var Simple = baja.Simple,
    cacheDecode = objUtil.cacheDecode,
    cacheEncode = objUtil.cacheEncode;

  /**
   * BajaScript representation of a `kitIo:TabularConversion` value.
   *
   * @class
   * @extends {baja.Simple}
   * @alias {module:nmodule/kitIo/rc/baja/TabularConversion}
   */
  var TabularConversion = function TabularConversion() {
    Simple.apply(this, arguments);
  };
  TabularConversion.prototype = Object.create(Simple.prototype);
  TabularConversion.prototype.constructor = TabularConversion;

  /**
   * Create a new `kitIo:TabularConversion` instance from the arguments.
   * A tabular conversion object in theory must have at least two points configured.
   * @returns {module:nmodule/kitIo/rc/baja/TabularConversion}
   */
  TabularConversion.make = function (desc, pnts) {
    if (!Array.isArray(pnts) || pnts.length < 2) {
      throw new Error('An array of at least two XYPoints is required.');
    }
    if (!desc) {
      desc = "";
    }

    var tc = new TabularConversion();

    tc.$desc = desc;
    tc.$pnts = pnts;
    return tc;
  };

  /**
   * @see .make
   * @returns {module:nmodule/kitIo/rc/baja/TabularConversion}
   */
  TabularConversion.prototype.make = function () {
    return TabularConversion.make.apply(TabularConversion, arguments);
  };

  /**
   * Decode an `TabularConversion` from a string.
   *
   * @function
   * @param {String} str
   * @returns {module:nmodule/kitIo/rc/baja/TabularConversion}
   */
  TabularConversion.prototype.decodeFromString = cacheDecode(function (str) {
    if (!str || str.split("|").length < 2) {
      return TabularConversion.DEFAULT;
    }
    var a = str.split("|");

    var pts = _.filter(a[1].split(";"), function (pt) {
      return pt !== "" && pt.split(',').length === 2;
    });
    return TabularConversion.make(baja.SlotPath.unescape(a[0]), _.map(pts, function (pt) {
      if (pt !== "") {
        return XYPoint.decodeFromString(pt);
      }
      return null;
    }));
  });

  /**
   * Encode this `TabularConversion` to a string of the form "<desc>|<pnts[0];pnts[1];...>".
   *
   * @function
   * @returns {String}
   */
  TabularConversion.prototype.encodeToString = cacheEncode(function () {
    return baja.SlotPath.escape(this.$desc) + "|" + _.map(this.$pnts, function (point) {
        return point.encodeToString();
      }).join(";");
  });

  /**
   * Get the XYPoint objects configured for this conversion
   * @returns {Array} Array of XYPoint
   */
  TabularConversion.prototype.getPoints = function () {
    return this.$pnts.slice();
  };
  /**
   * Get the description of this conversion object
   * @returns {*|string}
   */
  TabularConversion.prototype.getDescription = function () {
    return this.$desc;
  };

  TabularConversion.DEFAULT = TabularConversion.make("Linear", [new XYPoint({
    x: 0,
    y: 0
  }), new XYPoint({x: 100, y: 100})]);
  /**
   * Register this baja type as 'kitIo:TabularConversion'
   */
  baja.registerType('kitIo:TabularConversion', function () {
    return TabularConversion;
  });
  return TabularConversion;
});
