/*
 * Copyright 2001 Tridium, Inc. All Rights Reserved.
 */
package com.tridium.workbench.fieldeditors;

import java.lang.reflect.Method;
import java.util.Enumeration;
import javax.baja.naming.BOrd;
import javax.baja.sys.BComponent;
import javax.baja.sys.BObject;
import javax.baja.sys.BString;
import javax.baja.sys.Context;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import javax.baja.ui.BTextDropDown;
import javax.baja.util.Version;
import javax.baja.workbench.BWbShell;
import javax.baja.workbench.fieldeditor.BWbFieldEditor;

import com.tridium.crypto.core.cert.CertUtils;
import com.tridium.crypto.core.cert.KeyPurpose;
import com.tridium.crypto.core.cert.NX509CertificateEntry;
import com.tridium.crypto.core.io.CryptoStoreId;
import com.tridium.fox.sys.BFoxChannel;
import com.tridium.fox.sys.BFoxSession;
import com.tridium.sys.schema.Fw;

/**
 * BCertificateAliasFE allows selection of certificate aliases installed on the JACE.
 *
 * @author    Bill Smith       
 * @creation  13 Dec 11
 * @version   $Revision: Original$
 * @since     Baja 3.7
 */
public class BCertificateAliasFE
  extends BWbFieldEditor
{ 

////////////////////////////////////////////////////////////////
// Type
////////////////////////////////////////////////////////////////

  @Override
  public Type getType() { return TYPE; }
  public static final Type TYPE = Sys.loadType(BCertificateAliasFE.class);

////////////////////////////////////////////////////////////////
// Constructor
////////////////////////////////////////////////////////////////

  @SuppressWarnings("unused")
  public BCertificateAliasFE()
  {    
    this(CryptoStoreId.USER_KEY_STORE);
  }

  public BCertificateAliasFE(CryptoStoreId id)
  {
    setContent(field);
    linkTo("lk0", field, BTextDropDown.valueModified, setModified);
    linkTo("lk1", field, BTextDropDown.actionPerformed, actionPerformed);

    this.id = id;
  }

////////////////////////////////////////////////////////////////
// BWbEditor
////////////////////////////////////////////////////////////////

  @Override
  protected void doSetReadonly(boolean readonly)
  {
    field.getEditor().setEditable(!readonly);
    field.setDropDownEnabled(!readonly);
  }  

  @Override
  protected void doLoadValue(BObject v, Context cx)
  { 
    BString alias = (BString) v;
    field.setText(alias.getString());
    field.getList().removeAllItems();

    KeyPurpose keyPurpose = null;
    BString purposeId = (BString) cx.getFacet("purposeId");
    if (purposeId != null && !purposeId.getString().isEmpty())
    {
      keyPurpose = KeyPurpose.valueOf(purposeId.getString());
    }

    try
    {
      BWbShell shell = getWbShell();
      BOrd ord = shell.getActiveOrd();
      BComponent service = (BComponent) ord.get();

      Version version = (Version) service.fw(Fw.GET_REMOTE_VERSION, "web", null, null, null);
      if (version.compareTo(BCertificateAliasFE.version) < 0) { return; }
            
      BFoxSession session = (BFoxSession) service.getSession();
      BFoxChannel channel = session.getConnection().getChannels().get("crypto", Sys.getType("platCrypto:CryptoChannel"));
           
      Type type = Sys.getType("platCrypto:CryptoChannel");
      
      Class<?> cls = type.getTypeClass();
      Method mthd = cls.getMethod("keyStoreGetCertificates", CryptoStoreId.class);

      @SuppressWarnings("unchecked")
      Enumeration<NX509CertificateEntry> certs = (Enumeration<NX509CertificateEntry>) mthd.invoke(channel, id);

      while (certs.hasMoreElements())
      {
        NX509CertificateEntry entry = certs.nextElement();
        if (filterCert(entry, id, keyPurpose))
        {
          field.getList().addItem(entry.getAlias());
        }
      }
    }
    catch(Exception ignore)
    {}
  }

  /**
   * Return true if the certificate should be included for the specified
   * CryptoStoreId and KeyPurpose, false if it should be excluded
   *
   * @param entry the certificate entry
   * @param id the KeyPurpose to filter on
   * @param keyPurpose the CryptoStoreId
   * @return true if the cert should be included, false otherwise
   * @since Niagara 4.10
   */
  private static boolean filterCert(NX509CertificateEntry entry, CryptoStoreId id, KeyPurpose keyPurpose)
  {
    // return all aliases for user trust store with no key purpose specified
    if (id == CryptoStoreId.USER_TRUST_STORE && keyPurpose == null)
    {
      return true;
    }
    keyPurpose = keyPurpose != null ? keyPurpose : KeyPurpose.SERVER_CERT;
    switch (keyPurpose)
    {
      case CLIENT_CERT:
        return CertUtils.isClientCert(entry.getCertificate(0).getCertificate());
      case CA_CERT:
        return CertUtils.isCACertificate(entry.getCertificate(0).getCertificate());
      case CODE_SIGNING_CERT:
        return CertUtils.isCodeSigningCertificate(entry.getCertificate(0).getCertificate());
      case SERVER_CERT:
      default:
        return CertUtils.isServerCertificate(entry.getCertificate(0).getCertificate());
    }
  }

  @Override
  protected BObject doSaveValue(BObject v, Context cx)
  {                
    return BString.make(field.getText());
  }
  
////////////////////////////////////////////////////////////////
// Attributes
////////////////////////////////////////////////////////////////  

  private final BTextDropDown field = new BTextDropDown();
  private static final Version version = new Version("3.7");
  private final CryptoStoreId id;
}
