/*
 * Copyright (c) 2017 Tridium, Inc. All Rights Reserved.
 */
package com.tridium.testng;

import static org.testng.Assert.*;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.function.BooleanSupplier;
import java.util.function.Supplier;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.LogRecord;

import javax.baja.file.BFileSystem;
import javax.baja.file.BIFile;
import javax.baja.file.FilePath;
import javax.baja.naming.BOrd;
import javax.baja.nre.util.FileUtil;
import javax.baja.registry.Registry;
import javax.baja.registry.TypeInfo;
import javax.baja.sys.BAbsTime;
import javax.baja.sys.BComplex;
import javax.baja.sys.BFacets;
import javax.baja.sys.Clock;
import javax.baja.sys.Property;
import javax.baja.sys.Sys;
import javax.baja.ui.BWidget;

import org.testng.Assert;

import com.tridium.nre.RunnableWithException;

/**
 * Useful methods for test classes.
 *
 * @author Eric Anderson
 * @creation 4/19/2017
 * @since Niagara 4.4
 */
public final class TestUtil
{
  // private constructor
  private TestUtil()
  {
  }

  private static final int WAIT_FOR_TRUE_INTERVAL = 10;
  private static final int WAIT_FOR_TRUE_TIMEOUT = 5000;

  /**
   * @deprecated use {@link #assertWillBeTrue(BooleanSupplier, long, String)} instead; the seconds
   * parameter value will need to be converted to milliseconds
   */
  @Deprecated
  public static void waitFor(int seconds, BooleanSupplier condition, String message)
  {
    //noinspection MagicNumber
    assertTrue(waitFor(condition, seconds * 1000L, WAIT_FOR_TRUE_INTERVAL), message + ": wait time expired");
  }

  /**
   * Tests the supplied condition at the default interval of 10 ms for the default timeout of 5000
   * ms. If the condition is not satisfied before the timeout, {@link Assert#assertTrue} will fail
   * and log the supplied message.
   *
   * @param condition condition to test
   * @param message message to include with the assertTrue if the condition is not met before the
   *                default timeout of 5000 ms
   */
  public static void assertWillBeTrue(BooleanSupplier condition, String message)
  {
    assertTrue(waitFor(condition, WAIT_FOR_TRUE_TIMEOUT, WAIT_FOR_TRUE_INTERVAL), message);
  }

  /**
   * Tests the supplied condition at the default interval of 10 ms for the default timeout of 5000
   * ms. If the condition is not satisfied before the timeout, {@link Assert#fail(String)} with the
   * message supplier.
   *
   * @param condition condition to test
   * @param message message supplier to include with the Assert.fail if the condition is not met
   *                before the default timeout of 5000 ms; evaluated once the timeout is reached
   * @since Niagara 4.11
   * @since Niagara 4.10u5
   */
  public static void assertWillBeTrue(BooleanSupplier condition, Supplier<String> message)
  {
    if (!waitFor(condition, WAIT_FOR_TRUE_TIMEOUT, WAIT_FOR_TRUE_INTERVAL))
    {
      Assert.fail(message.get());
    }
  }

  /**
   * Tests the supplied condition at the default interval of 10 ms for the specified timeout. If the
   * condition is not satisfied before the timeout, {@link Assert#assertTrue} will fail and log the
   * supplied message.
   *
   * @param condition condition to test
   * @param timeout time to wait in milliseconds for condition to be true
   * @param message message to include with the assertTrue if the condition is not met before the
   *                timeout
   */
  public static void assertWillBeTrue(BooleanSupplier condition, long timeout, String message)
  {
    assertTrue(waitFor(condition, timeout, WAIT_FOR_TRUE_INTERVAL), message);
  }

  /**
   * Tests the supplied condition at the default interval of 10 ms for the specified timeout. If the
   * condition is not satisfied before the timeout, {@link Assert#fail(String)} with the message
   * supplier.
   *
   * @param condition condition to test
   * @param timeout time to wait in milliseconds for condition to be true
   * @param message message supplier to include with the Assert.fail if the condition is not met
   *                before the default timeout of 5000 ms; evaluated once the timeout is reached
   * @since Niagara 4.11
   * @since Niagara 4.10u5
   */
  public static void assertWillBeTrue(BooleanSupplier condition, long timeout, Supplier<String> message)
  {
    if (!waitFor(condition, timeout, WAIT_FOR_TRUE_INTERVAL))
    {
      Assert.fail(message.get());
    }
  }

  /**
   * Tests the supplied condition at the default interval of 10 ms until the supplied condition is
   * true or until the default timeout of 5000 ms is reached.
   *
   * @param condition condition to test every 10 ms
   * @return true if the condition is met before the default timeout of 5000 ms
   */
  public static boolean waitFor(BooleanSupplier condition)
  {
    return waitFor(condition, WAIT_FOR_TRUE_TIMEOUT, WAIT_FOR_TRUE_INTERVAL);
  }

  /**
   * @deprecated use {@link #waitFor(BooleanSupplier, long)} instead; the timeout parameter value
   * will need to be adjusted from seconds to milliseconds.
   */
  @Deprecated
  public static boolean waitFor(BooleanSupplier condition, double timeout)
  {
    return waitFor(condition, (long)(timeout * 1000), WAIT_FOR_TRUE_INTERVAL);
  }

  /**
   * Tests the supplied condition at the default interval of 10 ms until the supplied condition is
   * true or until the specified timeout is reached.
   *
   * @param condition condition to test every 10 ms
   * @param timeout time to wait in milliseconds for condition to be true
   * @return true if the condition is met before the timeout is reached
   */
  public static boolean waitFor(BooleanSupplier condition, long timeout)
  {
    return waitFor(condition, timeout, WAIT_FOR_TRUE_INTERVAL);
  }

  /**
   * @deprecated use {@link #waitFor(BooleanSupplier, long, long)} instead; the timeout parameter
   * value will need to be adjusted from seconds to milliseconds and the interval and timeout
   * parameters must switch positions
   */
  @Deprecated
  public static boolean waitFor(BooleanSupplier condition, long interval, double timeout)
  {
    return waitFor(condition, (long)(timeout * 1000), interval);
  }

  /**
   * Tests the supplied condition at the specified interval until the supplied condition is true or
   * until the specified timeout is reached.
   *
   * @param condition condition to test
   * @param timeout time to wait in milliseconds for condition to be true
   * @param interval number of milliseconds between condition tests
   * @return true if the condition is met before the timeout is reached
   */
  public static boolean waitFor(BooleanSupplier condition, long timeout, long interval)
  {
    long now = Clock.ticks();
    long end = now + timeout;
    while (now < end)
    {
      if (condition.getAsBoolean())
      {
        return true;
      }

      try
      {
        Thread.sleep(interval);
      }
      catch (InterruptedException e)
      {
        Thread.currentThread().interrupt();
        return false;
      }

      now = Clock.ticks();
    }

    return condition.getAsBoolean();
  }

  /**
   * Waits until there has been at least one millisecond change on the {@link Clock}.
   *
   * @since Niagara 4.13
   * @since Niagara 4.12u2
   * @since Niagara 4.10u5
   */
  public static void waitForClockChange()
  {
    BAbsTime start = Clock.time();
    waitFor(() -> Clock.time().getMillis() > start.getMillis());
  }

  /**
   * Tests if the passed FE Type has a ux_field_editor in its slot facets.
   */
  public static void fieldEditorSlotFacetsToHaveUxFieldEditorDefined(String expectedFeType, String expectedUxFeType)
  {
    assertNotNull(expectedFeType, "expectedFeType argument");
    assertNotNull(expectedUxFeType, "expectedUxFeType argument");

    Registry registry = Sys.getRegistry();

    for (TypeInfo type : registry.getConcreteTypes(BComplex.TYPE.getTypeInfo()))
    {
      // skip interfaces, abstract classes and BWidgets
      if (type.isAbstract() || type.isInterface() || type.is(BWidget.TYPE))
      {
        continue;
      }

      // Types without a default constructor cannot be checked
      BComplex instance;
      try
      {
        instance = type.getInstance().asComplex();
      }
      catch (Throwable ignore)
      {
        continue;
      }

      for (Property property : instance.getProperties())
      {
        BFacets facets = property.getFacets();
        String actualFeType = facets.gets(BFacets.FIELD_EDITOR, null);
        if (expectedFeType.equals(actualFeType))
        {
          String actualUxFeType = facets.gets(BFacets.UX_FIELD_EDITOR, null);
          String message = "actual UX field editor type for " + type.getModuleName() + ':' +
            type.getTypeName() + '.' + property.getName();
          assertNotNull(actualUxFeType, message);
          assertEquals(actualUxFeType, expectedUxFeType, message);
        }
      }
    }
  }

  /**
   * You can use this log handler to spy on the last log message
   */
  public static class LatestHandler extends Handler
  {

    String latestMessage;
    Level latestLevel;

    public Level getLatestLevel()
    {
      return latestLevel;
    }

    public String getLatestMessage()
    {
      return latestMessage;
    }

    @Override
    public void publish(LogRecord record)
    {
      latestMessage = record.getMessage();
      latestLevel = record.getLevel();
    }

    @Override
    public void flush()
    {

    }

    @Override
    public void close()
    {

    }
  }

  /**
   * Retrieve the current value for a private static variable and remove the
   * final modifier so that other unit tests can change them.
   * This is useful for trying out alternate System property values for a test.
   */
  public static Object getPrivateField(Object object, String variableName)
  {
    return actOnField(object, variableName, field -> field.get(object));
  }

  public static void setPrivateField(Object object, String variableName, Object value)
  {
    actOnField(object, variableName, field -> { field.set(object, value); return null; });
  }

  private interface ExceptionFunction<T, R, E extends Exception> {
    R apply(T t) throws E;
  }

  private static <T> T actOnField(Object object,
                                  String variableName,
                                  ExceptionFunction<Field, T, IllegalAccessException> function)
  {
    Field field = null;
    boolean fieldAccessible = false;
    Field modifierField = null;
    boolean modifiedFieldAccessible = false;
    int modifiers = 0;

    try
    {
      Class<?> cls = object instanceof Class ? (Class<?>) object : object.getClass();

      field = cls.getDeclaredField(variableName);
      fieldAccessible = field.isAccessible();
      field.setAccessible(true);

      modifierField = field.getClass().getDeclaredField("modifiers");
      modifiedFieldAccessible = modifierField.isAccessible();
      modifierField.setAccessible(true);
      modifiers = modifierField.getInt(field);
      modifierField.setInt(field, field.getModifiers() & ~Modifier.FINAL);

      return function.apply(field);
    }
    catch (Exception e)
    {
      throw new RuntimeException("Cannot remove final", e);
    }
    finally
    {
      try
      {
        if (modifierField != null && field != null)
        {
          modifierField.setInt(field, modifiers);
          modifierField.setAccessible(modifiedFieldAccessible);
          field.setAccessible(fieldAccessible);
        }
      }
      catch (IllegalAccessException e)
      {
        e.printStackTrace();
      }
    }
  }



  /**
   * Run your test with an alternate static final variable, then change the value back to the given original value.
   * This is useful for trying out alternate System properties.
   */
  public static <E extends Exception> void runWithAlternateValue(Object object,
                                                                 String variableName,
                                                                 Object alternateValue,
                                                                 RunnableWithException<E> r)
    throws E
  {
    Object originalValue = getPrivateField(object, variableName);
    setPrivateField(object, variableName, alternateValue);

    try
    {
      r.run();
    }
    finally
    {
      setPrivateField(object, variableName, originalValue);
    }
  }

  /**
   * Copy a file from one location to the desired FilePath.
   */
  public static void copyFile(BOrd original, FilePath destination)
    throws IOException
  {
    BIFile inFile = (BIFile) original.get();
    BIFile outFile = BFileSystem.INSTANCE.makeFile(destination);
    try (InputStream in = inFile.getInputStream(); OutputStream out = outFile.getOutputStream())
    {
      FileUtil.pipe(in, out);
      out.flush();
    }
  }

  /**
   * Unpack the module resource to a temp file that deletes on exit.
   *
   * @param moduleFilePath the full path within the module e.g. bajaTest/resources/zip/x.zip
   * @return a reference to the temp file
   * @throws IOException if the module source could not be found, or the tmp file could not be created
   * @since Niagara 4.10u8 / 4.13u3 / 4.14
   */
  public static BIFile unpackModuleFileAsTempFile(String moduleFilePath)
    throws IOException
  {
    try
    {
      BOrd moduleOrd = BOrd.make("module://" + moduleFilePath);
      BIFile moduleFile = (BIFile)moduleOrd.get();
      File tempFile = File.createTempFile(moduleFile.getFileName(), '.' + moduleFile.getExtension());
      tempFile.deleteOnExit();

      copyFile(moduleOrd, BFileSystem.INSTANCE.localFileToPath(tempFile));

      return BFileSystem.INSTANCE.localFileToOrd(tempFile)
        .get()
        .as(BIFile.class);
    }
    catch (Exception e)
    {
      // TestNg is fairly unhelpful printing exception details so catch and release here
      System.err.println("Failed to unpack file '" + moduleFilePath + "', exception '" + e.getMessage() + '\'');
      e.printStackTrace();
      throw e;
    }
  }
}
