/*
 * Copyright 2006 Tridium, Inc. All Rights Reserved.
 */
package com.tridium.ui.test;

import java.io.*;
import java.nio.charset.StandardCharsets;
import javax.baja.sys.*;
import javax.baja.test.*;
import com.tridium.ui.*;

/**
 * Test framework for KeyboardLayout.
 *
 * @author    Andy Frank
 * @creation  16 Nov 06
 * @version   $Revision: 3$ $Date: 8/28/07 11:17:09 AM EDT$
 * @since     Niagara 3.2
 */
public class BKeyboardTest
  extends BTest
{

////////////////////////////////////////////////////////////////
// Type
////////////////////////////////////////////////////////////////

  @Override
  public Type getType() { return TYPE; }
  public static final Type TYPE = Sys.loadType(BKeyboardTest.class);

////////////////////////////////////////////////////////////////
// Test
////////////////////////////////////////////////////////////////

  public void test() throws Exception
  {
    testFail(badRoot1);   // element name
    testFail(badRoot2);   // attributes

    testFail(badRow1);    // element name
    testFail(badRow2);    // attributes

    testFail(badKey1);    // element name
    testFail(badKey2);    // children
    testFail(badKey3);    // missing 'c' and 'n' attr
    testFail(badKey4);    // missing 'n' attr
    testFail(badKey5);    // missing 'c' attr
    testFail(badKey6);    // missing 'c' and 'n' attr
    testFail(badKey7);    // invalid attr
    testFail(badKey8);    // invalid code
    testFail(badKey9);    // invalid normal char
    testFail(badKey10);   // invalid shift char
    testFail(badKey11);   // invalid alt char

    KeyboardLayout kb = make(good);

    verify(kb.getRowCount()  == 2);
    verify(kb.getKeyCount(0) == 3);
    verify(kb.getKeyCount(1) == 4);

    verify(kb.getKey(0,0), 65, 'a', 'a', '%', null);
    verify(kb.getKey(0,1), 66, 'b', 'B', 'b', null);
    verify(kb.getKey(0,2), 67, 'c', 'C', 'c', "Foo");

    verify(kb.getKey(1,0), 1, 'w', '@', 'w', null);
    verify(kb.getKey(1,1), 2, 'x', '#', 'x', null);
    verify(kb.getKey(1,2), 3, 'y', '&', 'y', null);
    verify(kb.getKey(1,3), 4, 'z', '*', '[', "Bar");
  }

  public void testEN() throws Exception
  {
    KeyboardLayout kb = KeyboardLayout.make();

    verify(kb.getRowCount() == 5);

    verify(kb.getKeyCount(0) == 14);
    verify(kb.getKey(0,0),  192, '`', '~', '`', null);
    verify(kb.getKey(0,1),  49,  '1', '!', '1', null);
    verify(kb.getKey(0,2),  50,  '2', '@', '2', null);
    verify(kb.getKey(0,3),  51,  '3', '#', '3', null);
    verify(kb.getKey(0,4),  52,  '4', '$', '4', null);
    verify(kb.getKey(0,5),  53,  '5', '%', '5', null);
    verify(kb.getKey(0,6),  54,  '6', '^', '6', null);
    verify(kb.getKey(0,7),  55,  '7', '&', '7', null);
    verify(kb.getKey(0,8),  56,  '8', '*', '8', null);
    verify(kb.getKey(0,9),  57,  '9', '(', '9', null);
    verify(kb.getKey(0,10), 58,  '0', ')', '0', null);
    verify(kb.getKey(0,11), 45,  '-', '_', '-', null);
    verify(kb.getKey(0,12), 61,  '=', '+', '=', null);
    verify(kb.getKey(0,13), 8,   (char)8, (char)8, (char)8, "Backspace");

    verify(kb.getKeyCount(1) == 14);
    verify(kb.getKey(1,0),  9,  (char)9, (char)9, (char)9, "Tab");
    verify(kb.getKey(1,1),  81, 'q',  'Q', 'q',  null);
    verify(kb.getKey(1,2),  87, 'w',  'W', 'w',  null);
    verify(kb.getKey(1,3),  69, 'e',  'E', 'e',  null);
    verify(kb.getKey(1,4),  82, 'r',  'R', 'r',  null);
    verify(kb.getKey(1,5),  84, 't',  'T', 't',  null);
    verify(kb.getKey(1,6),  89, 'y',  'Y', 'y',  null);
    verify(kb.getKey(1,7),  85, 'u',  'U', 'u',  null);
    verify(kb.getKey(1,8),  73, 'i',  'I', 'i',  null);
    verify(kb.getKey(1,9),  79, 'o',  'O', 'o',  null);
    verify(kb.getKey(1,10), 80, 'p',  'P', 'p',  null);
    verify(kb.getKey(1,11), 91, '[',  '{', '[',  null);
    verify(kb.getKey(1,12), 92, ']',  '}', ']',  null);
    verify(kb.getKey(1,13), 93, '\\', '|', '\\', null);

    verify(kb.getKeyCount(2) == 12);
    verify(kb.getKey(2,0),  65,  'a',  'A',  'a',  null);
    verify(kb.getKey(2,1),  83,  's',  'S',  's',  null);
    verify(kb.getKey(2,2),  68,  'd',  'D',  'd',  null);
    verify(kb.getKey(2,3),  70,  'f',  'F',  'f',  null);
    verify(kb.getKey(2,4),  71,  'g',  'G',  'g',  null);
    verify(kb.getKey(2,5),  72,  'h',  'H',  'h',  null);
    verify(kb.getKey(2,6),  74,  'j',  'J',  'j',  null);
    verify(kb.getKey(2,7),  75,  'k',  'K',  'k',  null);
    verify(kb.getKey(2,8),  76,  'l',  'L',  'l',  null);
    verify(kb.getKey(2,9),  59,  ';',  ':',  ';',  null);
    verify(kb.getKey(2,10), 222, '\'', '\"', '\'', null);
    verify(kb.getKey(2,11), 10,  (char)10, (char)10, (char)10, "Enter");

    verify(kb.getKeyCount(3) == 11);
    verify(kb.getKey(3,0),  90, 'z', 'Z', 'z', null);
    verify(kb.getKey(3,1),  88, 'x', 'X', 'x', null);
    verify(kb.getKey(3,2),  67, 'c', 'C', 'c', null);
    verify(kb.getKey(3,3),  86, 'v', 'V', 'v', null);
    verify(kb.getKey(3,4),  66, 'b', 'B', 'b', null);
    verify(kb.getKey(3,5),  78, 'n', 'N', 'n', null);
    verify(kb.getKey(3,6),  77, 'm', 'M', 'm', null);
    verify(kb.getKey(3,7),  44, ',', '<', ',', null);
    verify(kb.getKey(3,8),  46, '.', '>', '.', null);
    verify(kb.getKey(3,9),  47, '/', '?', '/', null);
    verify(kb.getKey(3,10), 16, (char)16, (char)16, (char)16, "Shift");

    verify(kb.getKeyCount(4) == 6);
    verify(kb.getKey(4,0), 32,  (char)32,  (char)32,  (char)32,  "Space");
    verify(kb.getKey(4,1), 38,  (char)38,  (char)38,  (char)38,  "Up");
    verify(kb.getKey(4,2), 40,  (char)40,  (char)40,  (char)40,  "Down");
    verify(kb.getKey(4,3), 37,  (char)37,  (char)37,  (char)37,  "Left");
    verify(kb.getKey(4,4), 39,  (char)39,  (char)39,  (char)39,  "Right");
    verify(kb.getKey(4,5), 127, (char)127, (char)127, (char)127, "Del");
  }

////////////////////////////////////////////////////////////////
// Support
////////////////////////////////////////////////////////////////

  private void testFail(String xml)
  {
    try
    {
      InputStream in = new ByteArrayInputStream(xml.getBytes(StandardCharsets.UTF_8));
      new KeyboardLayout().parse(in);
      fail();
    }
    catch (Exception e)
    {
      //System.out.println("# " + e.getMessage());
      verify(true);
    }
  }

  private KeyboardLayout make(String xml) throws Exception
  {
    InputStream in = new ByteArrayInputStream(xml.getBytes(StandardCharsets.UTF_8));
    KeyboardLayout kb = new KeyboardLayout();
    kb.parse(in);
    return kb;
  }

  private void verify(KeyboardLayout.Key key, int code, char n, char s, char a, String label)
  {
    verify(key.code == code);
    verify(key.normal == n);
    verify(key.shift  == s);
    verify(key.alt    == a);

    if (key.label == null) verify(label == null);
    else verify(key.label.equals(label));
  }

////////////////////////////////////////////////////////////////
// XML snippets
////////////////////////////////////////////////////////////////

  String badRoot1 = "<keybaord></keybaord>";
  String badRoot2 = "<keyboard lang='en'></keyboard>";

  String badRow1  = "<keyboard><rows></rows></keyboard>";
  String badRow2  = "<keyboard><row foo='bar'></row></keyboard>";

  String badKey1   = "<keyboard><row><kye/></row></keyboard>";
  String badKey2   = "<keyboard><row><key><foo/></key></row></keyboard>";
  String badKey3   = "<keyboard><row><key/></row></keyboard>";
  String badKey4   = "<keyboard><row><key code='A'/></row></keyboard>";
  String badKey5   = "<keyboard><row><key normal='A'/></row></keyboard>";
  String badKey6   = "<keyboard><row><key shift='A'/></row></keyboard>";
  String badKey7   = "<keyboard><row><key code='65' normal='a' v='X'/></row></keyboard>";
  String badKey8   = "<keyboard><row><key code='65' normal='abc'/></row></keyboard>";
  String badKey9   = "<keyboard><row><key code='a' normal='a'/></row></keyboard>";
  String badKey10  = "<keyboard><row><key code='a' normal='a' shift='abc'/></row></keyboard>";
  String badKey11  = "<keyboard><row><key code='a' normal='a' alt='abc'/></row></keyboard>";

  String good =
    "<keyboard>\n" +
    " <row>\n" +
    "  <key code='65' normal='a' alt='%'/>\n" +
    "  <key code='66' normal='b' shift='B' />\n" +
    "  <key code='67' normal='c' shift='C' label='Foo' />\n" +
    " </row>\n" +
    " <row>\n" +
    "  <key code='1' normal='w' shift='@' />\n" +
    "  <key code='2' normal='x' shift='#' />\n" +
    "  <key code='3' normal='y' shift='&amp;' />\n" +
    "  <key code='4' normal='z' shift='*' alt='[' label='Bar' />\n" +
    " </row>\n" +
    "</keyboard>";

}

