/*
 * Copyright 2000 Tridium, Inc. All Rights Reserved.
 */
package com.tridium.kitControl.util;

import javax.baja.sys.*;
import javax.baja.status.*;

import javax.baja.control.*;
import com.tridium.kitControl.math.*;

 /**
 *  This function performs a "reset" on the value inA.
 * "Reset" is a HVAC term for scaling a number 
 * between two limits.  
 *
 * When inputLowLimit < inA < inputHighLimit, 
 * the output value  scales linearly 
 * between outputLowLimit and outputHighLimit.  
 *
 * If inA < inputLowLimit, the value is capped
 * at outputLowLimit.  
 * 
 * If inA > inputHighLimit, the value is capped 
 * at outputHighLimit.
 * 
 * To calculate out, the following equation is used:
 * <pre>
 *       (D-B)
 *   y = ----- * (x - A) + B
 *       (C-A)  
 *
 *  where 
 *     A = inputLowLimit
 *     B = outputLowLimit
 *     C = inputHighLimit
 *     D = outputHighLimit
 *     x = inA
 *     y = out
 * </pre>
 * 
 *  The equation can be derived by solving the 
 * three linear equations below simultaneously:
 *
 *   B = mA + b
 *   D = mC + b
 *   y = mx + b
 *
 * The proof is left as an exercise to the reader...
 *
 * @author    Dan Giorgis
 * @creation  30 Aug 2001
 * @version   $Revision: 13$ $Date: 3/30/2004 3:42:29 PM$
 * @since     Baja 1.0
 */
public class BReset
  extends BMath
{ 
  /*-
  
  class BReset
  {
    properties
    {
      inA: BStatusNumeric
        -- Input A
        default {[ new BStatusNumeric(0, BStatus.nullStatus) ]}
      inputLowLimit: BStatusNumeric
        -- Input low limit
        default {[ new BStatusNumeric() ]}
      inputHighLimit: BStatusNumeric
        -- Input high limit
        default {[ new BStatusNumeric() ]}
      outputLowLimit: BStatusNumeric
        -- Output low limit
        default {[ new BStatusNumeric() ]}
      outputHighLimit: BStatusNumeric
        -- Output high limit
        default {[ new BStatusNumeric() ]}        
    }
  }
  
  -*/
/*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
/*@ $com.tridium.kitControl.util.BReset(4185886917)1.0$ @*/
/* Generated Tue May 25 13:10:32 EDT 2004 by Slot-o-Matic 2000 (c) Tridium, Inc. 2000 */

////////////////////////////////////////////////////////////////
// Property "inA"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>inA</code> property.
   * Input A
   * @see com.tridium.kitControl.util.BReset#getInA
   * @see com.tridium.kitControl.util.BReset#setInA
   */
  public static final Property inA = newProperty(0, new BStatusNumeric(0, BStatus.nullStatus),null);
  
  /**
   * Get the <code>inA</code> property.
   * @see com.tridium.kitControl.util.BReset#inA
   */
  public BStatusNumeric getInA() { return (BStatusNumeric)get(inA); }
  
  /**
   * Set the <code>inA</code> property.
   * @see com.tridium.kitControl.util.BReset#inA
   */
  public void setInA(BStatusNumeric v) { set(inA,v,null); }

////////////////////////////////////////////////////////////////
// Property "inputLowLimit"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>inputLowLimit</code> property.
   * Input low limit
   * @see com.tridium.kitControl.util.BReset#getInputLowLimit
   * @see com.tridium.kitControl.util.BReset#setInputLowLimit
   */
  public static final Property inputLowLimit = newProperty(0, new BStatusNumeric(),null);
  
  /**
   * Get the <code>inputLowLimit</code> property.
   * @see com.tridium.kitControl.util.BReset#inputLowLimit
   */
  public BStatusNumeric getInputLowLimit() { return (BStatusNumeric)get(inputLowLimit); }
  
  /**
   * Set the <code>inputLowLimit</code> property.
   * @see com.tridium.kitControl.util.BReset#inputLowLimit
   */
  public void setInputLowLimit(BStatusNumeric v) { set(inputLowLimit,v,null); }

////////////////////////////////////////////////////////////////
// Property "inputHighLimit"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>inputHighLimit</code> property.
   * Input high limit
   * @see com.tridium.kitControl.util.BReset#getInputHighLimit
   * @see com.tridium.kitControl.util.BReset#setInputHighLimit
   */
  public static final Property inputHighLimit = newProperty(0, new BStatusNumeric(),null);
  
  /**
   * Get the <code>inputHighLimit</code> property.
   * @see com.tridium.kitControl.util.BReset#inputHighLimit
   */
  public BStatusNumeric getInputHighLimit() { return (BStatusNumeric)get(inputHighLimit); }
  
  /**
   * Set the <code>inputHighLimit</code> property.
   * @see com.tridium.kitControl.util.BReset#inputHighLimit
   */
  public void setInputHighLimit(BStatusNumeric v) { set(inputHighLimit,v,null); }

////////////////////////////////////////////////////////////////
// Property "outputLowLimit"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>outputLowLimit</code> property.
   * Output low limit
   * @see com.tridium.kitControl.util.BReset#getOutputLowLimit
   * @see com.tridium.kitControl.util.BReset#setOutputLowLimit
   */
  public static final Property outputLowLimit = newProperty(0, new BStatusNumeric(),null);
  
  /**
   * Get the <code>outputLowLimit</code> property.
   * @see com.tridium.kitControl.util.BReset#outputLowLimit
   */
  public BStatusNumeric getOutputLowLimit() { return (BStatusNumeric)get(outputLowLimit); }
  
  /**
   * Set the <code>outputLowLimit</code> property.
   * @see com.tridium.kitControl.util.BReset#outputLowLimit
   */
  public void setOutputLowLimit(BStatusNumeric v) { set(outputLowLimit,v,null); }

////////////////////////////////////////////////////////////////
// Property "outputHighLimit"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>outputHighLimit</code> property.
   * Output high limit
   * @see com.tridium.kitControl.util.BReset#getOutputHighLimit
   * @see com.tridium.kitControl.util.BReset#setOutputHighLimit
   */
  public static final Property outputHighLimit = newProperty(0, new BStatusNumeric(),null);
  
  /**
   * Get the <code>outputHighLimit</code> property.
   * @see com.tridium.kitControl.util.BReset#outputHighLimit
   */
  public BStatusNumeric getOutputHighLimit() { return (BStatusNumeric)get(outputHighLimit); }
  
  /**
   * Set the <code>outputHighLimit</code> property.
   * @see com.tridium.kitControl.util.BReset#outputHighLimit
   */
  public void setOutputHighLimit(BStatusNumeric v) { set(outputHighLimit,v,null); }

////////////////////////////////////////////////////////////////
// Type
////////////////////////////////////////////////////////////////
  
  public Type getType() { return TYPE; }
  public static final Type TYPE = Sys.loadType(BReset.class);

/*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

  public void onExecute(BStatusValue o, Context cx)
  {
    BStatusNumeric inA = getInA();
    BStatus sa = inA.getStatus();                     
    BStatusNumeric out = (BStatusNumeric)o;
    BStatusNumeric inputLowLimit = getInputLowLimit();
    BStatusNumeric inputHighLimit = getInputHighLimit();
    BStatusNumeric outputLowLimit = getOutputLowLimit();
    BStatusNumeric outputHighLimit = getOutputHighLimit();

    BStatus ilStatus = inputLowLimit.getStatus();
    BStatus ihStatus = inputHighLimit.getStatus();
    BStatus olStatus = outputLowLimit.getStatus();
    BStatus ohStatus = outputHighLimit.getStatus();

    //  if any of the limits is null or in the input is
    //  null, force the output to null  
    if (ilStatus.isNull() || ihStatus.isNull() || 
        olStatus.isNull() || ohStatus.isNull() || sa.isNull())
    {
      out.setValue(Double.NaN);
      out.setStatus(BStatus.nullStatus);
      
    }
    else
    {
      // status bits for input and output high low limits    
      int il = 0; int ih = 0; int ol = 0; int oh = 0;

      il = ilStatus.getBits();
      ih = ihStatus.getBits();
      ol = olStatus.getBits();
      oh = ohStatus.getBits();
            
      double result = reset(inA.getValue(), 
                            inputLowLimit.getValue(),
                            inputHighLimit.getValue(),
                            outputLowLimit.getValue(),
                            outputHighLimit.getValue());

      //out.setStatus(BStatus.make(sa.getBits() | il | ih | ol | oh));        

      out.setValue(result);
      out.setStatus(propagate(sa));
      
    }
  }  

  private static double reset(double x, 
                              double inputLowLimit, 
                              double inputHighLimit, 
                              double outputLowLimit, 
                              double outputHighLimit)
  {
    if (x < inputLowLimit)
      return outputLowLimit;
    else if (x > inputHighLimit)
      return outputHighLimit;
    else
    {
      return (((outputHighLimit-outputLowLimit)/(inputHighLimit-inputLowLimit)
              *  (x-inputLowLimit) + outputLowLimit));     
    }
  }

  /**
   * Get the icon.
   */
  public BIcon getIcon() { return icon; }
  private static final BIcon icon = BIcon.std("control/math/reset.png");

}
