/*
 * Copyright 2002 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.driver.util;

import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Stack;
import javax.baja.sys.Action;
import javax.baja.sys.BAbsTime;
import javax.baja.sys.BBoolean;
import javax.baja.sys.BFacets;
import javax.baja.sys.BRelTime;
import javax.baja.sys.Clock;
import javax.baja.sys.Flags;
import javax.baja.sys.Property;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;

/**
 * BPollScheduler is used to configure and manage a group
 * of BComponents to be polled.  The BPollScheduler provides
 * a flexible polling algorthim based on four "buckets".
 * <p>
 * Each BIPollable assigned to a BPollScheduler is configured
 * in one of three buckets: fast, normal, or slow.  In addition
 * there is a forth bucket called the "dibs stack".  Whenever
 * a BIPollable is subscribed it immediately gets "first dibs"
 * and goes on the top of the dibs stack.  The poll scheduler
 * always polls the dibs before doing anything else.  The dibs
 * stack is polled last-in, first-out (LIFO).  As long as entries
 * are in the dibs stack they are polled as fast as possible
 * with no artificial delays.
 * <p>
 * When the dibs stack is empty the scheduler attempts to
 * poll the components in each bucket using an algorthim
 * designed to create uniform network traffic.  For example
 * if the fast rate is configured to 10sec and there are 5
 * components currently subscribed in the fast bucket, then
 * the scheduler will attempt to poll one component every 2sec.
 * <p>
 * Every ten seconds the poll scheduler rechecks the buckets
 * for configuration changes.  So if a BIPollable's configuration
 * is changed from slow to fast, it takes at most ten seconds
 * for the change to take effect.  Statistics are also updated
 * every ten seconds.  Statistics may be manually reset using
 * the resetStatistics action.
 *
 * @author    Brian Frank
 * @creation  21 Jan 02
 * @version   $Revision: 26$ $Date: 4/14/11 9:38:34 AM EDT$
 * @since     Baja 1.0
 */
public abstract class BPollScheduler
  extends BAbstractPollService
  implements Runnable
{

  /*-

  class BPollScheduler
  {
    properties
    {
      fastRate: BRelTime
        -- The frequency used to poll components set to fast.
        default {[ BRelTime.make(1000) ]}
        slotfacets {[ BFacets.make(BFacets.MIN, BRelTime.make(1), BFacets.SHOW_MILLISECONDS, BBoolean.TRUE) ]}
      normalRate: BRelTime
        -- The frequency used to poll components set to normal.
        default {[ BRelTime.make(5000) ]}
        slotfacets {[ BFacets.make(BFacets.MIN, BRelTime.make(1), BFacets.SHOW_MILLISECONDS, BBoolean.TRUE) ]}
      slowRate: BRelTime
        -- The frequency used to poll components set to slow.
        default {[ BRelTime.make(30000) ]}
        slotfacets {[ BFacets.make(BFacets.MIN, BRelTime.make(1), BFacets.SHOW_MILLISECONDS, BBoolean.TRUE) ]}
      statisticsStart: BAbsTime
        -- Last reset time of statistics.
        flags { readonly, transient }
        default {[ BAbsTime.NULL ]}
      averagePoll: String
        -- Average time spent in each poll.
        flags { readonly, transient }
        default {[ "-" ]}
      busyTime: String
        -- Percentage of time spent busy doing polls.
        flags { readonly, transient }
        default {[ "-" ]}
      totalPolls: String
        -- Total number of polls made and time
        -- spent waiting for polls to execute.
        flags { readonly, transient }
        default {[ "-" ]}
      dibsPolls: String
        -- Total number of polls made processing the dibs stack.
        flags { readonly, transient }
        default {[ "-" ]}
      fastPolls: String
        -- Total number of polls made processing fast queue.
        flags { readonly, transient }
        default {[ "-" ]}
      normalPolls: String
        -- Total number of polls made processing normal queue.
        flags { readonly, transient }
        default {[ "-" ]}
      slowPolls: String
        -- Total number of polls made processing slow queue.
        flags { readonly, transient }
        default {[ "-" ]}
      dibsCount: String
        -- Current and average number of components in dibs stack.
        flags { readonly, transient }
        default {[ "-" ]}
      fastCount: String
        -- Current and average number of components in fast queue.
        flags { readonly, transient }
        default {[ "-" ]}
      normalCount: String
        -- Current and average number of components in normal queue.
        flags { readonly, transient }
        default {[ "-" ]}
      slowCount: String
        -- Current and average number of components in slow queue.
        flags { readonly, transient }
        default {[ "-" ]}
      fastCycleTime: String
        -- Average cycle time of the fast queue.
        flags { readonly, transient }
        default {[ "-" ]}
      normalCycleTime: String
        -- Average cycle time of the normal queue.
        flags { readonly, transient }
        default {[ "-" ]}
      slowCycleTime: String
        -- Average cycle time of the slow queue.
        flags { readonly, transient }
        default {[ "-" ]}
    }

    actions
    {
      resetStatistics()
        -- Reset all the statistics properties.
        flags { confirmRequired }
    }
  }

  -*/
/*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
/*@ $javax.baja.driver.util.BPollScheduler(170923748)1.0$ @*/
/* Generated Thu Apr 14 09:29:59 EDT 2011 by Slot-o-Matic 2000 (c) Tridium, Inc. 2000 */

////////////////////////////////////////////////////////////////
// Property "fastRate"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>fastRate</code> property.
   * The frequency used to poll components set to fast.
   * @see javax.baja.driver.util.BPollScheduler#getFastRate
   * @see javax.baja.driver.util.BPollScheduler#setFastRate
   */
  public static final Property fastRate = newProperty(0, BRelTime.make(1000),BFacets.make(BFacets.MIN, BRelTime.make(1), BFacets.SHOW_MILLISECONDS, BBoolean.TRUE));
  
  /**
   * Get the <code>fastRate</code> property.
   * @see javax.baja.driver.util.BPollScheduler#fastRate
   */
  public BRelTime getFastRate() { return (BRelTime)get(fastRate); }
  
  /**
   * Set the <code>fastRate</code> property.
   * @see javax.baja.driver.util.BPollScheduler#fastRate
   */
  public void setFastRate(BRelTime v) { set(fastRate,v,null); }

////////////////////////////////////////////////////////////////
// Property "normalRate"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>normalRate</code> property.
   * The frequency used to poll components set to normal.
   * @see javax.baja.driver.util.BPollScheduler#getNormalRate
   * @see javax.baja.driver.util.BPollScheduler#setNormalRate
   */
  public static final Property normalRate = newProperty(0, BRelTime.make(5000),BFacets.make(BFacets.MIN, BRelTime.make(1), BFacets.SHOW_MILLISECONDS, BBoolean.TRUE));
  
  /**
   * Get the <code>normalRate</code> property.
   * @see javax.baja.driver.util.BPollScheduler#normalRate
   */
  public BRelTime getNormalRate() { return (BRelTime)get(normalRate); }
  
  /**
   * Set the <code>normalRate</code> property.
   * @see javax.baja.driver.util.BPollScheduler#normalRate
   */
  public void setNormalRate(BRelTime v) { set(normalRate,v,null); }

////////////////////////////////////////////////////////////////
// Property "slowRate"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>slowRate</code> property.
   * The frequency used to poll components set to slow.
   * @see javax.baja.driver.util.BPollScheduler#getSlowRate
   * @see javax.baja.driver.util.BPollScheduler#setSlowRate
   */
  public static final Property slowRate = newProperty(0, BRelTime.make(30000),BFacets.make(BFacets.MIN, BRelTime.make(1), BFacets.SHOW_MILLISECONDS, BBoolean.TRUE));
  
  /**
   * Get the <code>slowRate</code> property.
   * @see javax.baja.driver.util.BPollScheduler#slowRate
   */
  public BRelTime getSlowRate() { return (BRelTime)get(slowRate); }
  
  /**
   * Set the <code>slowRate</code> property.
   * @see javax.baja.driver.util.BPollScheduler#slowRate
   */
  public void setSlowRate(BRelTime v) { set(slowRate,v,null); }

////////////////////////////////////////////////////////////////
// Property "statisticsStart"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>statisticsStart</code> property.
   * Last reset time of statistics.
   * @see javax.baja.driver.util.BPollScheduler#getStatisticsStart
   * @see javax.baja.driver.util.BPollScheduler#setStatisticsStart
   */
  public static final Property statisticsStart = newProperty(Flags.READONLY|Flags.TRANSIENT, BAbsTime.NULL,null);
  
  /**
   * Get the <code>statisticsStart</code> property.
   * @see javax.baja.driver.util.BPollScheduler#statisticsStart
   */
  public BAbsTime getStatisticsStart() { return (BAbsTime)get(statisticsStart); }
  
  /**
   * Set the <code>statisticsStart</code> property.
   * @see javax.baja.driver.util.BPollScheduler#statisticsStart
   */
  public void setStatisticsStart(BAbsTime v) { set(statisticsStart,v,null); }

////////////////////////////////////////////////////////////////
// Property "averagePoll"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>averagePoll</code> property.
   * Average time spent in each poll.
   * @see javax.baja.driver.util.BPollScheduler#getAveragePoll
   * @see javax.baja.driver.util.BPollScheduler#setAveragePoll
   */
  public static final Property averagePoll = newProperty(Flags.READONLY|Flags.TRANSIENT, "-",null);
  
  /**
   * Get the <code>averagePoll</code> property.
   * @see javax.baja.driver.util.BPollScheduler#averagePoll
   */
  public String getAveragePoll() { return getString(averagePoll); }
  
  /**
   * Set the <code>averagePoll</code> property.
   * @see javax.baja.driver.util.BPollScheduler#averagePoll
   */
  public void setAveragePoll(String v) { setString(averagePoll,v,null); }

////////////////////////////////////////////////////////////////
// Property "busyTime"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>busyTime</code> property.
   * Percentage of time spent busy doing polls.
   * @see javax.baja.driver.util.BPollScheduler#getBusyTime
   * @see javax.baja.driver.util.BPollScheduler#setBusyTime
   */
  public static final Property busyTime = newProperty(Flags.READONLY|Flags.TRANSIENT, "-",null);
  
  /**
   * Get the <code>busyTime</code> property.
   * @see javax.baja.driver.util.BPollScheduler#busyTime
   */
  public String getBusyTime() { return getString(busyTime); }
  
  /**
   * Set the <code>busyTime</code> property.
   * @see javax.baja.driver.util.BPollScheduler#busyTime
   */
  public void setBusyTime(String v) { setString(busyTime,v,null); }

////////////////////////////////////////////////////////////////
// Property "totalPolls"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>totalPolls</code> property.
   * Total number of polls made and time spent waiting for polls to execute.
   * @see javax.baja.driver.util.BPollScheduler#getTotalPolls
   * @see javax.baja.driver.util.BPollScheduler#setTotalPolls
   */
  public static final Property totalPolls = newProperty(Flags.READONLY|Flags.TRANSIENT, "-",null);
  
  /**
   * Get the <code>totalPolls</code> property.
   * @see javax.baja.driver.util.BPollScheduler#totalPolls
   */
  public String getTotalPolls() { return getString(totalPolls); }
  
  /**
   * Set the <code>totalPolls</code> property.
   * @see javax.baja.driver.util.BPollScheduler#totalPolls
   */
  public void setTotalPolls(String v) { setString(totalPolls,v,null); }

////////////////////////////////////////////////////////////////
// Property "dibsPolls"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>dibsPolls</code> property.
   * Total number of polls made processing the dibs stack.
   * @see javax.baja.driver.util.BPollScheduler#getDibsPolls
   * @see javax.baja.driver.util.BPollScheduler#setDibsPolls
   */
  public static final Property dibsPolls = newProperty(Flags.READONLY|Flags.TRANSIENT, "-",null);
  
  /**
   * Get the <code>dibsPolls</code> property.
   * @see javax.baja.driver.util.BPollScheduler#dibsPolls
   */
  public String getDibsPolls() { return getString(dibsPolls); }
  
  /**
   * Set the <code>dibsPolls</code> property.
   * @see javax.baja.driver.util.BPollScheduler#dibsPolls
   */
  public void setDibsPolls(String v) { setString(dibsPolls,v,null); }

////////////////////////////////////////////////////////////////
// Property "fastPolls"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>fastPolls</code> property.
   * Total number of polls made processing fast queue.
   * @see javax.baja.driver.util.BPollScheduler#getFastPolls
   * @see javax.baja.driver.util.BPollScheduler#setFastPolls
   */
  public static final Property fastPolls = newProperty(Flags.READONLY|Flags.TRANSIENT, "-",null);
  
  /**
   * Get the <code>fastPolls</code> property.
   * @see javax.baja.driver.util.BPollScheduler#fastPolls
   */
  public String getFastPolls() { return getString(fastPolls); }
  
  /**
   * Set the <code>fastPolls</code> property.
   * @see javax.baja.driver.util.BPollScheduler#fastPolls
   */
  public void setFastPolls(String v) { setString(fastPolls,v,null); }

////////////////////////////////////////////////////////////////
// Property "normalPolls"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>normalPolls</code> property.
   * Total number of polls made processing normal queue.
   * @see javax.baja.driver.util.BPollScheduler#getNormalPolls
   * @see javax.baja.driver.util.BPollScheduler#setNormalPolls
   */
  public static final Property normalPolls = newProperty(Flags.READONLY|Flags.TRANSIENT, "-",null);
  
  /**
   * Get the <code>normalPolls</code> property.
   * @see javax.baja.driver.util.BPollScheduler#normalPolls
   */
  public String getNormalPolls() { return getString(normalPolls); }
  
  /**
   * Set the <code>normalPolls</code> property.
   * @see javax.baja.driver.util.BPollScheduler#normalPolls
   */
  public void setNormalPolls(String v) { setString(normalPolls,v,null); }

////////////////////////////////////////////////////////////////
// Property "slowPolls"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>slowPolls</code> property.
   * Total number of polls made processing slow queue.
   * @see javax.baja.driver.util.BPollScheduler#getSlowPolls
   * @see javax.baja.driver.util.BPollScheduler#setSlowPolls
   */
  public static final Property slowPolls = newProperty(Flags.READONLY|Flags.TRANSIENT, "-",null);
  
  /**
   * Get the <code>slowPolls</code> property.
   * @see javax.baja.driver.util.BPollScheduler#slowPolls
   */
  public String getSlowPolls() { return getString(slowPolls); }
  
  /**
   * Set the <code>slowPolls</code> property.
   * @see javax.baja.driver.util.BPollScheduler#slowPolls
   */
  public void setSlowPolls(String v) { setString(slowPolls,v,null); }

////////////////////////////////////////////////////////////////
// Property "dibsCount"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>dibsCount</code> property.
   * Current and average number of components in dibs stack.
   * @see javax.baja.driver.util.BPollScheduler#getDibsCount
   * @see javax.baja.driver.util.BPollScheduler#setDibsCount
   */
  public static final Property dibsCount = newProperty(Flags.READONLY|Flags.TRANSIENT, "-",null);
  
  /**
   * Get the <code>dibsCount</code> property.
   * @see javax.baja.driver.util.BPollScheduler#dibsCount
   */
  public String getDibsCount() { return getString(dibsCount); }
  
  /**
   * Set the <code>dibsCount</code> property.
   * @see javax.baja.driver.util.BPollScheduler#dibsCount
   */
  public void setDibsCount(String v) { setString(dibsCount,v,null); }

////////////////////////////////////////////////////////////////
// Property "fastCount"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>fastCount</code> property.
   * Current and average number of components in fast queue.
   * @see javax.baja.driver.util.BPollScheduler#getFastCount
   * @see javax.baja.driver.util.BPollScheduler#setFastCount
   */
  public static final Property fastCount = newProperty(Flags.READONLY|Flags.TRANSIENT, "-",null);
  
  /**
   * Get the <code>fastCount</code> property.
   * @see javax.baja.driver.util.BPollScheduler#fastCount
   */
  public String getFastCount() { return getString(fastCount); }
  
  /**
   * Set the <code>fastCount</code> property.
   * @see javax.baja.driver.util.BPollScheduler#fastCount
   */
  public void setFastCount(String v) { setString(fastCount,v,null); }

////////////////////////////////////////////////////////////////
// Property "normalCount"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>normalCount</code> property.
   * Current and average number of components in normal
   * queue.
   * @see javax.baja.driver.util.BPollScheduler#getNormalCount
   * @see javax.baja.driver.util.BPollScheduler#setNormalCount
   */
  public static final Property normalCount = newProperty(Flags.READONLY|Flags.TRANSIENT, "-",null);
  
  /**
   * Get the <code>normalCount</code> property.
   * @see javax.baja.driver.util.BPollScheduler#normalCount
   */
  public String getNormalCount() { return getString(normalCount); }
  
  /**
   * Set the <code>normalCount</code> property.
   * @see javax.baja.driver.util.BPollScheduler#normalCount
   */
  public void setNormalCount(String v) { setString(normalCount,v,null); }

////////////////////////////////////////////////////////////////
// Property "slowCount"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>slowCount</code> property.
   * Current and average number of components in slow queue.
   * @see javax.baja.driver.util.BPollScheduler#getSlowCount
   * @see javax.baja.driver.util.BPollScheduler#setSlowCount
   */
  public static final Property slowCount = newProperty(Flags.READONLY|Flags.TRANSIENT, "-",null);
  
  /**
   * Get the <code>slowCount</code> property.
   * @see javax.baja.driver.util.BPollScheduler#slowCount
   */
  public String getSlowCount() { return getString(slowCount); }
  
  /**
   * Set the <code>slowCount</code> property.
   * @see javax.baja.driver.util.BPollScheduler#slowCount
   */
  public void setSlowCount(String v) { setString(slowCount,v,null); }

////////////////////////////////////////////////////////////////
// Property "fastCycleTime"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>fastCycleTime</code> property.
   * Average cycle time of the fast queue.
   * @see javax.baja.driver.util.BPollScheduler#getFastCycleTime
   * @see javax.baja.driver.util.BPollScheduler#setFastCycleTime
   */
  public static final Property fastCycleTime = newProperty(Flags.READONLY|Flags.TRANSIENT, "-",null);
  
  /**
   * Get the <code>fastCycleTime</code> property.
   * @see javax.baja.driver.util.BPollScheduler#fastCycleTime
   */
  public String getFastCycleTime() { return getString(fastCycleTime); }
  
  /**
   * Set the <code>fastCycleTime</code> property.
   * @see javax.baja.driver.util.BPollScheduler#fastCycleTime
   */
  public void setFastCycleTime(String v) { setString(fastCycleTime,v,null); }

////////////////////////////////////////////////////////////////
// Property "normalCycleTime"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>normalCycleTime</code> property.
   * Average cycle time of the normal queue.
   * @see javax.baja.driver.util.BPollScheduler#getNormalCycleTime
   * @see javax.baja.driver.util.BPollScheduler#setNormalCycleTime
   */
  public static final Property normalCycleTime = newProperty(Flags.READONLY|Flags.TRANSIENT, "-",null);
  
  /**
   * Get the <code>normalCycleTime</code> property.
   * @see javax.baja.driver.util.BPollScheduler#normalCycleTime
   */
  public String getNormalCycleTime() { return getString(normalCycleTime); }
  
  /**
   * Set the <code>normalCycleTime</code> property.
   * @see javax.baja.driver.util.BPollScheduler#normalCycleTime
   */
  public void setNormalCycleTime(String v) { setString(normalCycleTime,v,null); }

////////////////////////////////////////////////////////////////
// Property "slowCycleTime"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>slowCycleTime</code> property.
   * Average cycle time of the slow queue.
   * @see javax.baja.driver.util.BPollScheduler#getSlowCycleTime
   * @see javax.baja.driver.util.BPollScheduler#setSlowCycleTime
   */
  public static final Property slowCycleTime = newProperty(Flags.READONLY|Flags.TRANSIENT, "-",null);
  
  /**
   * Get the <code>slowCycleTime</code> property.
   * @see javax.baja.driver.util.BPollScheduler#slowCycleTime
   */
  public String getSlowCycleTime() { return getString(slowCycleTime); }
  
  /**
   * Set the <code>slowCycleTime</code> property.
   * @see javax.baja.driver.util.BPollScheduler#slowCycleTime
   */
  public void setSlowCycleTime(String v) { setString(slowCycleTime,v,null); }

////////////////////////////////////////////////////////////////
// Action "resetStatistics"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>resetStatistics</code> action.
   * Reset all the statistics properties.
   * @see javax.baja.driver.util.BPollScheduler#resetStatistics()
   */
  public static final Action resetStatistics = newAction(Flags.CONFIRM_REQUIRED,null);
  
  /**
   * Invoke the <code>resetStatistics</code> action.
   * @see javax.baja.driver.util.BPollScheduler#resetStatistics
   */
  public void resetStatistics() { invoke(resetStatistics,null,null); }

////////////////////////////////////////////////////////////////
// Type
////////////////////////////////////////////////////////////////
  
  public Type getType() { return TYPE; }
  public static final Type TYPE = Sys.loadType(BPollScheduler.class);

/*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

////////////////////////////////////////////////////////////////
// Overrides
////////////////////////////////////////////////////////////////

  /**
   * Invoke the poll callback and log any errors.  We
   * also keep stats on the time taken for each poll.
   */
  public final void poll(BIPollable p)
  {
    long t1 = Clock.ticks();

    try
    {
      doPoll(p);
    }
    catch(Throwable e)
    {
      // TODO need a robust poll logging mechanism that
      // doesn't litter standard output with messages
      e.printStackTrace();
    }

    long t2 = Clock.ticks();
    totalPollCount += 1;
    totalPollTime += (t2 - t1);
    average = (double)totalPollTime / (double)totalPollCount;
  }

  /**
   * Poll the specified point synchronously, blocking
   * until polled successfully or an failure is encountered.
   * Failures should throw an exception.
   */
  public abstract void doPoll(BIPollable p)
    throws Exception;

////////////////////////////////////////////////////////////////
// Thread
////////////////////////////////////////////////////////////////

  /**
   * Default for started() is to call startThread().
   */
  public void started() { if (Sys.isStationStarted()) startThread(); }
  
  /**
   * Overridden starting in Niagara 3.4 in order to delay
   * the polling until after the station has started
   */
  public void stationStarted()
    throws Exception
  { 
    super.stationStarted(); 
    startThread(); 
  }

  /**
   * Default for stopped() is to call stopThread().
   */
  public void stopped() { stopThread(); }

  /**
   * Start the poll scheduler on its own thread.
   */
  public void startThread()
  {
    doResetStatistics();
    stopThread();
    isAlive = true;
    thread = new Thread(this, "Poll:" + getParent().getName());
    thread.start();
  }

  /**
   * Stop the poll scheduler thread.
   */
  public void stopThread()
  {
    isAlive = false;
    if (thread != null) thread.interrupt();
  }


  /**
   * Main loop of poll scheduler.
   */
  public void run()
  {
    long stats = Clock.ticks() + 10000;
    while(isAlive)
    {
      try
      {
        // Allow pollEnable to short circuit poll
        if(!getPollEnabled())
        {
          Thread.sleep(1000);
          continue;
        }

        // poll everything in the dibs queue
        pollDibs();

        // poll the next guy in our regular queues
        pollQueues();

        // sleep for a required bit of time
        long sleep = computeSleep();
        if (sleep > 0) Thread.sleep(sleep);

        // take stats every 10sec
        if (Clock.ticks() > stats)
        {
          checkBucketConfig();
          updateStats();
          stats = Clock.ticks() + 10000;
        }
      }
      catch(InterruptedException e)
      {
      }
      catch(Throwable e)
      {
        e.printStackTrace();
      }
    }
  }

  /**
   * Poll everything in the dibs queue immediately
   * in a tight loop.  We don't return until the
   * dibs queue is empty.
   */
  private void pollDibs()
  {
    while(true)
    {
      BIPollable next = null;
      synchronized(lock)
      {
        if (dibs.empty()) return;
        next = dibs.pop();
        dibsPollTotal++;
      }
      poll(next);
    }
  }

  /**
   * Poll the next guy on our standard queues.
   */
  private void pollQueues()
  {
    pollQueue(fast);
    pollQueue(norm);
    pollQueue(slow);
  }

  /**
   * Check if the deadline has been met for
   * polling the next guy in the specified bucket.
   */
  private void pollQueue(Bucket bucket)
  {
    // bail if we haven't met our deadline,
    // but allow a small fudge factor of 5ms
    if (bucket.nextTicks > Clock.ticks() + 5) return;

    // get the next guy in the list
    BIPollable p = null;
    int size = 0;
    synchronized(lock)
    {
      int index = bucket.index;
      size = bucket.q.size();
      if (size > 0)
      {
        if (index >= size) { index = 0; bucket.cycleTotal++; }
        p = bucket.q.get(index);
        bucket.index = index + 1;
      }
      else
      {
        bucket.cycleTotal++;
      }
    }

    // poll the little bastard
    long lastPollTime = 0;

    if (p != null)
    {
      long t1 = Clock.ticks();
      poll(p);
      lastPollTime = Clock.ticks() - t1;

      bucket.pollTotal++;
    }

    // compute our sleep time
    long rate = ((BRelTime)get(bucket.rateProp)).getMillis();
    long sleep = rate;
    if (size > 0)
    {
      double dRate = (double)rate;
      double dSize = (double)size;
      //sleep = (long)((dRate - dSize*average) / dSize);
      sleep = (long)((dRate - dSize*lastPollTime) / dSize);
    }
    else
      sleep = 1000;

    // set our next deadline
    bucket.nextTicks = Clock.ticks() + sleep;
  }

  /**
   * Compute the sleep time.
   */
  private long computeSleep()
  {
    long now = Clock.ticks();
    long sleep = 1000;
    sleep = Math.min(fast.nextTicks-now, sleep);
    sleep = Math.min(norm.nextTicks-now, sleep);
    sleep = Math.min(slow.nextTicks-now, sleep);
    return sleep;
  }

////////////////////////////////////////////////////////////////
// Bucket Config
////////////////////////////////////////////////////////////////

  /**
   * Every 10sec we check to make sure each pollable
   * is in its configured bucket.
   */
  private void checkBucketConfig()
  {
    synchronized(lock)
    {
      ArrayList<BIPollable> newFast = new ArrayList<>();
      ArrayList<BIPollable> newNorm = new ArrayList<>();
      ArrayList<BIPollable> newSlow = new ArrayList<>();

      reSort(fast.q, newFast, newNorm, newSlow);
      reSort(norm.q, newFast, newNorm, newSlow);
      reSort(slow.q, newFast, newNorm, newSlow);

      fast.q = newFast;
      norm.q = newNorm;
      slow.q = newSlow;
    }
  }

  /**
   * Resort the buckets.
   */
  private static void reSort(ArrayList<BIPollable> orig, ArrayList<BIPollable> newFast, ArrayList<BIPollable> newNorm, ArrayList<BIPollable> newSlow)
  {
    int size = orig.size();
    for(int i=0; i<size; ++i)
    {
      BIPollable p = orig.get(i);
      switch(p.getPollFrequency().getOrdinal())
      {
        case BPollFrequency.FAST:   newFast.add(p); break;
        case BPollFrequency.NORMAL: newNorm.add(p); break;
        case BPollFrequency.SLOW:   newSlow.add(p); break;
        default: throw new IllegalStateException();
      }
    }
  }

////////////////////////////////////////////////////////////////
// Statistics
////////////////////////////////////////////////////////////////

  /**
   * Action implementation for resetStatistics().
   */
  public void doResetStatistics()
  {
    totalPollTime  = 0;
    totalPollCount = 0;
    average        = 0;
    statsCount     = 0;
    dibsPollTotal  = 0;
    dibsSizeTotal  = 0;
    fast.reset();
    norm.reset();
    slow.reset();
    start = Clock.ticks();
    setStatisticsStart(Clock.time());
    updateStats();
  }

  /**
   * Update all the statistics fields.
   */
  private void updateStats()
  {
    long now = Clock.ticks();
    long uptime = now - start;
    statsCount++;

    // average poll time
    setAveragePoll(timeFormat.format(average));

    // busy time
    if (totalPollTime > 0)
    {
      setBusyTime(""+(int)(100d * ((double)totalPollTime/(double)uptime)) +
        "% (" + duration(totalPollTime) + "/" + duration(uptime) + ")");
    }

    // total polls for each bucket
    setTotalPolls("" + count(totalPollCount) + " over " + duration(totalPollTime));
    setDibsPolls(toPollTotal(dibsPollTotal));
    setFastPolls(toPollTotal(fast.pollTotal));
    setNormalPolls(toPollTotal(norm.pollTotal));
    setSlowPolls(toPollTotal(slow.pollTotal));

    // current and average bucket counts
    setDibsCount(toCount(dibs.size(), dibsSizeTotal)); dibsSizeTotal += dibs.size();
    setFastCount(toCount(fast.q.size(), fast.sizeTotal)); fast.sizeTotal += fast.q.size();
    setNormalCount(toCount(norm.q.size(), norm.sizeTotal)); norm.sizeTotal += norm.q.size();
    setSlowCount(toCount(slow.q.size(), slow.sizeTotal)); slow.sizeTotal += slow.q.size();

    // cycle times
    setFastCycleTime(toCycle(fast, uptime));
    setNormalCycleTime(toCycle(norm, uptime));
    setSlowCycleTime(toCycle(slow, uptime));
  }

  /**
   * Get a string for poll totals.
   */
  private String toPollTotal(int bucketTotal)
  {
    int total = totalPollCount;

    StringBuffer s = new StringBuffer();
    if (total == 0)
      s.append('-');
    else
      s.append((int)(100d * (double)bucketTotal/(double)total));

    s.append("% (").append(count(bucketTotal)).append('/').append(count(total)).append(')');
    return s.toString();
  }

  /**
   * Get a string for current and average count
   */
  private String toCount(int current, int total)
  {
    return "current=" + current + " average=" + (total/statsCount);
  }

  /**
   * Get a string for cycle stats.
   */
  private String toCycle(Bucket bucket, long uptime)
  {
    int cycles = bucket.cycleTotal;
    if (cycles == 0) return "-";
    return "average = " + (uptime/cycles) + "ms";
  }

  private String count(int count)
  {
    if (count < 10000) return String.valueOf(count);
    else return String.valueOf(count/1000) + "k";
  }

  private String duration(long duration)
  {
    if (duration < 10000L) return String.valueOf(duration) + "ms";
    else return String.valueOf(duration/1000) + "sec";
  }

////////////////////////////////////////////////////////////////
// Subscriptions
////////////////////////////////////////////////////////////////

  /**
   * Subscribe the pollable and start polling it
   * until it is unsubscrbed.
   */
  public void subscribe(BIPollable p)
  {
    synchronized(lock)
    {
      // push it on the first dibs stack
      // to get polled immediately
      dibs.push(p);

      // add it to its configured queue
      switch(p.getPollFrequency().getOrdinal())
      {
        case BPollFrequency.FAST:   fast.q.add(p); break;
        case BPollFrequency.NORMAL: norm.q.add(p); break;
        case BPollFrequency.SLOW:   slow.q.add(p); break;
        default: throw new IllegalStateException();
      }
    }
  }

  /**
   * Unsubscribe the pollable and stop polling it.
   * @returns true if the pollable point was subscribed,
   *          false if the point was not in any buckets.
   */
  public boolean unsubscribe(BIPollable p)
  {
    synchronized(lock)
    {
      // remove it from the dibs stack
      for(int i=0; i<dibs.size(); ++i)
        if (dibs.get(i) == p) { dibs.remove(i); break; }

      // remove it from the queues, just to
      // be safe we check all of them
      if (fast.remove(p)) return true;
      if (norm.remove(p)) return true;
      if (slow.remove(p)) return true;
      return false;
    }
  }


////////////////////////////////////////////////////////////////
// Bucket
////////////////////////////////////////////////////////////////

  static class Bucket
  {
    Bucket(Property rateProp)
    {
      this.q = new ArrayList<>();
      this.rateProp = rateProp;
    }

    boolean remove(BIPollable p)
    {
      int size = q.size();
      for(int i=0; i<size; ++i)
        if (q.get(i) == p) { q.remove(i); return true; }
      return false;
    }

    void reset()
    {
      index      = 0;
      nextTicks  = 0;
      pollTotal  = 0;
      sizeTotal  = 0;
      cycleTotal = 0;
    }

    Property rateProp;  // property for rate config
    ArrayList<BIPollable> q; // list of pollables
    int index;          // next index into queue
    long nextTicks;     // ticks of next poll
    int pollTotal;      // total number of polls
    int sizeTotal;      // total of q.size()
    int cycleTotal;     // total number of cycles completed
  }

////////////////////////////////////////////////////////////////
// Attributes
////////////////////////////////////////////////////////////////

  static final DecimalFormat timeFormat = new DecimalFormat("0.0#ms");

  boolean isAlive;              // thread alive flag
  Thread thread;                // poll thread
  long start;                   // ticks at start of poll thread
  long totalPollTime;           // total time spent in poll
  int totalPollCount;           // total number of polls
  double average;               // average time taken in each poll
  Stack<BIPollable> dibs = new Stack<>(); // first dips (those just subscribed)
  int dibsPollTotal;            // total polls on dibs
  int dibsSizeTotal;            // total size of dibs stack
  Object lock = new Object();   // synchronization monitor
  int statsCount;               // number of statistics updates

  Bucket fast = new Bucket(fastRate);
  Bucket norm = new Bucket(normalRate);
  Bucket slow = new Bucket(slowRate);

}
