/*
 * Copyright 2015 Tridium, Inc. All Rights Reserved.
 */
package com.tridium.ddfIp.ui;

import javax.baja.sys.BComplex;
import javax.baja.sys.BObject;
import javax.baja.sys.BVector;
import javax.baja.sys.Context;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import javax.baja.ui.BTextDropDown;
import javax.baja.ui.BTextField;
import javax.baja.ui.list.BList;
import javax.baja.util.Lexicon;
import javax.baja.workbench.BWbEditor;
import javax.baja.workbench.CannotSaveException;
import javax.baja.workbench.fieldeditor.BWbFieldEditor;

import com.tridium.ddfIp.comm.BDdfIpAdapter;
import com.tridium.ddfIp.comm.BDdfIpCommunicator;

/**
 * Edits a BDdfIpAdapter that references an Ip Address that corresponds
 * to a local network interface on the host that is running the station.
 * 
 * The user interface consists of a text drop down list box.
 * 
 * All available adapters (as reported by the Ip Communicator) appear in the drop down list.
 * The current adapter is automatically selected. The list also features a [*** Default Local Host ***]
 * adpater (the text for "Default Local Host" is localized to the user's local language).
 * 
 * If the Ip Communicator does not report any adpaters then the text box portion will be editable.
 * This accomodates offline-editing scenarios (the Ip Communicator does not report any available
 * network interfaces if the station is in offline mode). This would also accomodate the extremely
 * rare scenario where the station is running on a platform computer that has no network interfaces
 * (extremely unlikely!). The real purpose here is to support offline editing.
 * 
 * @author lperkins
 */
public class BDdfIpAdapterEditor
  extends BWbFieldEditor
{
  /*-
  class BDdfIpAdapterEditor
  {
  }
  -*/
/*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
/*@ $com.tridium.ddfIp.ui.BDdfIpAdapterEditor(3642467206)1.0$ @*/
/* Generated Tue Jun 12 10:08:05 EDT 2007 by Slot-o-Matic 2000 (c) Tridium, Inc. 2000 */

////////////////////////////////////////////////////////////////
// Type
////////////////////////////////////////////////////////////////
  
  public Type getType() { return TYPE; }
  public static final Type TYPE = Sys.loadType(BDdfIpAdapterEditor.class);

/*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/
  
////////////////////////////////////////////////////////////////
//BWbEditor
////////////////////////////////////////////////////////////////
  
  /**
   * This the override method used to populate the 
   * editor's state based on the specified value.
   */
  protected void doLoadValue(BObject value, Context context)
    throws Exception
  {
    ipAdapter = (BDdfIpAdapter)value;
    layoutUi();
  }
  
  /**
   * This is the override method for saving the editor's
   * state back into an BObject form.  If the editor is
   * a edit-by-reference editor (BComplex) then it should
   * save its state back to the specified value instance and 
   * return that instance.  If the editor is edit-by-value 
   * (BSimple) then it should create a new instance and 
   * return it.
   * <p>
   * The default implementation assumes an editor which
   * is view only and does nothing but return the value.
   */
  protected BObject doSaveValue(BObject value, Context cx)
    throws CannotSaveException, Exception
  {
    BDdfIpAdapter saveValue = (BDdfIpAdapter)value;
    
    // Gets the position of the selected item from the drop down
    int selectedAdapterPosition = adapterDropDown.getList().getSelectedIndex();
    
    // Since the drop-down-list-box's text box is editable, the user could
    // have entered something that is not in the drop-down-list-box. This would
    // be an acceptable thing to do, especially when configuring a station in 
    // offline mode.
    
    if (selectedAdapterPosition<0) // If the user entered something that is not in the list
    { 
      // Sets the adapter description according to what the user entered
      saveValue.becomeDescriptionOnlyAdapter( adapterDropDown.getText() );
    }
    else if (selectedAdapterPosition == 0) // If the user selected "[Default Local Host]"
    { 
      // The Ip Communicator use whatever Java deteremines to be the local host.
      saveValue.becomeDefaultLocalHostAdapter();
    }
    else // selectedAdapterPosition >= 1
    { 
      // Sets the adapter description according to what the user entered
      AdapterDropDownListItem selectedItem = (AdapterDropDownListItem)adapterDropDown.getList().getSelectedItem();
      
      saveValue.setDescription( selectedItem.description );
      
      saveValue.setAdapterId( selectedItem.adapterId );
      
      saveValue.setIpAddress( selectedItem.ipAddress );
    }
    return saveValue;
  }
  

////////////////////////////////////////////////////////////////
// BDdfIpAdapterEditor
////////////////////////////////////////////////////////////////
  
  /**
   * Constructs the BTextDropDown that will serve as the "Adapter" drop drown. This method adds all
   * available adapters (as reported by the Ip Communicator) to the drop down list. The current
   * adapter is automatically selected. The list also features a [*** Default Local Host ***]
   * adpater. If the Ip Communicator does not report any adpaters then the text box portion will
   * be editable. This accomodates offline-editing scenarios (the Ip Communicator will not report
   * any available network interfaces if the station is in offline mode) 
   */
  protected BTextDropDown makeAdapterDropDown()
  { 
    // Creates the text box portion of drop-down list box
    BTextField textField = new BTextField();
    
    // Creates the list portion of the drop-down list box.
    // The Adapter Id's drop-down contains at least one item, the
    // Text that represents the [*** DEFAULT ***] adapter id.
    BList list = new BList();

    // The first item in the list is "[*** Default Local Host ***]". If selected, the Ip Communicator will
    // Use whatever Java determines to be the local host
    AdapterDropDownListItem defaultListItem = new AdapterDropDownListItem(DEFAULT_LOCAL_HOST_TEXT, "","");
    
    list.addItem( defaultListItem ); // Adds the default lost host item first in the list
    list.setSelectedItem(defaultListItem); // Selects the default local host item
    
    // This variable tracks the length of the longest String in the drop down
    int widestTextItem = DEFAULT_LOCAL_HOST_TEXT.length();
    
    // Gets all of the BTcpIpAdapterSettings from the hostIpAdapters BVector
    BVector hostIpAdapters = getIpCommunicator().refreshHostIpAdapters();
    BDdfIpAdapter[] hostEthernetAdapters = hostIpAdapters.getChildren(BDdfIpAdapter.class);

    // Adds a string to the drop-down for each adapter id on the
    // Host platform computer.
    for (int i=0; i<hostEthernetAdapters.length; i++)
    { 
      // Creates an object to place into the drop down list
      AdapterDropDownListItem listItem = new AdapterDropDownListItem( hostEthernetAdapters[i].getDescription(),
                                                                      hostEthernetAdapters[i].getAdapterId()  ,
                                                                      hostEthernetAdapters[i].getIpAddress()  );
      
      list.addItem( listItem ); // Adds the item to the drop down list
      
      // Checks if the current ipAdapter matches the list item
      if ( ipAdapter.getDescription().equals( listItem.description ) &&
           ipAdapter.getAdapterId()  .equals( listItem.adapterId   ) &&
           ipAdapter.getIpAddress()  .equals( listItem.ipAddress   )  )
      { 
        // If the current ipAdapter matches the list item, then it becomes selected by default
        list.setSelectedItem(listItem);
      }
      
      // Keeps track of the widest text item...the text drop down does not
      // Do a very good job at setting its own width. We will attempt to do
      // A better job ourselves.
      if (listItem.toString().length()>widestTextItem)
      {
        widestTextItem = listItem.toString().length();
      }
    }
    
    // If the current ipAdapter is a description-only adapter -- a special internal case when the station
    // Is being edited offline, then we allow the user to edit the text to free-handedly enter the network
    // Interface. The Ip Communicator will try to match the description to an adapter on the running station
    // As soon as possible.
    if (ipAdapter.isDescriptionOnlyAdapter())
    { 
      // In this scenario, the ipAdapter value specifies a description but does not identify
      // An adapter that is available on the host computer. This scenario could occur if the user
      // Edits the station offline. To accomodate this, lets set the text in the text field's text
      // Equal to the description of the current adapter. 
      textField.setText( ipAdapter.getDescription() );
      
      // And let's make sure that the list knows that nothing is selected in it.
      list.setSelectedIndex(-1);
      
      // Keeps track of the widest text item...the text drop down does not
      // Do a very good job at setting its own width. We will attempt to do
      // A better job ourselves.
      if (ipAdapter.getDescription().length()>widestTextItem)
      {
        widestTextItem = ipAdapter.getDescription().toString().length();
      }
      
    }
    else
    { 
      // This has to explicitly set the text in the text box equal to
      // the text of the initially selected item, otherwise, the textbox
      // will appear empty (a caveat)
      textField.setText( list.getSelectedItem().toString() );
    }
    
    // Creates the text, drop-down-list-box
    //BTextDropDown textDropDown= new BTextDropDown(textField,list);  <-- For 3.0 compatibility, I cannot use this constructor -- bummer!
    BTextDropDown textDropDown=new BTextDropDown();                // <-- Instead, I have to do copy everything over manually -- bummer! 
    
    for (int i=0; i<list.getItemCount(); i++)                      // <-- Instead, I have to do copy everything over manually -- bummer!
      textDropDown.getList().addItem( list.getItem(i));
    
    textDropDown.setText(textField.getText());                     // <-- Instead, I have to do copy everything over manually -- bummer!
    
    
    // Allows the user to free-handedly edit the textbox IF there are no adapters to choose from. The majority
    // Of the time when no adapters are available indicates that the user is editing the station in offline
    // mode (since the adpaters are retrieved from the platform Tcp/Ip service which is only availabe in a
    // running station)
    textDropDown.getEditor().setEditable(hostEthernetAdapters.length==0);
    
    // The text drop down does not do a very good job at setting its own width. We will attempt to do
    // A better job ourselves by explicitly setting the wight to the lesser of 50 characters
    // Or the 'widestTextItem'+5 --> We add five to account for the down arrow icon. Otherwise, the
    // Text in the text box could have a few characters behind the down arrow.
    textDropDown.getEditor().setVisibleColumns(widestTextItem+5>50?50:widestTextItem+5);
    
    // Returns a new BTextDropDown that is a combination of
    // the text field and the list
    return textDropDown;
  }
  
  /**
   * Contructs this widget's content as a BGridPane with 2 columns. Inside the grid
   * pane is the localized label "Adapter" on the left and a text drop down on the
   * right.
   */
  protected void layoutUi()
  { 
    // Makes the 'adapterDropDown' serve as the ui content for this widget
    setContent(adapterDropDown = makeAdapterDropDown());
    // Hooks up the adapterDropDown's textModified topic to this widget's
    // pluginModified topic. This automatically informs the Niagara AX ui 
    // Core whenever this editor changes.
    linkTo(adapterDropDown, BTextDropDown.valueModified, BWbEditor.setModified);
  }
  
  /**
   * This method gets the most immediate Ip Communicator parent of
   * the 'ipAddress' structure.
   */
  protected BDdfIpCommunicator getIpCommunicator()
  {
    BComplex p = ipAdapter;
    
    while (!(p instanceof BDdfIpCommunicator) && p!=null)
    {
      p = p.getParent();
    }
    
    return (BDdfIpCommunicator)p;
  }
  
////////////////////////////////////////////////////////////////
// AdapterDropDownListItem
////////////////////////////////////////////////////////////////
  
  /**
   * Objects of this class are placed into the BList of
   * the adapterDropDown.
   * 
   * The description is displayed in the adapterDropDown
   * (since this class's 'toString' method returns the
   * description), however, each item in the adapterDropDown
   * stores the description, adapterId, and ipAddress.
   * 
   * @author lperkins
   */
  public static class AdapterDropDownListItem
  {
    AdapterDropDownListItem(String desc, String id, String ipAddr)
    {
      this.description=desc;
      this.adapterId=id;
      this.ipAddress=ipAddr;
    }
    
    /**
     * The description is displayed in the adapterDropDown,
     * however, each item in the adapterDropDown stores the
     * description, adapterId, and ipAddress.
     */
    public String toString()
    {
      return description;
    }
    
    public final String description;
    public final String adapterId;
    public final String ipAddress;
  }
 
////////////////////////////////////////////////////////////////
// Attributes
////////////////////////////////////////////////////////////////
  
 /**
  * This is the current Ip Adapter that this widget is operating upon. More specifically, this is
  * the most recent value that was passed to the doLoadValue method.
  */
  protected BDdfIpAdapter ipAdapter = null;
  
  public static final Lexicon LEX = Lexicon.make(BDdfIpAdapterEditor.class);
  /**
   * This is the key into the devIpDriver's lexicon for the localized "Adapter" string.
   */
  public static final String IP_ADAPTER_KEY = "Adapter";
  /**
   * This is the localized text that means "Adapter" as taken from the devIpDriver lexicon.
   */
  public static final String IP_ADAPTER_TEXT = LEX.getText(IP_ADAPTER_KEY);
  /**
   * This is the key into the devIpDriver's lexicon for the localized "Default Local Host" string.
   */
  public static final String DEFAULT_LOCAL_HOST_KEY = "DefaultLocalHost";
  /**
   * This is the localized text that means "Default Local Host" as taken from the devIpDriver lexicon.
   */
  public static final String DEFAULT_LOCAL_HOST_TEXT = "[*** "+LEX.getText(DEFAULT_LOCAL_HOST_KEY)+" ***]";
  
  /**
  
  
  /**
   * This is the text box / drop-down-list-box where the end-user selects the adapter. The
   * adapter description is what the user really sees, although the adapterId is
   * what really uniquely identifies the adapter (it is really quite un-user-friendly
   * so this drop down displays the adapter description
   */
  protected BTextDropDown adapterDropDown=null;
}
