/*
 * Copyright 2015 Tridium, Inc. All Rights Reserved.
 */
package com.tridium.ddfIp.udp.comm;

import java.net.DatagramPacket;
import java.net.InetAddress;

import javax.baja.sys.BSimple;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;

import com.tridium.ddf.comm.IDdfDataFrame;
import com.tridium.ddf.comm.defaultComm.BDdfNullReceiver;

/**
 * By default, the Udp/Ip receiver uses the same Udp/Ip address and Udp/Ip
 * port as the Udp communicator component's transmitter. 
 * 
 * Note: Some protocols use a different Udp/Ip receive port. If that is the
 * case then the developer should extend this class, add a property of
 * type BDdfUdpIpAddress, override this method, and return the property's
 * value from this method. That will cause this object to use its own Udp/Ip
 * adress and Udp/Ip port.
 * 
 * This class extends BDdfNullReceiver so that it does not have to implement the 
 * 'isStartOfFrame', etc. method.
 * @author lperkins
 *
 */
public class BDdfUdpReceiver
  extends BDdfNullReceiver
{
  /*-
   class BDdfUdpReceiver
   {
     properties
     {
     }
   }
   -*/
/*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
/*@ $com.tridium.ddfIp.udp.comm.BDdfUdpReceiver(2812432306)1.0$ @*/
/* Generated Tue Jun 12 10:08:05 EDT 2007 by Slot-o-Matic 2000 (c) Tridium, Inc. 2000 */

////////////////////////////////////////////////////////////////
// Type
////////////////////////////////////////////////////////////////
  
  public Type getType() { return TYPE; }
  public static final Type TYPE = Sys.loadType(BDdfUdpReceiver.class);

/*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

  public BDdfUdpReceiver()
  {
    udpReceiveBuffer = new byte[getUdpReceiveBufferSize()];
    udpReceivePacket = new DatagramPacket(udpReceiveBuffer,udpReceiveBuffer.length);
    internalReceiveFrame = new UdpReceiveFrame(); // This wraps the udpReceiveBuffer
  }
  
  /**
   * Gets the Udp port that the receiver's Udp DatagramSocket will be bound to on the host (Jace).
   * 
   * @return this default implementation returns -1 which causes the BDdfUdpCommunicator
   * to allow Java to choose any available Udp port. If the developer's driver requires
   * a particular Udp origination port for the receive socket then the developer may
   * override this method and supply a Udp Port as he or she sees fit.
   */
  public int getReceivePort()
  {
    return -1;
  }  


  public BDdfUdpCommunicator getUdpCommunicator()
  {
    return (BDdfUdpCommunicator)getDdfCommunicator();
  }  

  /**
   * Overrides BDdfNullReceiver.checkFrame to always return true, instead of false,
   * as BDdfNullReceiver does. This method returns can safely return true because
   * checksum checking is not necessary since Udp/Ip already has data integrity checks built-in.
   */
  protected boolean checkFrame(IDdfDataFrame completeFrame)
  {
    return true;
  }
  
  /**
   * The developer may override this to declare a buffer size that is more suitable
   * to his or her protocol.
   * 
   * This method is called from the constructor to initialize the internal Udp/Ip
   * receive buffer.
   * 
   * @return by default, this returns 1024. I figure this is about the large that we
   * can get away with...since we do not know anything about any particular Udp/Ip
   * protocol at this layer.
   */
  protected int getUdpReceiveBufferSize()
  {
    return 1024;
  }

  /**
   * This method must be implemented to recognize and receive a frame of data from the field-bus.
   * 
   * If overridden then this method must block and only return once a frame of data is received.
   *
   * Hopefully, the default implementation of this method will suffice. It would allow the descendant
   * to simply implement the abstract isCompleteFrame and isStartOfFrame methods.
   * 
   * @return An object that implements the IDdfDataFrame interface that represents some data that was
   * just received from the field-bus.
   */
  protected IDdfDataFrame doReceiveFrame()
    throws Exception
  { 
    // The socket's receive method blocks until a Udp/Ip packet is received
    getUdpCommunicator().getUdpReceiveSocket().receive(udpReceivePacket);
    // By making it to this line without throwing an exception, we are gauranteed that
    // Java has filled the udpReceivePacket with the Udp/Ip received data
    // This returns the internalReceiveFrame, which wraps the Udp/Ip packet and data buffer
    internalReceiveFrame.takeSnapshot();
    return internalReceiveFrame;
  }
  
  public class UdpReceiveFrame
    implements IDdfDataFrame
  {
    byte[] frameBytes;
    int frameSize;
    InetAddress udpOriginationAddress;
    int udpOriginationPort;
    
    /**
     * Configures the bytes and length from the internal receive packet as of its current state
     *
     */
    synchronized void takeSnapshot()
    {
      frameBytes=udpReceiveBuffer;
      frameSize=udpReceivePacket.getLength();
      udpOriginationAddress=udpReceivePacket.getAddress();
      udpOriginationPort=udpReceivePacket.getPort();
    }
    
    /**
     * Gets the InetAddress that refers to the field-device that sent this frame's data
     * to the Jace.
     */
    public InetAddress getUdpOriginationAddress(){return udpOriginationAddress;}
    
    /**
     * Gets the Udp/Ip port from the field-device that sent this frame's data to the Jace.
     */
    public int getUdpOriginationPort(){return udpOriginationPort;}

    /**
     * Gets the Udp/Ip data bytes of this frame.
     */
    public byte[] getFrameBytes(){return frameBytes; }

    /**
     * Makes a clone of this frame.
     */
    public synchronized IDdfDataFrame getFrameCopy()
    { // Makes a new instance of UdpReceiveFrame
      UdpReceiveFrame copy = new UdpReceiveFrame();
      // Copies the frameSize, frameBytes, udpOriginationAddress, and udpOriginationPort from this
      // instance of UdpReceiveFrame into the new copy
      copy.frameSize = frameSize;
      copy.frameBytes = new byte[frameSize];
      System.arraycopy(this.frameBytes, 0, copy.frameBytes, 0, frameSize);
      copy.udpOriginationAddress = udpOriginationAddress;
      copy.udpOriginationPort = udpOriginationPort;
      return copy;
    }

    /**
     * Gets the Udp/Ip data frame size.
     */
    public int getFrameSize()
    {
      return frameSize;
    }

    /**
     * Returns whatever is returned by the 'computeTag(this)' method on the
     * outer instance.
     */
    public BSimple getFrameTag()
    {
      return computeTag(this);
    }
  }
  
  protected UdpReceiveFrame internalReceiveFrame = null;
  protected DatagramPacket udpReceivePacket = null;
  protected byte[] udpReceiveBuffer = null;
  protected int availableUdpBytes = 0;
  protected int udpReceiveBufferOffset=0;
}
