/*
 * Copyright 2015 Tridium, Inc. All Rights Reserved.
 */
package com.tridium.ddfIp.comm;

import javax.baja.sys.Action;
import javax.baja.sys.ActionInvokeException;
import javax.baja.sys.BFacets;
import javax.baja.sys.BRelTime;
import javax.baja.sys.BVector;
import javax.baja.sys.BajaRuntimeException;
import javax.baja.sys.Clock;
import javax.baja.sys.Flags;
import javax.baja.sys.Property;
import javax.baja.sys.ServiceNotFoundException;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;

import com.tridium.ddf.comm.defaultComm.BDdfCommunicator;
import com.tridium.platform.tcpip.BTcpIpAdapterSettings;
import com.tridium.platform.tcpip.BTcpIpHostSettings;
import com.tridium.platform.tcpip.BTcpIpPlatformService;

/**
 * This class is the most basic override point for an IP based
 * driver. It adds a 'networkInterface' to the other communicator
 * properties.
 * 
 * @author lperkins
 *
 */
public class BDdfIpCommunicator
  extends BDdfCommunicator
{
  /*-
    class BDdfIpCommunicator
    {
      properties
      {
        networkInterface : BDdfIpAdapter
          -- Specifies the network interface adapter to use for I.P. communications.
          -- To get a reference to the network interface for development purposes,
          -- developers should call the getNetworkAdapter method instead. It
          -- performs some necessary house-keeping operations on the value that
          -- this property stores.
          default{[BDdfIpAdapter.makeDefaultLocalHost()]}
          slotfacets{[BFacets.make(BFacets.FIELD_EDITOR, "devIpDriver:DdfIpAdapterEditor")]}
        hostIpAdapters : BVector
          -- This is a copy of some of the information pertaining to each network
          -- interface adapter that is on the host computer. This is for internal
          -- use only and should remain hidden and transient.
          flags{hidden,transient}
          default{[new BVector()]}
      
      }
      actions
      {
        refreshHostIpAdapters() : BVector
          -- Asks the station-side copy of the Ip Communicator to analyze
          -- Its platform's ip adapters and return a BVector containing
          -- a BDdfIpAdapter for each available Ethernet adapter.
          -- This maybe invoked programmatically from the client-side or the
          -- server side to arrive at an array of the Jace's Ethernet adapters
          flags{hidden}
      }      
    }
   -*/
/*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
/*@ $com.tridium.ddfIp.comm.BDdfIpCommunicator(4256538294)1.0$ @*/
/* Generated Mon May 03 13:37:08 EDT 2010 by Slot-o-Matic 2000 (c) Tridium, Inc. 2000 */

////////////////////////////////////////////////////////////////
// Property "networkInterface"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>networkInterface</code> property.
   * Specifies the network interface adapter to use for
   * I.P. communications. To get a reference to the network
   * interface for development purposes, developers should
   * call the getNetworkAdapter method instead. It performs
   * some necessary house-keeping operations on the value
   * that this property stores.
   * @see com.tridium.ddfIp.comm.BDdfIpCommunicator#getNetworkInterface
   * @see com.tridium.ddfIp.comm.BDdfIpCommunicator#setNetworkInterface
   */
  public static final Property networkInterface = newProperty(0, BDdfIpAdapter.makeDefaultLocalHost(),BFacets.make(BFacets.FIELD_EDITOR, "devIpDriver:DdfIpAdapterEditor"));
  
  /**
   * Get the <code>networkInterface</code> property.
   * @see com.tridium.ddfIp.comm.BDdfIpCommunicator#networkInterface
   */
  public BDdfIpAdapter getNetworkInterface() { return (BDdfIpAdapter)get(networkInterface); }
  
  /**
   * Set the <code>networkInterface</code> property.
   * @see com.tridium.ddfIp.comm.BDdfIpCommunicator#networkInterface
   */
  public void setNetworkInterface(BDdfIpAdapter v) { set(networkInterface,v,null); }

////////////////////////////////////////////////////////////////
// Property "hostIpAdapters"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>hostIpAdapters</code> property.
   * This is a copy of some of the information pertaining
   * to each network interface adapter that is on the host
   * computer. This is for internal use only and should
   * remain hidden and transient.
   * @see com.tridium.ddfIp.comm.BDdfIpCommunicator#getHostIpAdapters
   * @see com.tridium.ddfIp.comm.BDdfIpCommunicator#setHostIpAdapters
   */
  public static final Property hostIpAdapters = newProperty(Flags.HIDDEN|Flags.TRANSIENT, new BVector(),null);
  
  /**
   * Get the <code>hostIpAdapters</code> property.
   * @see com.tridium.ddfIp.comm.BDdfIpCommunicator#hostIpAdapters
   */
  public BVector getHostIpAdapters() { return (BVector)get(hostIpAdapters); }
  
  /**
   * Set the <code>hostIpAdapters</code> property.
   * @see com.tridium.ddfIp.comm.BDdfIpCommunicator#hostIpAdapters
   */
  public void setHostIpAdapters(BVector v) { set(hostIpAdapters,v,null); }

////////////////////////////////////////////////////////////////
// Action "refreshHostIpAdapters"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>refreshHostIpAdapters</code> action.
   * Asks the station-side copy of the Ip Communicator to analyze Its platform's ip adapters and return a BVector containing a BDdfIpAdapter for each available Ethernet adapter. This maybe invoked programmatically from the client-side or the server side to arrive at an array of the Jace's Ethernet adapters
   * @see com.tridium.ddfIp.comm.BDdfIpCommunicator#refreshHostIpAdapters()
   */
  public static final Action refreshHostIpAdapters = newAction(Flags.HIDDEN,null);
  
  /**
   * Invoke the <code>refreshHostIpAdapters</code> action.
   * @see com.tridium.ddfIp.comm.BDdfIpCommunicator#refreshHostIpAdapters
   */
  public BVector refreshHostIpAdapters() { return (BVector)invoke(refreshHostIpAdapters,null,null); }

////////////////////////////////////////////////////////////////
// Type
////////////////////////////////////////////////////////////////
  
  public Type getType() { return TYPE; }
  public static final Type TYPE = Sys.loadType(BDdfIpCommunicator.class);

/*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/
  
////////////////////////////////////////////////////////////////
// BDdfCommunicator
////////////////////////////////////////////////////////////////
  
  public final void communicatorStarted() throws Exception
  {
    super.communicatorStarted();
    
    try
    {
      // Updates the internal list of 'ip adapters' right now
      refreshHostIpAdapters();
      // Schedules to have the list of ip adapters updated every 1 hour
      Clock.schedulePeriodically(this, BRelTime.makeHours(1), refreshHostIpAdapters, null);
    }
    catch (ActionInvokeException aie)
    {
      if (aie.getCause() instanceof ServiceNotFoundException)
      {
        // This could happen if running in proxy mode
        getLog().message("Ddf Ip Communicator running in proxy mode: "+getSlotPath());
      }
      else
      {
        throw new BajaRuntimeException(aie);
      }
    }
    
    ipCommunicatorStarted();
  }  
  
  public final void communicatorStopped() throws Exception
  {
    ipCommunicatorStopped();
    super.communicatorStopped();
  }
  
////////////////////////////////////////////////////////////////
// BDdfIpCommunicator API
////////////////////////////////////////////////////////////////
  
  /**
   * This is the override point for a developer to customize the
   * startup behavior.
   * 
   * @throws any exception may be thrown
   */
  public void ipCommunicatorStarted()
    throws Exception 
  {
    
  }
  
  /**
   * This is the override point for a developer to customize the
   * shutdown behavior.
   * 
   * @throws any exception may be thrown
   */
  public void ipCommunicatorStopped()
    throws Exception
  {
    
  } 
  
  /**
   * Implements the getHostIpAdapters action.
   * 
   * @return a BVector containing a BDdfIpAdapter structure for each ethernet adapter on the host that is
   * running the station.
   */
  public BVector doRefreshHostIpAdapters()
  {
    // Allocates the return object for this method
    getHostIpAdapters().removeAll();
    // Gets an array, bloated with "BTcpIpAdapterSetting"s components representing each of the ethernet adapters
    BTcpIpAdapterSettings[] ethernetAdapters = getHostEthernetAdapters();
    // Adds a BDdfUdpIpAdapter to the platformIpAdapters vector, for each BTcpIpAdapterSettings object
    for (int i=0; i<ethernetAdapters.length; i++)
    {
      if ( (ethernetAdapters[i].getIsAdapterEnabled()) &&
           (!(ethernetAdapters[i].getIpAddress().equals(""))))
      {
        getHostIpAdapters().add("adapter?", new BDdfIpAdapter( ethernetAdapters[i].getAdapterId(),
                                                                 ethernetAdapters[i].getDescription(),
                                                                 ethernetAdapters[i].getIpAddress()),  Flags.READONLY);
      }
    }
    
    return getHostIpAdapters();
  }  
  
  /**
   */
  public BDdfIpAdapter[] getHostIpAdaptersArray()
  { 
    // Gets a reference to the location where the action was coded to place
    // The adapter structures
    BVector hostIpAdapters = getHostIpAdapters();
    
    // Gets all of the BTcpIpAdapterSettings from the hostIpAdapters BVector
    return hostIpAdapters.getChildren(BDdfIpAdapter.class);
  }

  /**
   * Gets the BDdfIpAdapter structure representing the host's network interface to use
   * for sending and receiving DatagramPackets. This default implementation looks at the
   * "Network Interface" property. Before returning it 
   */
  public BDdfIpAdapter getNetworkAdapter()
  {
    try
    {
      // Gets the network interface to 
      BDdfIpAdapter networkInterface = getNetworkInterface();
      
      // In the event that the user specified an adapter (presumably when the station was
      //  in its infancy in offline mode) that the offline-mode VM was not able to match
      // to a host interface (since offline mode does not have a host-station) then this
      // attempts to match up the user-defined text for the adapter to a network interface
      // on the host that is running this station.
      if (networkInterface.isDescriptionOnlyAdapter())
        updateNetworkInterfaceFromDescription(networkInterface);
    
      // Updates the Ip Address, in case the network interface uses dynamic Ip addressing
      updateIpAddress(networkInterface);
      
      return networkInterface;
    }
    catch (ServiceNotFoundException snfe)
    {
      return localHostAdapter;
    }
  }
  
  /**
   * In the event that the user specified an adapter (presumably when the station was
   * in its infancy in offline mode) that the offline-mode VM was not able to match
   * to a host interface (since offline mode does not have a host-station) then this
   * attempts to match up the user-defined text for the adapter to a network interface
   * on the host that is running this station.
   * 
   * @param networkInterface the BDdfUdpIpAdapter to update
   * 
   * 
   * @return true if this method is able to match up the given networkInterface to
   * a host adapter and false otherwise. If true is returned then the given
   * networkInterface's adapterId and ipAddress are automatically updated.
   */
  protected boolean updateNetworkInterfaceFromDescription(BDdfIpAdapter networkInterface)
  {
    // Attempt to determine the adapterId and ipAddress from the the given networkInterface's description
    BDdfIpAdapter[] hostAdapters = getHostIpAdaptersArray();
    
    boolean adapterFound=false;
    
    // Does a case-sensitive match, looking for a host adapter whose description matches the
    // Description of the given network interface
    for (int i=0; i<hostAdapters.length && !adapterFound; i++)
    {
      if (hostAdapters[i].getDescription().trim().equals(networkInterface.getDescription()))
      {
        adapterFound=true;
        networkInterface.copyFrom(hostAdapters[i]);
      }
    }
    
    // If not yet found, then this does a non-case-sensitive match, looking for a host adapter whose description matches the
    // Description of the given network interface
    if (!adapterFound)
    {
      for (int i=0; i<hostAdapters.length && !adapterFound; i++)
      {
        if (hostAdapters[i].getDescription().trim().equalsIgnoreCase(networkInterface.getDescription()))
        {
          adapterFound=true;
          networkInterface.copyFrom(hostAdapters[i]);
        }
      }
    }
    return adapterFound;
  }
  
  /**
   * Finds the corresponding network adapter under the Platform Tcp/Ip service and
   * returns its ipAddress. This is necessary since the IP Address can change in a 
   * DHCP configuration.
   * 
   * @param networkInterface the BddfUdpIpAdapter
   * 
   * @return
   */
  protected void updateIpAddress(BDdfIpAdapter givenNetworkInterface)
  { 
    // Gets an array of information about all network interfaces. Gets this directly from
    // The platform Tcp/Ip service. Presumably the adapter information is updated whenever
    // Appropriate to account for a change in a dynamic Ip Address assignment
    BTcpIpAdapterSettings[] networkInterfaces = getHostEthernetAdapters();
    
    // Loops through the array containing the latest settings for all network adapters on
    // The host
    for (int i=0; i<networkInterfaces.length; i++)
    { 
      // Finds the one in the list whose adapter ID matches the adapter ID of the givenNetworkInterface
      if ( networkInterfaces[i].getAdapterId().equals(givenNetworkInterface.getAdapterId()))
      { 
        // If found, then this updates the Ip address of the givenNetworkInterface
        givenNetworkInterface.setIpAddress(networkInterfaces[i].getIpAddress());
        
        // Breaks out of the loop...we are done...there should not be multiple adapters with the
        // Same adapter id, otherwise, there would be chaos!
        break;
      }
    }
  }
  
  /**
   * This method retrieves an array of "BTcpIpAdapterSettings" objects from the
   * Platform Tcp/Ip service. Each item in the array represents an available
   * Ethernet adapter on the host. 
   */
  private BTcpIpAdapterSettings[] getHostEthernetAdapters()
  {
    // Gets the Niagara AX Tcp/Ip service, leases it, and gets its "adapter settings" objects
    BTcpIpPlatformService svc = (BTcpIpPlatformService) Sys.getService(BTcpIpPlatformService.TYPE);
    svc.lease();
    svc.checkPropertiesLoaded();
    
    BTcpIpHostSettings platformTcpSettings = svc.getSettings();
    return platformTcpSettings.getAdapters().getChildren(BTcpIpAdapterSettings.class);
  }
  
////////////////////////////////////////////////////////////////
// Attributes
////////////////////////////////////////////////////////////////
  
  /**
   * According to http://en.wikipedia.org/wiki/List_of_TCP_and_UDP_port_numbers, this
   * is the lowest possible "dynamic" port for which no applications are officially allowed
   * to reserve. 
   */
  public static final int MIN_DYNAMIC_PORT = 49152;
  /**
   * According to http://en.wikipedia.org/wiki/List_of_TCP_and_UDP_port_numbers, this
   * is the highest possible "dynamic" port for which no applications are officially allowed
   * to reserve. 
   */
  public static final int MAX_DYNAMIC_PORT = 65535;
  
  private static BDdfIpAdapter localHostAdapter = new BDdfIpAdapter("localhost","localhost","localhost");
  
  

}
