/*
 * Copyright 2008 Tridium, Inc. All Rights Reserved.
 */
package com.tridium.ddfHttp.comm.rsp;

import java.util.Enumeration;

import javax.baja.data.BIDataValue;
import javax.baja.net.HttpConnection;
import javax.baja.sys.BFacets;
import javax.baja.sys.BString;
import javax.baja.nre.util.Array;

import com.tridium.util.EscUtil;

/**
 * Converts the HTTP response header from the given HttpConnection into an instance
 * of BFacets whose facet keys are named the same as the names in the response
 * header and whose values are strings that are equal to the corresponding response
 * header values.
 * 
 * @author lperkins
 */
public class DdfHttpResponseHeaderUtil
{
  /**
   * Converts the HTTP response header from the given HttpConnection into an instance
   * of BFacets whose facet keys are named the same as the names in the response
   * header and whose values are strings that are equal to the corresponding response
   * header values.
   * 
   * @param connection the given HttpConnectino as described
   * 
   * @return the resulting BFacets as described
   */
  public static BFacets responseHeaderToFacets(HttpConnection connection)
  {
    // Gets all header names from response header
    Enumeration<String> rspHeaderNames = connection.getResponseHeaderNames();
    
    Array<String> facetKeys = new Array<>(String.class);
    Array<BIDataValue> facetValues = new Array<>(BIDataValue.class);
    
    // Adds the HttpConnection's std http response version to the set of key/value
    // pairs that this is building up in order to create a set of facets from
    facetKeys.add(HTTP_HEADER_RESPONSE_VERSION_KEY);
    
    // Gets the std http response version and makes sure that it is not null. In the
    // worst-case scenario, it will be an empty string instead1
    String responseVersion = connection.getResponseVersion();
    if (responseVersion==null)
      responseVersion = "";
    facetValues.add(BString.make(responseVersion));
    
    // Loops through all header names
    while (rspHeaderNames.hasMoreElements())
    {
      // Gets the next header name
      String rspHeaderName = rspHeaderNames.nextElement();
      
      // Gets the next header string value
      String rspHeaderVal = connection.getResponseHeader(rspHeaderName);
      
      facetKeys.add(EscUtil.slot.escape(rspHeaderName));
      facetValues.add(BString.make(EscUtil.slot.escape(rspHeaderVal)));
    }
    
    // Returns a set of facets for the response header key/value pairs
    return BFacets.make(facetKeys.trim(), facetValues.trim());
  }
  
  public static final String HTTP_HEADER_RESPONSE_VERSION_KEY = "ResponseVersion";

}
