/*
 * Copyright 2008 Tridium, Inc. All Rights Reserved.
 */
package com.tridium.ddfHttp.comm;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.LinkedList;
import java.util.ListIterator;

import javax.baja.net.Http;
import javax.baja.net.HttpException;
import javax.baja.net.UrlConnection;
import javax.baja.security.BUsernameAndPassword;
import javax.baja.sys.BComplex;
import javax.baja.sys.BValue;
import javax.baja.sys.BajaRuntimeException;
import javax.baja.sys.ServiceNotFoundException;
import javax.baja.sys.Sys;
import javax.baja.nre.util.IntHashMap;
import javax.baja.util.Lexicon;
import javax.baja.util.Queue;

import com.tridium.ddf.comm.BIDdfCommunicator;
import com.tridium.ddf.comm.req.BIDdfRequest;
import com.tridium.ddfHttp.comm.req.BIDdfHttpStreamRequest;
import com.tridium.platform.tcpip.BTcpIpPlatformService;

/**
 * This class encapsulates the details that must occur in order to
 * open an Http connection, possibly authenticate, and possibly post
 * form data from the 'forceTransmit' method on the outer instance
 * which occurs on the 'communicator' thread.
 *
 * This class also encapsulates the details that must occur in order
 * to read the content from the Http connection. This occurs on the
 * 'receive' thread.
 * 
 * The BDdfHttpTransmitter and BDdfHttpReceiver both rely on this class for
 * inter-thread synchronization. The BDdfHttpTransmitter calls the synchronized
 * 'transmitWhenReadyToReceive' method on the communicator thread when the next
 * devDriver request needs to be transmitted. The BDdfHttpReceiver calls the
 * synchronized 'waitReceiveNextAndClose' method on the receive thread in the
 * receive thread's tight loop to continually receive incoming dev driver
 * data frames. The 'waitReceiveNextAndClose' method blocks, waits for the
 * 'transmitWhenReadyToReceive' method to transmit over HTTP, 
 * 
 * While this is devHttpDriver's best attempt at HTTP transmit and receive, its creators realize that some
 * web servers could implement HTTP transmit and receive differently (hopefully just slightly). For that reason,
 * all fields are granted protected access and all methods are split up into many
 * smaller methods, all of which are granted protected access. This should allow
 * developers to customize any pieces of the authentication that might be necessary on a
 * per-driver basis.
 * 
 * To plug-in the custom DdfHttpHelper (which uses the custom DdfAuthenticationHelper, override
 * the 'makeHttpHelper' method on the driver's transmitter class which extends BDdfHttpTransmitter.
 * 
 * @see DdfAuthenticationHelper
 * @see BDdfHttpTransmitter
 * @see BDdfHttpReceiver
 * 
 * @author    Lenard Perkins
 * @creation  01 Jan 08
 * @version   $Revision$ $Date: 02/19/2009 3:42:00 PM$
 * @since     Baja 1.0
 */
public class DdfHttpHelper
{

  /**
   * @param ddfHttpTransmitter
   */
  public DdfHttpHelper(BDdfHttpTransmitter ddfHttpTransmitter)
  {
    this.ddfHttpTransmitter = ddfHttpTransmitter;
  }
  
  /**
   * ASSUMPTIONS: httpCommunicator is not null
   * @return
   */
  protected String getWatchdogThreadName()
  {
    String watchdogThreadName = "ConnectionWatchdog";

    String suffix = getThreadSuffix(httpCommunicator);
    if (suffix!=null && suffix.length()>0)
    {
      watchdogThreadName += ':' + suffix;
    }
    watchdogThreadName = watchdogThreadName + ':' + httpCommunicator.getHandle();
    return watchdogThreadName;
  }
  
  protected static String getThreadSuffix(BIDdfCommunicator ddfCommunicator)
  {
    if (ddfCommunicator instanceof BComplex)
    {
      // Fails safely if the parent becomes null. This might happen
      // during client-side video stream communications if the Niagara
      // core attempts to cleanup the proxy version of this object.
      // Even if and after that occurs, this object needs to still be
      // able to function with throwing a null pointer exception.
      if (((BComplex)ddfCommunicator).getParent()!=null)
        return ((BComplex)ddfCommunicator).getParent().getName();
      else
        return "";
    }
    else
      return "";

  }  
  
  /**
   * Cleans up the list
   */
  public void stopHelping()
  {
    httpHelperStopped=true;
    cleanupListOfOutstandingUrlConnections();
  }

  /**
   * Calls 'close' on all UrlConnections in the {@link #openHttpConnections} list
   * that are still open. This method is called from the {@link #stopHelping()} method.
   */
  protected void cleanupListOfOutstandingUrlConnections()
  {
    ListIterator<UrlConnection> i = openHttpConnections.listIterator();
    
    while (i.hasNext())
    {
      UrlConnection potentiallyOpenUrlConnection = i.next();
      if (potentiallyOpenUrlConnection.isOpen())
      {
        potentiallyOpenUrlConnection.close();
      }
    }
  }

  /**
   * Waits for the 'receiverReadyToReceive' to be set to true and then calls
   * 'transmit' and passes in the given ddfRequst.
   *
   * @param ddfReqest
   *
   * @throws Exception
   */
  public synchronized void transmitWhenReadyToReceive(BIDdfRequest ddfRequest) throws Exception
  {
    if (httpHelperStopped)
    {
      throw new BajaRuntimeException("Ddf Http Helper Stopped");
    }
    else
    {
      if (! (ddfRequest instanceof BIDdfHttpStreamRequest) )
      {
        // If the 'waitReceiveNextAndClost' method  is not ready to receive then this
        // waits for it to notify
        if (!receiverReadyToReceive)
        {
          wait();
        }
      }
      // else, the given ddfRequest is a BIDdfHttpStreamRequest -- meaning that the developer
      // wants to manually process the HTTP data stream rather than allow the receiver to
      // process it as an ordinary dev driver transaction. The transmit method will directly
      // call the 'processHttpStream' method on the request
  
      try
      {
        transmit(ddfRequest);
      }
      catch (IOException ioe)
      {
        ddfHttpTransmitter.getDdfCommunicator().getLog().error(ioe.toString());
        
        // NOTE: BIDdfHttpStreamRequests do not get a devDrivertransaction, but
        // if it fails to even transmit then the transmit should be retried
        if (ddfRequest instanceof BIDdfHttpStreamRequest)
        {
          // Determines the number of permissible retries for the particular
          // request
          int remainingRetries = ddfRequest.getRemainingRetryCount();
          if (remainingRetries > 0) // If any retries remain for the particular
          {                         // request...
            
            // Consumes a retry
            ddfRequest.setRemainingRetryCount(remainingRetries - 1);
            
            // Updates transmission statistics
            ddfHttpTransmitter.setRetransmissionCount( ddfHttpTransmitter.getRetransmissionCount()+1);
            
            // Recurses to transmit again.
            transmitWhenReadyToReceive(ddfRequest);
          }
        }
      }
      catch (IcmpPingFailure icmpPingFailure)
      {
        String httpTxType = (mostRecentUrlPostData == null)?"GET":"POST";
        if (httpCommunicator.getLog().isTraceOn())
          httpCommunicator.getLog().trace("HTTP "+httpTxType+" Failed - "+icmpPingFailure.getLocalizedMessage());
        if (ddfRequest instanceof BValue)
        {
          // Causes the request to maybe be transmitted again
          httpCommunicator.getTransactionManager().checkOutstandingTimeout((BValue)ddfRequest);
        }
      }
    }
  }

  /**
   * Transmits the given 'ddfRequest' over Http by calling the 'toByteArray' method on it and using
   * the return as a url that optionally includes form data delimited from the url address by a
   * question mark -- just as web browsers currently post form data.
   * 
   * @param ddfRequest
   * 
   * @throws Exception
   */
  protected void transmit(BIDdfRequest ddfRequest) throws Exception
  {

    mostRecentDdfRequest = ddfRequest;
    httpCommunicator = this.ddfHttpTransmitter.getHttpCommunicator();

    // Parses the mostRecentUrlString, mostRecentRequestMethod, mostRecentUrlAddress,
    // mostRecentPostData,
    // and mostRecentJavaUrl from whatever is returned by calling 'toByteArray'
    // on the given ddfRequest
    parseJavaUrl();

    if (this.ddfHttpTransmitter.getDdfCommunicator().getLog().isTraceOn())
      trace();
    
    // Ensures that there is a watchdog thread running to monitor the
    // HTTP transmit (get/post) transaction that is about to take
    // place
    if (connectionWatchdogThread==null)
    {
      connectionWatchdogThread = new ConnectionWatchdogThread();
      connectionWatchdogThread.start();
    }

    // TODO: the following line was commented out to correct Issue 18052.
    // Clean up required.
    //PreemptConnectionAfterTimeout connectionPreemption = new PreemptConnectionAfterTimeout(ddfRequest);
    
    // Prepares to force a close on the httpConnection that the 'transmit' method
    // will establish. This force of close occurs after the response time interval
    // expires. This is the only way to force the return of a blocking 'read'
    // operation that could occur during the HTTP send/receive handshaking.
    //connectionPreemption.beginPreemptionCountdown();
    
    // Opens an Http connection to the parsed URL, performs authentication if necessary,
    // and posts data if necessary. This does not read any content back though.
    transmitHttp();

    // Cancels the outstanding attempt to preempt the HttpConnection.
    // TODO: the following line was commented out to correct Issue 18052.
    // Clean up required.
    //connectionPreemption.cancelPreemptionCountdown();
    
    if (ddfRequest instanceof BIDdfHttpStreamRequest)
    {
      if (httpCommunicator.getLog().isTraceOn())
        httpCommunicator.getLog().trace("HTTP <Rx Stream>");
      boolean shouldCloseAutomatically = true;
      try
      {
        // Allows the request to process the Http data stream
        shouldCloseAutomatically = 
          ((BIDdfHttpStreamRequest) ddfRequest).
            processHttpStream(mostRecentBajaUrlConnection);
      }
      catch (Exception e)
      {
        httpCommunicator.getLog().error(LEX.getText("DriverUnableToProcessHttpStream"), e);
        shouldCloseAutomatically = true;
      }
      finally
      {
        if (shouldCloseAutomatically)
        {
          // Closes the HTTP connection
          mostRecentBajaUrlConnection.close();
        }
        else
        {
          // TODO: Consider finding a better place to perform this housekeeping. It might
          // not be too bad to do it here since the purpose of this is to prevent a memory
          // leak and the outstanding streams would only 'leak' if the driver continuously
          // requests direct access to the stream without returning true from
          // BIDdfHttpStreamRequest.processHttpStream and thereby denying this helper
          // the opportunity to close the stream automatically.
          cleanupClosedOutstandingStreams();
          
          // Keeps track of the BajaUrlConnection so that this can
          // close it upon station shutdown, if necessary.
          openHttpConnections.add(mostRecentBajaUrlConnection);
        }
      }
    }
    else
    {
      // Notifies the receive thread that is blocked inside the 'waitReceiveNextAndClose' method.
      // The receive thread will wake up and process the response data
      notifyAll();
    }
  }
  
  /**
   * Removes references to any UrlConnections that were passed to a 
   * BIDdfStreamRequest that were not closed at the time but have
   * since been closed.
   */
  protected void cleanupClosedOutstandingStreams()
  {
    // Prepares to walk the linked list of openHttpConnections
    ListIterator<UrlConnection> i = openHttpConnections.listIterator();
    
    // Walks the linked list of openHttpConnections
    while (i.hasNext())
    {
      // Looks at the next UrlConnection that might still be open 
      UrlConnection urlConnection = i.next();
      
      // If the url connection has been closed then this removes it
      // from the linked list so that it can be garbage collected
      if (!(urlConnection.isOpen()))
      { 
        i.remove();
      }
    }
  }
  
  /**
   * Blocks on the instance of DdfHttpHelper until the DdfHttpHelper opens an Http connection and
   * transmits. After that, this reads the content from the Http connection, closes the Http
   * connection, and returns the Http content.
   *
   * @return the Http content from the next Http connection that is opened.
   *
   * @throws Exception
   *           passes along any I/O or InterruptedException that could occur as a result of
   *           waiting or performing Http I/O.
   */
  public synchronized byte[] waitReceiveNextAndClose() throws Exception, HttpException
  {
    try
    {
      if (httpHelperStopped)
      {
        throw new BajaRuntimeException("Ddf Http Helper Stopped");
      }
      else
      {
        // Notifies the 'transmit' method that this method is ready to receive
        receiverReadyToReceive = true;
        notifyAll();
  
        // Waits for the http transmitter to perform an http connection. When this
        // wakes up it will re-own the monitor on this object. That is
        // how a java synchronized...wait...notify works. The notify occurs on the
        // http transmitter's callback which occurs on the communicator thread
        wait();
  
        // Reads the Http return data and closes the stream.
        return readContentAndClose();
      }
    }
    finally
    {
      // This tells the 'transmit' method that this method is __not__ ready to
      // receive.
      receiverReadyToReceive = false;
    }
  }

  /**
   * Reads the content from the most recent baja url connection.
   *
   * @return the bytes of the conent from the most recent baja url connection.
   *
   * @throws Exception any I/O exception that might occur while reading content
   */
  protected byte[] readContent() throws Exception
  {
    return mostRecentBajaUrlConnection.readContent();
  }

  /**
   * Calls readContent() and then calls closeConnection()
   *
   * @return the byte array that readContent() returns
   *
   * @throws Exception any I/O exception that might occur when calling
   * the readContent() and closeConnection() methods.
   *
   * @see #readContent
   * @see #closeConnection()
   */
  public byte[] readContentAndClose() throws Exception, HttpException
  {
    
    verifyOk();
    byte[] content = readContent();
    
//    System.out.println("content="+new String(content));
    closeConnection();
    return content;
  }

  /**
   * Closes the most recent baja url connection. This is called on the receive thread
   * after reading the contents of an HTTP response.
   * 
   * @see #readContentAndClose()
   */
  protected void closeConnection()
  {
    mostRecentBajaUrlConnection.close();
  }
  
  /**
   * Sets mostRecentBajaUrlConnection to a new UrlConnection for the mostRecentJavaUrl.
   * 
   * This is called by the transmitHttp method to prepare for the actual HTTP GET or POST
   * operation.
   * 
   * @see #transmitHttp()
   * 
   * @throws Exception if anything goes wrong. This exception is passed up to a higher
   * level where it can be placed in the communicator's log.
   */
  protected void makeConnection()
    throws Exception
  {
    try
    {
      /* BTcpIpPlatformService tcpIpService = (BTcpIpPlatformService) */Sys.getService(BTcpIpPlatformService.TYPE);
    
      // NOTE: tcpIpService.ping throws an Exception if the ping fails
      //tcpIpService.doPing( new BPingArgs(mostRecentJavaUrl.getHost()));
    }
    catch(ServiceNotFoundException snfe)
    {
      // If the service is not available then that means that this is running on the proxy-side VM,
      // for example, for video streaming. This proceeds to attempt the http connection without
      // first attempting an ICMP ping to see if the field-device is reachable.
    }
    catch (RuntimeException rte) // NOTE: BTcpIpPlatformService.doPing (called above
    {                            //       throws a RuntimeException if the ICMP ping fails)
//      throw new IcmpPingFailure(rte.getLocalizedMessage());
    }

    // Creates a javax.baja.net.UrlConnection
    mostRecentBajaUrlConnection = new UrlConnection(mostRecentJavaUrl);

    mostRecentBajaUrlConnection.setTimeout(httpCommunicator.getSoTimeout(mostRecentDdfRequest));
  }
  
  /**
   * This attempts to perform the actual HTTP GET or POST operation. The operation will
   * be a POST if the 'toByteArray' method of the mostRecentDdfRequest returns an ascii
   * byte array with the String "POST " on the front. Otherwise, this attempts to perform
   * an HTTP GET.
   * 
   * @return the status code from the HTTP GET or POST connection
   * 
   * @see #transmitHttp()
   * @see #reviewStatusCode(int)
   * 
   * @throws Exception if anything goes wrong. This exception is passed up to a higher
   * level where it can be placed in the communicator's log.
   */
  protected int doGetOrPost()
    throws Exception
  {
    if (mostRecentRequestMethod.equals("GET"))
    {
      // Connects to the URL
      return mostRecentBajaUrlConnection.connect();
    }
    else
    // if (mostRecentUrlPostData != null)
    // Posts any form data that the mostRecentDdfRequest.toByteArray() method
    // might have placed in the URL to the left of a question mark character.
    {
      // Works around a bug in the net.jar whereby it needs to place an "expect" into
      // the request header per HTTP 1.1 specifications. This but is resolved in the
      // net.jar module of release 3.4.12 but could be present in any earlier revision
      //mostRecentBajaUrlConnection.setRequestHeader("expect", "100-continue");
      return mostRecentBajaUrlConnection.post(
           mostRecentUrlAddress,
           httpCommunicator.getHttpContentType(mostRecentDdfRequest),
           mostRecentUrlPostData.getBytes());
    }    
  }

  /**
   * Connects to the 'mostRecentJavaUrl' (authenticates first if possible) and performs
   * an HTTP GET or POST according to the return value of the 'toByteArray' method on
   * the mostRecentDdfRequest.
   * 
   * This calls the following extendable methods in order:
   * 1. makeConnection
   * 2. considerPreAuthorization
   * 3. doGetOrPost
   * 4. reviewStatusCode
   * 
   * @see #makeConnection()
   * @see #considerPreAuthentication()
   * @see #doGetOrPost()
   * @see #reviewStatusCode(int)
   * 
   * @throws Exception if anything goes wrong. This exception is passed up to a higher
   * level where it can be placed in the communicator's log.
   */
  protected void transmitHttp() throws Exception
  {
    // Creates the javax.baja.net.UrlConnection object to use for the HTTP transmission
    makeConnection();
      

    // Attempts to add authentication headers if possible, based on any previous
    // authentication to the url
    considerPreAuthentication();

    // Performs the HTTP transmission
    int statusCode = doGetOrPost();
    
    // Reviews the resulting status code (considers performing initial authentication)
    reviewStatusCode(statusCode);
    
    
  }
  
  /**
   * This is called from the 'transmitHttp' method after attempting the HTTP GET or POST.
   * 
   * This reviews the HTTP status code and takes any necessary action as a result of the
   * HTTP status code. At present, this checks if the status code is a code 401 (401=
   * unauthorized), attempts to add authentication to the HTTP request, and attempts
   * to retry the HTTP GET or POST with the user name, password and appropriate encryption
   * algorithm (basic vs. digest) 
   * 
   * @param statusCode the status code returned from the most recent attempt to perform
   * an HTTP GET or POST.
   * 
   * @see #transmitHttp()
   * 
   * @throws Exception if anything goes wrong. This exception is passed up to a higher
   * level where it can be placed in the communicator's log.
   */
  protected void reviewStatusCode(int statusCode)
    throws Exception
  {
    // If the HTTP transmission attempt results in a code 401 
    if (statusCode == Http.SC_UNAUTHORIZED)
    {
      // Then this attempts authentication based on the WWW-Authenticate header in the
      // 401 response.
      addAuthenticationBasedOnHttpResponse();
      
      // Performs the HTTP transmission again, this time with the encrypted log-in
      // information
      doGetOrPost();
    }
  }
  
  protected void verifyOk()
    throws HttpException
  {
    mostRecentBajaUrlConnection.checkOk();
  }
  
  /**
   * Gets the BUsernameAndPassord to use for the current HTTP transmission.
   * 
   * @see #addAuthenticationBasedOnHttpResponse()
   * 
   * @return ddfHttpTransmitter.getHttpCommunicator().getHttpCredentials(mostRecentDdfRequest) 
   */
  protected BUsernameAndPassword getUserNameAndPasswordForTransaction()
  {
    return (BUsernameAndPassword)ddfHttpTransmitter.
                                 getHttpCommunicator().
                                 getHttpCredentials(mostRecentDdfRequest);
  }
  
  /**
   * Creates a DdfAuthenticationHelper to authenticate the connection to the
   * mostRecentBajaUrlConnection (which just failed with an HTTP code 401)
   * using the given user name and password credentials.
   * 
   * @param credentials the user name and password to use to log into the
   * web server in order to gain access to the mostRecentBajaUrlConnection.
   * 
   * @return a new instance of DdfAuthenticationHelper to authenticate the
   * connection to the mostRecentBajaUrlConnection.
   */
  protected DdfAuthenticationHelper makeAuthenticationHelper(BUsernameAndPassword credentials)
  {
    return new DdfAuthenticationHelper(
                                       mostRecentJavaUrl,
                                       credentials,
                                       mostRecentBajaUrlConnection );
  }
  
  /**
   * Adds authentication to the mostRecentBajaUrlConnection based on the HTTP response
   * header of the most recent failed (http code 401) response header in the mostRecentBajaUrlConnection.
   * 
   * This is called after a GET or POST is made to a URL that fails with HTTP code 401
   * (401=unauthorized). This reviews the response header for the failure. From the
   * response code, this determines whether the URL requires basic or digest authentication.
   * Then this adds the appropriate request header back onto the mostRecentBajaUrlConnection.
   * This allows for another GET or POST attempt but hopefully with the correction username
   * and log-in credentials.
   * 
   * @see #reviewStatusCode(int)
   * @see #transmitHttp()
   * 
   * @throws Exception if anything goes wrong. This exception is passed up to a higher
   * level where it can be placed in the communicator's log.
   */
  protected void addAuthenticationBasedOnHttpResponse() throws Exception
  {
    // Gets the user name and password for the BIDdfRequest
    BUsernameAndPassword credentials = getUserNameAndPasswordForTransaction();
    
    // Creates a DdfAuthenticationHelper for the URL and credentials
    DdfAuthenticationHelper authenticationHelper = makeAuthenticationHelper(credentials);
    
    // Stores the authentication helper for pre-authorization during the next HTTP
    // transaction to the same URL. Stores in a hash table, hashed by
    // url/username/password for constant look-up time during future retrieval
    authenticationHelpers.put(
      computeHashCodeForAuthenticationHelper(mostRecentJavaUrl, credentials),
      authenticationHelper );  
    
    // If the other end of the connection wishes for us to completely close
    // down the HTTP connection 
    if (mostRecentBajaUrlConnection.shouldClose())
    {
      // Closes the HTTP connection
      mostRecentBajaUrlConnection.close();
      
      // Creates a new HTTP connection object but does not connect just yet.
      makeConnection();
    }    
    
    // Adds the authorization (basic base 64 or digest) to the HTTP header
    authenticationHelper.addRequestAuthorization(mostRecentBajaUrlConnection);
  }
  
  /**
   * Computes a hash code taken from the string that results when combining the url, username, and password.
   */
  protected int computeHashCodeForAuthenticationHelper(URL javaUrl, BUsernameAndPassword credentials)
  {
    //System.out.println(">>>> mostRecentUrlRightOfQuestionMark="+mostRecentUrlRightOfQuestionMark);
    String obj = new String(mostRecentUrlRightOfQuestionMark + ':' + credentials.getUsername() + ':' + AccessController.doPrivileged((PrivilegedAction<String>)credentials.getPassword()::getValue));
    
    //System.out.println(">>>> obj="+obj);
    
    return obj.hashCode();
  }  

  /*
   * Checks if the httpCommunicator.getAuthenticate() method returns true. If so then this adds a
   * Base64 authorization to the Http Request Header
   */
  protected void considerPreAuthentication()
    throws Exception
  {
    BUsernameAndPassword credentials = (BUsernameAndPassword)
                                        ddfHttpTransmitter.
                                        getHttpCommunicator().
                                        getHttpCredentials(mostRecentDdfRequest);
    
    int helperCode = computeHashCodeForAuthenticationHelper(mostRecentJavaUrl, credentials);
    
    DdfAuthenticationHelper authenticationHelper = (DdfAuthenticationHelper)authenticationHelpers.get(helperCode);
    
    if (authenticationHelper != null)
    {
      authenticationHelper.addRequestAuthorization(mostRecentBajaUrlConnection);
    }
  }

  /*
   * Sets 'mostRecentJavaUrl' equal to a java.net.URL that encapsulates
   * the address portion of the url string that is returned by the
   * mostRecentDdfRequest.toByteArray method.
   *
   * @throws MalformedURLException and MalformedURLException exception
   * that the java.net.URL constructor might throw.
   */
  protected void parseJavaUrl() throws MalformedURLException
  {
    parseUrlString();
    parseRequestMethod();
    parseAddressAndData();
    mostRecentJavaUrl = new URL(mostRecentUrlAddress);
  }

  protected void trace()
  {
    StringBuffer buf = new StringBuffer("HTTP ");
    if (mostRecentUrlPostData == null)
    {
      buf.append("GET ");
    }
    else
    {
      buf.append("POST");
    }
    buf.append(mostRecentUrlString);
    httpCommunicator.getLog().trace(buf.toString());
  }

  /*
   * Calls toByteArray on the mostRecentDdfRequest and saves the return
   * value as the bytes of the mostRecentUrlString string.
   */
  protected void parseUrlString()
  {
    // The request's byte array encodes the URL
    mostRecentUrlString = new String(mostRecentDdfRequest.toByteArray());

  }

  /*
   * Checks if the URL string has the text
   */
  protected void parseRequestMethod()
  {
    int indexOfSpace = mostRecentUrlString.indexOf(' ');
    if (indexOfSpace >= 0)
    {
      mostRecentRequestMethod = mostRecentUrlString.substring(0, indexOfSpace).trim().toUpperCase();
    }
    else
      mostRecentRequestMethod = "GET";

  }

  /*
   * Reviews the 'mostRecentUrlString' for the Url address and the form
   * data to post. Any form data to post will be encoded to the right of
   * an optional question mark. The url address is located to the right
   * of the optional question mark.
   *
   * If there is no question mark then the 'mostRecentUrlPostData' will
   * be set to null and the 'mostRecentUrlAddress' will contain the
   * entire 'mostRecentUrlString'
   */
  protected void parseAddressAndData()
  {
    // Gets index of '?' in the mostRecentUrlString
    int urlIndexOfSpace = mostRecentUrlString.indexOf(" ");
    int urlIndexOfQuestionMark = mostRecentUrlString.indexOf("?");
    //int urlIndexOfEqualsSign = mostRecentUrlString.indexOf("=");

    if (urlIndexOfSpace < 0)
      urlIndexOfSpace = 0;

    // If there is a '?' in the mostRecentUrlString and a '=' after the question mark
    // __AND__ this is not a GET
    if ((urlIndexOfQuestionMark >= 0) && !(mostRecentRequestMethod.equals("GET")))
    {
      // Then the mostRecentUrlAddress is the left of the '?'
      mostRecentUrlAddress =
          mostRecentUrlString.substring(urlIndexOfSpace + 1, urlIndexOfQuestionMark);
      
      mostRecentUrlRightOfQuestionMark = mostRecentUrlAddress;

      // And the form data is the right of the '?'
      mostRecentUrlPostData = mostRecentUrlString.substring(urlIndexOfQuestionMark + 1);
    }
    else
    {
      
      // Else, the mostRecentUrlAddress is the entire mostRecentUrlString
      mostRecentUrlAddress = mostRecentUrlString.substring(urlIndexOfSpace);
      
      if (urlIndexOfQuestionMark >= 0)
        mostRecentUrlRightOfQuestionMark = mostRecentUrlString.substring(urlIndexOfSpace, urlIndexOfQuestionMark);
      else
        mostRecentUrlRightOfQuestionMark = mostRecentUrlAddress;
      

      // And there is no form data
      mostRecentUrlPostData = null;
    }
  }
  
  /**
   * An instance of this exeption is thrown and caught internally
   * when trying to establish an HTTP connection if an ICMP ping
   * fails to get a response from the other end of the HTTP
   * connection.
   * 
   * @author lperkins
   *
   */
  public class IcmpPingFailure extends Exception
  {
    public IcmpPingFailure(String localizedMessage)
    {
      super(localizedMessage);
    }
  }
  
  public class PreemptConnectionAfterTimeout
    implements Runnable
  {
    public long interval;
    public volatile boolean needToPreEmptUponWake;
    
    /**
     * Assumption: DdfHttpHelper.this.httpCommunicator is not null.
     */
    public PreemptConnectionAfterTimeout(BIDdfRequest ddfRequest)
    {
      needToPreEmptUponWake = true;

      // Prepares to preempt the HTTP connection after the
      // response timeout interval
      interval = httpCommunicator.getSoTimeout(ddfRequest);

      // Safeguards against not guarding a send-only transaction
      if (interval <= 0)
      {
        interval = httpCommunicator.getHttpReceiver().getResponseTimeout().getMillis();
      }
    }
    
    public synchronized void run()
    {
      try
      {
        if (httpCommunicator.getLog().isTraceOn())
        {
          httpCommunicator.getLog().trace("Countdown to pre-empt in "+interval+" millis.");
        }
        
        // Waits for the 'interval'. If this object is not
        // notified during the wait then the logic will
        // proceed to force a close on the 'mostRecentBajaUrlConnection'
        wait(interval);
        
        if (needToPreEmptUponWake && mostRecentBajaUrlConnection!=null)
        {
          if (httpCommunicator.getLog().isTraceOn())
          {
            httpCommunicator.getLog().trace("Pre-empting HTTP connection.");
          }
          
          // With HTTP sockets, this is sometimes the only way
          // to wake up a blocking 'read' operation. That is
          // pretty much what this is guarding against
          mostRecentBajaUrlConnection.close();
        }
        else
        {
          if (httpCommunicator.getLog().isTraceOn())
          {
            httpCommunicator.getLog().trace("No need to pre-empt HTTP connection.");
          }
          
        }
      }
      catch (InterruptedException e)
      {
        // Silently returns so that the 'while' loop can
        // return gracefully if 'isAlive' is set to
        // false. Otherwise, the thread continues to process
        // work items. The thread is only intended to be
        // interrupted if its stopWatchdog method is called
        // or if 
      }
    }
    
    public synchronized void cancelPreemptionCountdown()
    {
      needToPreEmptUponWake = false;
      notifyAll();
    }
    
    public void beginPreemptionCountdown()
    {
      connectionWatchdogThread.watchdogQ.enqueue(this); 
    }
  }
  
  
  public class ConnectionWatchdogThread
    extends Thread
  {
    public Queue watchdogQ;
    public boolean isAlive;
    
    /**
     * Assumption: DdfHttpHelper.this.httpCommunicator is not null.
     */
    public ConnectionWatchdogThread()
    {
      super(getWatchdogThreadName());
      watchdogQ = new Queue();
    }
    
    public void stopWatchdog()
    {
      isAlive = false;
      interrupt();
    }
    
    public void run()
    {
      isAlive = true;
      while(isAlive)
      {
        try
        { 
          // wait for work      
          Runnable work = watchdogQ.todo(500);
          
          process(work);
        }
        catch(InterruptedException e)
        {
          // silent
        }
        catch(Throwable e)
        {
          e.printStackTrace();
        }
      }
    }
      
    /**
     * This method is called on the worker's thread when a 
     * new item of work is available.  If the todo timed out
     * then null is passed.
     */
    protected void process(Runnable work)
      throws Exception
    {
      if (work != null) work.run();
    }       
    
  }

  /**
   * This is the entire url that is received as a result of calling 'toByteArray'
   * on the BIDdfRequest that was most recently passed to the 'transmit'
   * method. If the driver developer wishes to post form data then that data
   * is delimited from the URL by a question mark.
   */
  protected String mostRecentUrlString;

  /**
   * This is a string that indicates the HTTP request type for the most recent
   * call to the 'transmit' method. For example, the will be "GET" or "POST"
   */
  protected String mostRecentRequestMethod = null;

  /**
   * This is the address portion of the mostRecentUrlString. The address portion is the
   * string content on the left-side of the optional question mark delimiter. It
   * will be the entire mostRecentUrlString if the mostRecentUrlString does not have a question mark
   * in it.
   */
  protected String mostRecentUrlAddress = null;

  /**
   * This is the portion of the mostRecentUrlString to the right of the question mark, if any.
   */
  String mostRecentUrlRightOfQuestionMark = null;

  /**
   * This is the form data portion of the mostRecentUrlString. The form data is the part
   * of the string content on the right-side of the optional question mark
   * delimiter. This will be null if the mostRecentUrlString does not have a question
   * mark.
   */
  protected String mostRecentUrlPostData = null;

  /**
   * This is the BIDdfRequest that was most recently passed to the 'transmit'
   * method.
   */
  protected BIDdfRequest mostRecentDdfRequest;

  /**
   * This is a java.net.URL object that represents the mostRecentUrlString.
   */
  protected URL mostRecentJavaUrl = null;

  /**
   * This is the BDdfHttpCommunicator that was returned when the 'trasmit' method
   * last called the getHttpCommunicator() method.
   */
  protected BDdfHttpCommunicator httpCommunicator = null;

  /**
   * This is the javax.baja.net.UrlConnection object that was created and connected
   * to during the most recent call to the 'transmit' method.
   */
  protected UrlConnection mostRecentBajaUrlConnection = null;
  
  /**
   * This is used a part of the inter-thread communication between the 'transmitWhenReadyToReceive'
   * method (called on the communicator thread) and the 'waitToReceiveNextAndClose()' method
   * (called on the receiver thread).
   *
   * More specifically, the 'transmitWhenReadyToReceive' method will first check this variable. If
   * this variable is false then it will wait.
   *
   * The 'waitToReceiveNextAndClose' method will first set this variable to true and then notify
   * the 'transmitWhenReadyToReceive' method.
   */
  protected boolean receiverReadyToReceive = false;
  
  /**
   * This variable is used to prevent the 'waitReceiveNextAndClose' method from
   * blocking forever it the receive thread tries to receive after the driver
   * has been shut down.
   */
  protected boolean httpHelperStopped = false;

  /**
   * This is the BDdfHttpTransmitter that is passed to the constructor.
   */
  protected BDdfHttpTransmitter ddfHttpTransmitter;

  /**
   * This is an array of UrlConnections for all BIDdfHttpStreamRequests
   * that return 'false' from their 'processHttpStream' method. Keeping
   * track of this array allows the UrlConnections to at least be closed
   * upon station shut down. This array is also checked periodically
   * to see if any UrlConnections are subsequently closed. If so then
   * the closed UrlConnections are removed from the Array so as to
   * prevent a memory leak.  
   */
  protected LinkedList<UrlConnection> openHttpConnections = new LinkedList<>();
  
  /**
   * This hashes DdfAuthenticationHelper objects to Url/UserName/Password hash codes.
   */
  protected IntHashMap authenticationHelpers = new IntHashMap();
  
  protected ConnectionWatchdogThread connectionWatchdogThread;
  
  /**
   * This provides access to the localized text lexicon of the devHttpDriver module.
   */
  public static final Lexicon LEX = Lexicon.make(DdfHttpHelper.class);

}