/*
 * @copyright 2005 Tridium Inc.
 */
package com.tridium.ddf.ui.point;

import javax.baja.control.BControlPoint;
import javax.baja.driver.point.BProxyExt;
import javax.baja.driver.ui.point.PointController;
import javax.baja.sys.BComponent;
import javax.baja.sys.Context;
import javax.baja.ui.BWidget;
import javax.baja.ui.CommandArtifact;
import javax.baja.ui.pane.BBorderPane;
import javax.baja.ui.pane.BEdgePane;
import javax.baja.nre.util.Array;
import javax.baja.workbench.mgr.BAbstractManager;
import javax.baja.workbench.mgr.MgrEdit;
import javax.baja.workbench.mgr.MgrEditRow;
import javax.baja.workbench.mgr.MgrTypeInfo;

import com.tridium.ddf.IDdfFacetConst;
import com.tridium.ddf.discover.BDdfPointDiscoveryLeaf;
import com.tridium.ddf.discover.BIDdfDiscoveryLeaf;
import com.tridium.ddf.discover.BIDdfPointDiscoveryLeaf;
import com.tridium.ddf.identify.BDdfIdParams;
import com.tridium.ddf.point.BDdfProxyExt;
import com.tridium.ddf.ui.DdfMgrControllerUtil;
import com.tridium.ddf.ui.DdfMgrControllerUtil.DdfPointMgrAgentCommand;

/**
 * This is the MgrController for the BDdfPointManager.
 *
 * I had to customize the MgrEdit in various ways to allow this to
 * update the control point and proxy extension fields without
 * using MgrColumns.
 *
 * @author lperkins
 */
public class DdfPointController
  extends PointController
  implements IDdfFacetConst
{
  public DdfPointController(BDdfPointManager manager)
  {
    super(manager);
  }

////////////////////////////////////////////////////////////////
// MgrController
////////////////////////////////////////////////////////////////

  /**
   * Uses a DdfMgrPointEdit (that extends MgrEdit) instead of simply an
   * MgrEdit. The return object processes the dialog box that appears
   * when adding points to the database.
   */
  public MgrEdit makeEdit(String label)
  {
    return new DdfMgrPointEdit(getManager(), label);
  }

  /**
   * Called when the user clicks the "Discover" button. This calls
   * DdfMgrControllerUtil.doDiscover.
   *
   * @see DdfMgrControllerUtil.doDiscover
   */
  public CommandArtifact doDiscover(Context cx)
    throws Exception
  {
    return DdfMgrControllerUtil.doDiscover(getManager(),cx);
  }

  /**
   * Adds IMgrCommands for each BIDdfPointMgrAgent that
   * is declared on the device or point-device-ext.
   */
  protected IMgrCommand[] makeCommands()
  {
    return DdfMgrControllerUtil.makeCommands(
        super.makeCommands(),
        (BDdfPointManager)getManager());
  }

  /**
   * Enables/Disables the IMgrCommands for each BIDdfPointMgrAgent that
   * is declared on the device or point-device-ext.
   */
  public void updateCommands()
  {
    DdfMgrControllerUtil.updateCommands(getManager());
    super.updateCommands();
  }

  /**
   * Overrides the default action bar to add a second row of buttons
   * to contain the developer's button agents (provided that the developer
   * adds two or more of them). If the developer only adds one button
   * to the manager, then this will still just use a single row of
   * buttons.
   */
  public BWidget makeActionBar()
  {
    // Gets all IMgrCommands for this device
    IMgrCommand[] mgrCommands = getCommands();

    // Allocates two array buffers, one to hold the framework commands
    // the other to hold the developer (user) commands
    Array<IMgrCommand> firstRow= new Array<>(IMgrCommand.class);
    Array<IMgrCommand> secondRow = new Array<>(IMgrCommand.class);

    // Loops through the mgrCommands and divides them between the firstRow
    // and secondRow. The firstRow contains those commands that are added
    // by the framework itself. The secondRow contains those commands that
    // are added by the developer by means of Ddf manager agents
    for (int i=0; i<mgrCommands.length; i++)
    {
      if (mgrCommands[i] instanceof DdfPointMgrAgentCommand)
        secondRow.add(mgrCommands[i]);
      else
        firstRow.add(mgrCommands[i]);
    }

    // If the secondRow would not have at least two items in it then
    // let's just use one row...calling super.makeActionBar will
    // accomplish this.
    if (secondRow.size() < 2 )
    {
      return super.makeActionBar();
    }
    else
    {
      IMgrCommand[] row1Commands = firstRow.trim();
      IMgrCommand[] row2Commands = secondRow.trim();

      BEdgePane pane = new BEdgePane();
      pane.setTop(makeActionPane(row1Commands));
      pane.setCenter(new BBorderPane(makeActionPane(row2Commands)));

      return pane;
    }
  }

  /**
   * Initializes the standard properties on the given control point
   * and its proxy extension. These properties include the point facets,
   * deviceFacets, conversion, and tuningPolicyName.
   *
   * @param discovery
   *
   * @param newCtPt
   */
  public static void initializeControlPoint(Object discovery, BControlPoint newCtPt)
  {
    if (discovery instanceof BDdfPointDiscoveryLeaf)
    {
      // Initializes the proxy ext's 'Read Parameters', 'Write Parameters', and
      // 'Point Id'. This is specifically beneficial in that it updates even those
      // Values that do not have a column in the "Discovered" list.
      BDdfPointDiscoveryLeaf discoveryLeaf = (BDdfPointDiscoveryLeaf)discovery;

      BDdfProxyExt ddfProxyExt = (BDdfProxyExt)newCtPt.getProxyExt();

      ddfProxyExt.setReadParameters((BDdfIdParams)discoveryLeaf.getReadParameters().newCopy());
      ddfProxyExt.setWriteParameters((BDdfIdParams)discoveryLeaf.getWriteParameters().newCopy());
      ddfProxyExt.setPointId((BDdfIdParams)discoveryLeaf.getPointId().newCopy());
    }
    // If the discovery object is a BIDdfPointDiscoveryLeaf then we might be able to
    // Assign some more initial values to the control point and/or its proxy
    if (discovery instanceof BIDdfPointDiscoveryLeaf)
    {
      BIDdfPointDiscoveryLeaf ptDis = (BIDdfPointDiscoveryLeaf)discovery;

      if (ptDis.getDiscoveryPointFacets()!=null) // Possibly initializes the point facets
        newCtPt.setFacets(ptDis.getDiscoveryPointFacets());

      BProxyExt proxy = (BProxyExt)newCtPt.getProxyExt();

      if (ptDis.getDiscoveryDeviceFacets()!=null)  // Possibly initializes the device facets
        proxy.setDeviceFacets(ptDis.getDiscoveryDeviceFacets());

      if (ptDis.getDiscoveryConversion()!=null) // Possibly initializes the 'conversion' object
        proxy.setConversion(ptDis.getDiscoveryConversion());

      if (ptDis.getDiscoveryTuningPolicyName()!=null) // Possibly initilizes the tuning policy name
        proxy.setTuningPolicyName(ptDis.getDiscoveryTuningPolicyName());
    }
  }

////////////////////////////////////////////////////////////////
// DdfMgrEditRow
////////////////////////////////////////////////////////////////

  /**
   * This class overcomes a side-effect of the DdfMgrPointEdit class, which
   * this framework uses instead of the standard MgrEdit class.
   *
   * Customizes the MgrEditRow.setType method to stop it from eliminating
   * the initializations that this framework makes to new control points.
   *
   * The aforementioned initializations occur in the DdfMgrPointEdit.addRow
   * method.
   *
   * This plus into the existing MgrEdit framework by returning
   * an instance of this from the DdfMgrPointEdit.makeRow method.
   *
   * @author lperkins
   */
  class DdfMgrEditRow
    extends MgrEditRow
  {
    public DdfMgrEditRow(BComponent target, Object discovery, MgrTypeInfo[] types)
    {
      super(target,discovery,types);
    }

    /**
     * Overrides MgrEditRow.setType (called if the user changes the type of a row from
     * the MgrEdit dialog). The super implementation changes the target to
     * getManager().getModel().newInstance(MgrTypeInfo) but that eliminates any customizations
     * that this framework makes to the point facets, device facets, conversion, and tuningPolicyName.
     *
     * This overridden version overcomes the aforementioned side-effect.
     */
    public void setType(MgrTypeInfo type)
      throws Exception
    {
      // The call to super.setType(type) eliminates the default name,
      // but not the custom name if the user provided one. The gets
      // whatever the name was before the Type changes
      String nameBeforeTypeChange = getName();

      // Changes the type
      super.setType(type);

      // This undoes the side-effect of super.setType resetting the target
      initializeControlPoint(getDiscovery(),(BControlPoint)getTarget());

      // Loads the cells (field editors, I believe) to make the new initialized
      // values take effect, if they have corresponding fields on the Mgr Edit dialog.
      loadCells();

      // Updates the row that is being edited, to reflect the new initialization
      // That was done to the target
      if (getDiscovery() != null)
        getLearn().toRow(getDiscovery(), this);

      // The call to super.setType followed by loadCells eliminates
      // the name completely. This restores the name as it was before
      // processing the type change.
      setName(getEdit().getUniqueName(this,nameBeforeTypeChange));
    }

  } // End of inner class DdfMgrEditRow


////////////////////////////////////////////////////////////////
// DdfMgrPointEdit
////////////////////////////////////////////////////////////////

  /**
   * The DdfMgrPointEdit is used instead of MgrEdit for the doAdd functionality.
   *
   * NOTE: The ancestor's doAdd calls makeEdit, which DdfPointController redefines.
   *
   * DdfPointController.makeEdit returns an instance of this.
   *
   * This allows the ddf to initialize the proxy extension before displaying the edit
   * dialog, that way the facets, tuning policy name, etc. will take effect in
   * the edit dialog (if the user has facet'ed them to be shown) without having
   * to set them using the MgrColumn set api's. Using the MgrColumn set api's would
   * require an assumptiong that the developer facet'ed the most of the proxy's properties
   * as being in the MgrEdit dialog. Since this cannot be assured, it is invalid to make
   * such an assumption .
   *
   * @author lperkins
   *
   */
  class DdfMgrPointEdit
    extends MgrEdit
  {
    /**
     * Constructor -- This basically just calls the super constructor.
     */
    public DdfMgrPointEdit(BAbstractManager manager, String title)
    {
      super(manager,title);
    }

    /**
     * Overrides the MgrEdit.addRow method and initializes the standard fields on
     * the control point and on the proxy extension to the values specified by the
     * discovery object.
     */
    public MgrEditRow addRow(Object discovery, MgrTypeInfo[] types)
      throws Exception
    {
      // Makes a new instance of control point, to ultimately be the target
      BControlPoint newCtPt = (BControlPoint)getManager().getModel().newInstance(types[0]);

      // If the discovery object is a BIDdfPointDiscoveryLeaf then we might be able to
      // Assign some more initial values to the control point and/or its proxy
      initializeControlPoint(discovery, newCtPt);

      // Creates the MgrEditRow that will appear in the edit dialog and adds
      // It to the MgrEditRow dialog
      MgrEditRow newMgrRow = addRow(makeRow(newCtPt,discovery,types));

      // Possibly sets the default name in the MgrEditRow, if the driver developer
      // Chose to specify one.
      if (discovery instanceof BIDdfDiscoveryLeaf)
      {
        BIDdfDiscoveryLeaf ptDis = (BIDdfDiscoveryLeaf)discovery;
        if (ptDis.getDiscoveryName()!=null)
          newMgrRow.setDefaultName(ptDis.getDiscoveryName());
      }
      return newMgrRow;
    }

    /**
     * Overrides MgrEdit.makeRow to return an instance of
     * DdfMgrEditRow instead of just MgrEditRow.
     *
     *  @see DdfMgrEditRow for more details.
     */
    protected MgrEditRow makeRow(BComponent target, Object discovery, MgrTypeInfo[] types)
      throws Exception
    {
      return new DdfMgrEditRow(target, discovery, types);
    }

  } // End of inner class DdfMgrPointEdit

} // End of class DdfPointController
