/*
 * @copyright 2005 Tridium Inc.
 */
package com.tridium.ddf.ui.device;

import javax.baja.driver.BIDeviceFolder;
import javax.baja.sys.Type;
import javax.baja.util.Lexicon;
import javax.baja.workbench.mgr.BAbstractManager;

/**
 * This class contains methods and attributes that help clean up other areas
 * of the source code that are used to display localized text.
 * 
 * @author lperkins
 */
public class DdfDeviceManagerLexicon
{
  public static final Lexicon LEX = Lexicon.make(DdfDeviceManagerLexicon.class);
  
  // For example: The device type "BModbusDevice" returned by the getDeviceType method of "BModbusNetwork" does not extend BDdfDevice.
  // NetworkDeviceTypeNotDdfDevice = The device type "{0}" returned by the getDeviceType method of "{1}" does not extend BDdfDevice.
  public static final String NETWORK_DEVICE_TYPE_NOT_DEV_DEVICE = "NetworkDeviceTypeNotDdfDevice";
  
  public static String networkDeviceTypeNotDdfDevice(Type networkType, Type deviceType)
  {
    return LEX.getText(NETWORK_DEVICE_TYPE_NOT_DEV_DEVICE,
        new Object[]{networkType.getDisplayName(null),deviceType.getDisplayName(null)});
  }
  
  // DiscoveryParameters = Discovery Parameters
  public static final String DISCOVERY_PARAMETERS = "DiscoveryParameters";
  
  public static final String discoveryParameters = LEX.getText(DISCOVERY_PARAMETERS);

  
  
  // NOTE: Parameter {0} is the name of the workbench manager type that exhibits the problem that requires this message to be displayed to the user!
  // NOTE: Parameter {1} is the name of a module that extends from the DdfDriver framework
  // For example: Cannot find a ddf Discovery Host to associate with this workbench manager. Please have the "Modbus" driver fixed.
  // CannotFindDdfDiscoveryHostForWorkbenchManager = Cannot find a ddf Discovery Host to associate with the "{0}" workbench manager. Please have the "{1}" driver fixed.
  public static final String CANNOT_FIND_DEV_NETWORK_FOR_DEVICE_MANAGER = "CannotFindDdfNetworkForDeviceManager";
  
  public static String cannotFindDdfDiscoveryHostForWorkbenchManager(BAbstractManager ezMgr)
  {
    return LEX.getText(CANNOT_FIND_DEV_NETWORK_FOR_DEVICE_MANAGER,
        new Object[]{
          ezMgr.getType().getDisplayName(null),
          ezMgr.getType().getModule().getModuleName()} );
  }
  
  // This is the key for "DeviceTypeCannotBeAbstract"
  // This error message appears in the stdout of Workbench on the client-side if a user visits a driver's device manager but the driver's network and/or
  // device folder returns an abstract Type from the getDeviceType() method.
  // NOTE: Parameter {0} indicates the location of the 'getDeviceType()' method that the developer needs to fix.
  // NOTE: Parameter {1} is the device class followed by ".TYPE"
  // NOTE: Parameter {2} is the module:type typeSpec of the type that is abstract.
  public static final String DEVICE_TYPE_CANNOT_BE_ABSTRACT = "DeviceTypeCannotBeAbstract";
  
  public static String deviceTypeCannotBeAbstract( BIDeviceFolder offendingDeviceFolder, Type deviceType )
  {
    // Gets the folder class name (minus the package name) of 'offendingDeviceFolder' in a string
    Class<?> offendingDeviceFolderClass = offendingDeviceFolder.getClass();
    String offendingDeviceFolderClassName = offendingDeviceFolderClass.getName();
    String shortOffendingFolderClassName = offendingDeviceFolderClassName.substring( offendingDeviceFolderClassName.lastIndexOf(".")+1 );
    
    // This will be placed in parameter 0 for the lexicon look-up
    String folderHintString = shortOffendingFolderClassName + ".getDeviceType()";
    
    // Gets the device class name (minus the package name) of the deviceType's class
    Class<?> offendingDeviceClass = deviceType.getTypeClass();
    String offendingDeviceClassName = offendingDeviceClass.getName();
    String shortOffendingDeviceClassName = offendingDeviceClassName.substring( offendingDeviceClassName.lastIndexOf(".")+1 );

    // This will placed in parameter 1 for the lexicon look-up
    String deviceHintString = shortOffendingDeviceClassName + ".TYPE";
    
    // Note that the device type spec will be placed in parameter 2 for the lexicon look-up
    return LEX.getText(DEVICE_TYPE_CANNOT_BE_ABSTRACT,
        new Object[]{ folderHintString, deviceHintString, deviceType.getTypeSpec() } );
  }
}
