/*
 * @copyright 2005 Tridium Inc.
 */
package com.tridium.ddf.ui;

import javax.baja.driver.BIDeviceFolder;
import javax.baja.driver.point.BIPointFolder;
import javax.baja.sys.BComplex;
import javax.baja.sys.BComponent;
import javax.baja.sys.BajaRuntimeException;
import javax.baja.sys.Type;
import javax.baja.util.BFolder;
import javax.baja.workbench.mgr.BAbstractManager;

import com.tridium.ddf.BDdfDevice;
import com.tridium.ddf.BDdfNetwork;
import com.tridium.ddf.BDdfPointDeviceExt;
import com.tridium.ddf.IDdfFacetConst;
import com.tridium.ddf.discover.BDdfDiscoveryPreferences;
import com.tridium.ddf.discover.BIDdfDiscoveryHost;
import com.tridium.ddf.discover.BIDdfDiscoveryLeaf;
import com.tridium.ddf.discover.BIDdfDiscoveryObject;
import com.tridium.ddf.discover.auto.BDdfAutoDiscoveryPreferences;
import com.tridium.ddf.ui.device.BDdfDeviceManager;
import com.tridium.ddf.ui.point.BDdfPointManager;

/**
 * This class holds the common functionality that exists between the
 * point manager ui classes and the device manager ui classes.
 * 
 * @author lperkins
 *
 */
public class DdfMgrUtil
  implements IDdfFacetConst
{
  /**
   * Cannot instantiate! Access static methods only.
   */
  protected DdfMgrUtil(){}

  
/////////////////////////////////////////////////////////////////////////
// Util
/////////////////////////////////////////////////////////////////////////
    
  /**
   * Gets the discovery leaf type for a BIDdfDiscoveryManager. This is
   * determined by finding the BIDdfDiscoveryHost (usually the driver's
   * point device extension or the driver's network) for the givne mgr,
   * calling getDiscoveryPreferences() on the discovery host, and then
   * calling getDiscoveryLeafType() on the discovery host's discovery
   * preferences structure.
   */
  public static Type getDiscoveryLeafType(BAbstractManager mgr)
  {
    BIDdfDiscoveryHost ddfDiscoveryHost = findDiscoveryHost(mgr);
    
    if (ddfDiscoveryHost!=null) // If a discovery host parent was found
    { 
      // Then we get the discovery preferences
      BDdfDiscoveryPreferences ddfDiscoveryPrefs = ddfDiscoveryHost.getDiscoveryPreferences();
      
      if (ddfDiscoveryPrefs!=null && // The discovery preferences cannot be null
          ddfDiscoveryPrefs instanceof BDdfAutoDiscoveryPreferences && // They must be a BDdfAutoDiscoveryPreferences
          ddfDiscoveryPrefs.getType()!=BDdfAutoDiscoveryPreferences.TYPE) // And they must actually extend BDdfAutoDiscoveryPreferences (not be a direct instance of BDdfAutoDiscoveryPreferences)
        return ((BDdfAutoDiscoveryPreferences)ddfDiscoveryPrefs).getDiscoveryLeafType();
    }
    return null;
  }
  /**
   * Gets a prototype instance of the discovery leaf type for a BAbstractManager.
   */
  public static BIDdfDiscoveryLeaf getDiscoveryLeafInstance(BAbstractManager mgr)
  { 
    // Gets the discovery leaf type, as defined by the mgr's discovery host's discovery preferences
    Type discoveryLeafType = getDiscoveryLeafType(mgr);
    
    if (discoveryLeafType!=null)
    { 
      // Verifies that the discoveryLeafType implements BIDdfDiscoveryLeaf
      if (discoveryLeafType.is(BIDdfDiscoveryLeaf.TYPE))
        return (BIDdfDiscoveryLeaf)discoveryLeafType.getInstance(); // Returns a default instance of the discovery leaf type
      else
        return null;
    }
    else
      return null;
  }
  /**
   * Returns the driver's device type if mgr is a BDdfDeviceManager.
   * 
   * Returns the driver's proxy ext type if the mgr is a BDdfPointManager.
   * 
   * Returns null otherwise (this shouldn't happen unless somebody extends ddf driver)
   */
  public static Type getDatabaseType(BAbstractManager mgr)
  {
    if (mgr instanceof BDdfPointManager)
    {
      Type proxyExtType = ((BIPointFolder)mgr.getCurrentValue()).getProxyExtType();
      
      if (proxyExtType == null)
      {
        throw new BajaRuntimeException(
          DdfUiLexicon.LEX.getText("MustSpecifyProxyExtType",
                                   new Object[]{
                                     mgr.
                                       getCurrentValue().
                                         getType().
                                           getTypeSpec()}));
      }
      return proxyExtType;
    }
    else if (mgr instanceof BDdfDeviceManager)
    {
      Type deviceType = ((BIDeviceFolder)mgr.getCurrentValue()).getDeviceType();
      
      if (deviceType == null)
      {
        throw new BajaRuntimeException(
                                       DdfUiLexicon.LEX.getText("MustSpecifyDeviceType",
                                                                new Object[]{
                                                                  mgr.
                                                                    getCurrentValue().
                                                                      getType().
                                                                        getTypeSpec()}));
        
      }
      return deviceType;
    }
    else
      throw new BajaRuntimeException("Not Implemented");
  }
  /**
   * Returns a default instance of the given manager's database type.
   */
  public static BComponent getDatabaseInstanceDefault(BAbstractManager mgr)
  {
    return (BComponent)getDatabaseType(mgr).getInstance();
  }

  /**
   * Gets the discovery host for a BAbstractManager.
   */
  public static BIDdfDiscoveryHost findDiscoveryHost(BAbstractManager mgr)
  {
    BIDdfDiscoveryHost ddfDiscoveryHost = findDiscoveryHost((BComplex)mgr.getCurrentValue());
    
    return ddfDiscoveryHost;
  }

  /**
   * Finds the most immediate ancestor in the Niagara AX navigation tree above
   * the given obj that is an instance of BIDdfDiscoveryHost. This should
   * typically be the network in the case of a device manager or the point
   * device extension in the case of a point manager.
   */
  protected static BIDdfDiscoveryHost findDiscoveryHost(BComplex obj)
  {
    if (obj==null)
      return null;
    else if (obj instanceof BIDdfDiscoveryHost)
      return (BIDdfDiscoveryHost)obj;
    else
      return findDiscoveryHost(obj.getParent());
  }
  
  /**
   * Finds the network that is above or that is the component on
   * which the abstract manager is an agent.
   */
  public static BDdfNetwork findDdfNetwork(BAbstractManager mgr)
  {
    return findDdfNetwork((BComplex)mgr.getCurrentValue());
  }
  
  /**
   * Finds the ddf device that is above or that is the component on
   * which the abstract manager is an agent. Obviously the given mgr must
   * be the point manager (this could not possibly work for the device mgr)
   */
  public static BDdfDevice findDdfDevice(BAbstractManager mgr)
  {
    return findDdfDevice((BComplex)mgr.getCurrentValue());
  }
  
  /**
   * Finds the ddf-point-device-ext that is above or that is the component on
   * which the abstract manager is an agent. Obviously the given mgr must
   * be the point manager (this could not possibly work for the device mgr)
   */
  public static BDdfPointDeviceExt findDdfPointDeviceExt(BAbstractManager mgr)
  {
    return findDdfPointDeviceExt((BComplex)mgr.getCurrentValue());
  }

  /**
   * This method is called by the findDdfNetwork method that takes
   * an abstract manager.
   * @param obj
   * @return
   */
  protected static BDdfNetwork findDdfNetwork(BComplex obj)
  {
    if (obj==null)
      return null;
    else if (obj instanceof BDdfNetwork)
      return (BDdfNetwork)obj;
    else
      return findDdfNetwork(obj.getParent());
  }

  /**
   * This method is called by the findDdfDevice method that takes
   * an abstract manager.
   * @param obj
   * @return the ddf device that is above or that is the component on
   * which the abstract manager is an agent.
   */
  protected static BDdfDevice findDdfDevice(BComplex obj)
  {
    if (obj==null)
      return null;
    else if (obj instanceof BDdfDevice)
      return (BDdfDevice)obj;
    else
      return findDdfDevice(obj.getParent());
  }

  /**
   * This method is called by the findDdfPointDeviceExt method that takes
   * an abstract manager.
   * @param obj
   * @return the ddf point-device-ext that is above or that is the component on
   * which the abstract manager is an agent.
   */
  protected static BDdfPointDeviceExt findDdfPointDeviceExt(BComplex obj)
  {
    if (obj==null)
      return null;
    else if (obj instanceof BDdfPointDeviceExt)
      return (BDdfPointDeviceExt)obj;
    else
      return findDdfPointDeviceExt(obj.getParent());
  }

  /**
   * This is called by the ddf device manager and ddf point manager's
   * doLoadValue method, when the manager is first loaded. This sets
   * the initial set of discovery object from the discovery host's
   * discovery folder, if any are cached or defined in there.
   */
  public static void doLoadManager(BAbstractManager mgr)
  { // Gets the discovery host
    BIDdfDiscoveryHost discoveryHost =
      DdfMgrUtil.findDiscoveryHost(mgr);
    
    // Sanity check.
    if (discoveryHost!=null)
    { 
      // Gets the discovery folder
      BFolder discoveryFolder = discoveryHost.getDiscoveryFolder();
      
      // If the discovery folder is specified
      if (discoveryFolder!=null)
      { 
        // Then let's update the MgrLearn's roots with the discovery objects
        Object[] discoveryObjects = discoveryFolder.getChildren(BIDdfDiscoveryObject.class);
        
        if (discoveryObjects!=null)
        { 
          // Adds the discovery objects to the manager
          mgr.getLearn().updateRoots(discoveryObjects);
          
          // Reveals the "discovered" pane.
          mgr.getController().learnMode.setSelected(true);
        }
      }
    }
  }
}
