/*
 * @copyright 2005 Tridium Inc.
 */
package com.tridium.ddf.ui;

import java.util.Hashtable;

import javax.baja.control.BBooleanPoint;
import javax.baja.control.BBooleanWritable;
import javax.baja.control.BControlPoint;
import javax.baja.control.BEnumPoint;
import javax.baja.control.BEnumWritable;
import javax.baja.control.BNumericPoint;
import javax.baja.control.BNumericWritable;
import javax.baja.control.BStringPoint;
import javax.baja.control.BStringWritable;
import javax.baja.driver.point.BProxyExt;
import javax.baja.gx.BImage;
import javax.baja.job.BJob;
import javax.baja.registry.TypeInfo;
import javax.baja.sys.BComponent;
import javax.baja.sys.BComponentEvent;
import javax.baja.sys.BString;
import javax.baja.sys.BValue;
import javax.baja.sys.Property;
import javax.baja.sys.Subscriber;
import javax.baja.sys.Type;
import javax.baja.ui.BLabel;
import javax.baja.ui.BWidget;
import javax.baja.ui.pane.BGridPane;
import javax.baja.nre.util.Array;
import javax.baja.util.Lexicon;
import javax.baja.nre.util.TextUtil;
import javax.baja.workbench.mgr.BAbstractManager;
import javax.baja.workbench.mgr.MgrColumn;
import javax.baja.workbench.mgr.MgrEditRow;
import javax.baja.workbench.mgr.MgrLearn;
import javax.baja.workbench.mgr.MgrTypeInfo;

import com.tridium.ddf.BDdfDevice;
import com.tridium.ddf.IDdfFacetConst;
import com.tridium.ddf.discover.BDdfDiscoveryJob;
import com.tridium.ddf.discover.BDdfPointDiscoveryLeaf;
import com.tridium.ddf.discover.BIDdfDeviceDiscoveryLeaf;
import com.tridium.ddf.discover.BIDdfDiscoveryGroup;
import com.tridium.ddf.discover.BIDdfDiscoveryHost;
import com.tridium.ddf.ui.discover.BIDdfDiscoveryIcon;
import com.tridium.ddf.discover.BIDdfDiscoveryLeaf;
import com.tridium.ddf.discover.BIDdfDiscoveryObject;
import com.tridium.ddf.discover.BIDdfPointDiscoveryLeaf;
import com.tridium.ddf.discover.auto.BDdfAutoDiscoveryJob;
import com.tridium.ddf.identify.BDdfDeviceId;
import com.tridium.ddf.identify.BDdfIdParams;
import com.tridium.ddf.point.BDdfProxyExt;
import com.tridium.ddf.ui.DdfMgrColumnUtil.DdfMgrProp;
import com.tridium.ddf.ui.DdfMgrColumnUtil.DdfMgrPropPath;
import com.tridium.ddf.ui.device.BDdfDeviceManager;
import com.tridium.ddf.ui.point.BDdfPointManager;
import com.tridium.workbench.job.BJobBar;

/**
 * This customized version of MgrLearn is used on the point manager and the device manager.
 *
 * This is a fully-functional MgrLearn. The discovery columns are based on introspection. The
 * 'toRow' method is based on introspection.
 *
 * @author lperkins
 */
public class DdfMgrLearn
  extends MgrLearn
  implements IDdfFacetConst
{
	public DdfMgrLearn(BAbstractManager manager)
	{
	  super(manager);
	}

////////////////////////////////////////////////////////////////
// JobBar - Various Hacks to Allow for a Customized Job Bar Name
////////////////////////////////////////////////////////////////

	private String customizeJobBarNameBasedOnDriverName(BIDdfDiscoveryHost discoveryHost)
	{
    // Gets the name of the driver
    String driverName = discoveryHost.getType().getModule().getModuleName();

    if (driverName!= null && driverName.length()>0)
    {
      driverName = TextUtil.toFriendly(driverName);

      return driverName + ' ' +  DdfUiLexicon.LEX.getText("Discovery");
    }
    //else
    //{
    //  // Unable to determine the name of the driver -- impossible!
    //  // jobBarName stays null so the 'updateJobBarName' method
    //  // basically does nothing
    //}
    return null;
	}

	private String getJobBarNameFromDriverLexicon(BIDdfDiscoveryHost discoveryHost)
	{
    // Gets the Lexicon for the particular driver that defines the
    // discover host class
    Lexicon driverLexicon =
      discoveryHost.getType().getModule().getLexicon();

    // Determines a Lexicon key to look for in the particular driver to
    // see if the driver wants to customize the name on the job bar
    String lexiconKeyForJobBarName =
      discoveryHost.getType().getTypeName() + ".discovery.jobBar.name";

    // Determines if there is a name for the job bar defined in the
    // driver's lexicon
    return driverLexicon.get(lexiconKeyForJobBarName);

	}

	/**
	 * This method has to be private since it is being added after
	 * the release of the devDriver API.
	 */
  private void customizeJobBarName()
  {
    BWidget jobBar = getJobBar();

    if (jobBar != null)
    {
      // Gets the discovery host, which I expect to be either the network
      // or the point device extension
      BIDdfDiscoveryHost discoveryHost =
        DdfMgrUtil.findDiscoveryHost( getManager() );

      // Gets the job bar name from the discoveryHost's driver's lexicon
      String jobBarName = getJobBarNameFromDriverLexicon( discoveryHost );

      // If the driver itself does not customize the job name, then this
      // customizes it based on the driver name
      if (jobBarName == null)
      {

        // Customizes the job bar name based on the discoveryHost's driver's
        // name
        jobBarName = customizeJobBarNameBasedOnDriverName(discoveryHost);
      }
      // else
      // {
      //   // Else, the driver's Lexicon file customizes the job bar name.
      //   // This falls through and allows the 'updateJobBarName' method
      //   // to use the driver's version of the jobBarName
      // }

      // Customizes the name on the job bar
      updateJobBarName(jobBarName);
    }
  }

  /**
   * This method has to be limited access since it is being added after
   * the release of the devDriver API.
   *
   * @param jobBarName the customized name for the job bar or null if
   * no customization is desired.
   */
  void updateJobBarName(String jobBarName)
  {
    if (jobBarName!=null)
    {
      BLabel jobBarNameLabel = getJobBarNameLabel();

      if (jobBarNameLabel !=null)
      {
        if (!(jobBarName.equals(jobBarNameLabel.getText())))
        {
          jobBarNameLabel.setText(jobBarName);
        }

        if (jobBarSubscriber==null)
        {
          jobBarSubscriber = new JobBarSubscriber(jobBarName);
          jobBarSubscriber.subscribe(jobBarNameLabel);
        }
      }
    }
  }


////////////////////////////////////////////////////////////////
// MgrLearn
////////////////////////////////////////////////////////////////

  public void setJob(BJob job)
  {
    super.setJob(job);

    if (job instanceof BDdfAutoDiscoveryJob)
    {
      customizeJobBarName();
    }
  }


  /**
   * Gets the BLabel on the BJobBar that contains the 'name'.
   *
   * NOTE that this is private on the BJobBar so this method
   * has to get it generically by walking through the BJobBar's
   * slotmap.
   *
   * @return the BLabel that contains the BJobBar's 'name' or
   * null if not found.
   */
  BLabel getJobBarNameLabel()
  {
    if (getJobBar()!=null && getJobBar() instanceof BJobBar)
    {
      BWidget jobBarCenter = ((BJobBar)getJobBar()).getCenter();

      // NOTE: The BJobBar source code indicates that the
      //       center is a BGridPane
      if (jobBarCenter instanceof BGridPane)
      {
        BGridPane jobBarGridPane = (BGridPane)jobBarCenter;

        // NOTE:  The BJobBar source code adds the 'name'
        // field to the BJobBar second. This gets the second
        // dynamic widget property value and hopes its the
        // name BLabel
        Property[] gridWidgetProps =
          jobBarGridPane.getDynamicPropertiesArray();

        // Sanity check....verifies that there are at least two
        // widgets on the job bar's grid pane
        if (gridWidgetProps.length>2)
        {
          // Gets the second widget on the job bar's grid pane. This
          // should be the name label
          BValue gridValue2 = jobBarGridPane.get( gridWidgetProps[1] );

          if (gridValue2 instanceof BLabel)
          {
            return (BLabel) gridValue2;
          }
        }
      }
    }
    return null;
  }

  class JobBarSubscriber
    extends Subscriber
  {
    String jobBarName;

    JobBarSubscriber(String jobBarName)
    {
      this.jobBarName = jobBarName;
    }

    public void event(BComponentEvent event)
    {
      updateJobBarName(jobBarName);
    }
  }


  /**
   * Automatically determines the MgrColumns for the discovery list.
   */
  protected MgrColumn[] makeColumns()
  {
    return DdfMgrLearnUtil.makeColumns(getManager());
  }

  /**
   * Automatically synchronizes the data in a discovery object with an
   * instance of MgrEditRow representing a database object to be modified.
   */
  public void toRow(Object discovery, MgrEditRow row)
  {
    if (discovery instanceof BIDdfDiscoveryLeaf)
    {
      if (getManager() instanceof BDdfDeviceManager)
        deviceManagerToRow(discovery,row); // Uses intelligence to match the discovery and database columns that pertain to the deviceId

      else if (getManager() instanceof BDdfPointManager)
        pointManagerToRow(discovery,row);// Uses intelligence to match the discovery and database columns that pertain to the readParameters, writeParameters, or pointId
    }
  }

  /**
   * This callback is automatically invoked when the
   * current job set via <code>setJob()</code> completes.
   */
  public void jobComplete(BJob job)
  {
    if (job instanceof BDdfDiscoveryJob)
    {
      updateRoots(
          ((BDdfDiscoveryJob)job).getRootDiscoveryObjects());
    }
  }

  /**
   * All depths are potentially expandable, since we do not really
   * know how the driver developer is going to use this.
   */
  public boolean isDepthExpandable(int depth)
  {
    return true;
  }

  /**
   * A discovery object hasChildren is the discovery object
   * implements BIDdfDiscoveryGroup.
   */
  public boolean hasChildren(Object discovery)
  {
    return (discovery instanceof BIDdfDiscoveryGroup);
  }

  public boolean isExisting(Object discovery, BComponent component)
  {
    if (discovery==null || component==null) // Sanity check!
      return false;

    if (discovery instanceof BIDdfDiscoveryLeaf)
    {
      if (getManager() instanceof BDdfDeviceManager && component instanceof BDdfDevice )
      {
        // Uses intelligence to determine if the given discovery leaf matches the given database component
        return isExistingDevice((BIDdfDiscoveryLeaf)discovery, (BDdfDevice)component);
      }
      else if (getManager() instanceof BDdfPointManager && component instanceof BControlPoint)
      {
        // Uses intelligence to determine if the given discovery leaf matches the given database component
        return isExistingPoint((BIDdfDiscoveryLeaf)discovery, (BControlPoint)component);
      }
      else
      {
        // Else, the manager is neither a BDdfDeviceManager nor a BDdfPointManager...this is chaotic! Somebody must be using this on his/her own view!!!
        return super.isExisting(discovery, component);
      }
    }
    else
      return super.isExisting(discovery, component);
  }

  /**
   * Casts the given discovery object to a BIDdfDiscoveryGroup,
   * gets all BIDdfDiscoveryObject's under the given discovery
   * object, registers for component events on any that are BComponents,
   * and returns the Object[] array of the child BIDdfDiscoveryObjects.
   */
  public Object[] getChildren(Object discovery)
  {
    // Verifies that the given discovery object is BIDdfDiscoveryGroup
    if (discovery instanceof BIDdfDiscoveryGroup)
    {
      // Casts the given discovery object to a BIDdfDiscoveryGroup
      BIDdfDiscoveryGroup discoveryGroup = (BIDdfDiscoveryGroup)discovery;

      // Gets the discoveryGroup's Niagara AX children that implement BIDdfDiscoveryObject
      Object[] discoveryChildren = discoveryGroup.getChildren(BIDdfDiscoveryObject.class);

      // Loops through all of the discoveryGroup's Niagara AX children that implement BIDdfDiscoveryObject
      for (int i=0; i<discoveryChildren.length; i++)
        if (discoveryChildren[i] instanceof BComponent) // If any of the discovery children are subclasses of BComponent
          getManager().registerForComponentEvents((BComponent)discoveryChildren[i]);  // Then we register for component events so that we can receive synchronization and updates from their master copy that is in the station

      return discoveryChildren;
    }
    else
      return null;
  }

  /**
   * Overrides MgrLearn.getIcon(Object) to define an icon. If the given discovery
   * object implements BIDdfDiscoveryIcon, then we call getDiscoveryIcon on the
   * discovery object and return the resulting BImage. Otherwise, we compute a
   * reasonable default icon for the discovery object.
   */
  public BImage getIcon(Object discovery)
  {
    // If the discovery object customizes the icon
    // Then we will use the discovery icon.
    if (discovery instanceof BIDdfDiscoveryIcon)
      return ((BIDdfDiscoveryIcon)discovery).getDiscoveryIcon();
    else
      return getIconDefault(discovery);
  }

  /**
   * Determines the Niagara AX database types that are valid for
   * the given discovery object.
   */
  public MgrTypeInfo[] toTypes(Object discovery)
  {
    return DdfMgrLearnUtil.toTypes(discovery);
  }

  /**
   * A discovery object is a group if it implements BIDdfDiscoveryGroup.
   */
  public boolean isGroup(Object discovery)
  {
    return (discovery instanceof BIDdfDiscoveryGroup);
  }

////////////////////////////////////////////////////////////////
// DdfMgrLearn - support for the 'toRow' method (point mgr mode)
////////////////////////////////////////////////////////////////

  /**
   * Updates the standard properties that are on the control point. This is used in the toRow logic.
   *
   * @param editDlgColPropPath
   * @param discoveryPointLeaf
   *
   *
   * @param row
   */
  protected void updateControlPointProps(DdfMgrColumnUtil.DdfMgrProp editDlgColProp, BIDdfPointDiscoveryLeaf discoveryPointLeaf, MgrEditRow row)
  {
    // If the given DdfMgrProp is associated with the control point's facets
    // And if the given ddf point discovery leaf defines facets
    if (editDlgColProp.getProp().equals(BControlPoint.facets) &&
        discoveryPointLeaf.getDiscoveryPointFacets()!=null) // Possibly initializes the point facets
      row.setCell(editDlgColProp,discoveryPointLeaf.getDiscoveryPointFacets());

  }

  /**
   * Updates the properties that are on the proxy extension. This is used in the toRow logic.
   *
   * @param editDlgColPropPath
   * @param discoveryPointLeaf
   *
   *
   * @param row
   */
  protected void updateControlPointProps(DdfMgrColumnUtil.DdfMgrPropPath editDlgColPropPath, BIDdfPointDiscoveryLeaf discoveryPointLeaf, MgrEditRow row)
  {
    // Gets the prop path (under the control point) that the given DdfMgrPropPath is associated with
    Property[] colPropPath = editDlgColPropPath.getPropPath();

    // If the length is exactly two and the first part of the prop path is associated with the control
    // Point's proxyExt, then we look further
    if (colPropPath.length==2 && colPropPath[0].equals(BControlPoint.proxyExt))
    {
      // If the given DdfMgrPropPath is associated with the control point's proxy ext's 'deviceFacets'
      // And if the given ddf point discovery leaf defines device facets.
      // Then we transfer the discovery's device facets into that particular item's column on the MgrEdit dialog.
      if (colPropPath[1].equals(BProxyExt.deviceFacets) &&
          discoveryPointLeaf.getDiscoveryDeviceFacets()!=null)  // Possibly initializes the device facets
        row.setCell(editDlgColPropPath,discoveryPointLeaf.getDiscoveryDeviceFacets());

      // If the given DdfMgrPropPath is associated with the control point's proxy ext's 'conversion'
      // And if the given ddf point discovery leaf defines the conversion.
      // Then we transfer the discovery's conversion into that particular item's column on the MgrEdit dialog.
      if (colPropPath[1].equals(BProxyExt.conversion) &&
          discoveryPointLeaf.getDiscoveryConversion()!=null)
        row.setCell(editDlgColPropPath,discoveryPointLeaf.getDiscoveryConversion());

      // If the given DdfMgrPropPath is associated with the control point's proxy ext's 'tuningPolicyName'
      // And if the given ddf point discovery leaf defines the tuning policy name.
      // Then we transfer the discovery's tuning policy into that particular item's column on the MgrEdit dialog.
      if (colPropPath[1].equals(BProxyExt.tuningPolicyName) &&
          discoveryPointLeaf.getDiscoveryTuningPolicyName()!=null)
        row.setCell(editDlgColPropPath,BString.make(discoveryPointLeaf.getDiscoveryTuningPolicyName()));
    }
  }

  /**
   * This is part of the point manager's toRow logic.
   */
  protected void updateControlPointProps(MgrColumn editDlgCol, BIDdfPointDiscoveryLeaf discoveryPointLeaf, MgrEditRow row)
  {
    if (editDlgCol instanceof DdfMgrColumnUtil.DdfMgrProp)
    {
      // Updates the properties that are directly on the control point, currently this includes only the 'facets'
      updateControlPointProps((DdfMgrColumnUtil.DdfMgrProp)editDlgCol,discoveryPointLeaf,row);
    }
    else if (editDlgCol instanceof DdfMgrColumnUtil.DdfMgrPropPath)
    {
      // Updates the properties that are on the proxy extension, currently this includes the 'device facets', 'conversion', and 'tuningPolicyName'
      updateControlPointProps((DdfMgrColumnUtil.DdfMgrPropPath)editDlgCol,discoveryPointLeaf, row);
    }
  }

  /**
   * This possibly updates the facets, device facets, conversion, and tuning policy name for
   * the MgrEditRow, if the MgrEditRow has DdfMgrColumnUtil.DdfMgrProp or DdfMgrPropPaths
   * that point to those properties.
   *
   * @param discoveryPointLeaf
   *
   * @param row
   */
  protected void updateControlPointProps(BIDdfPointDiscoveryLeaf discoveryPointLeaf, MgrEditRow row)
  {
    MgrColumn[] editDlgCols = row.getColumns();

    for (int i=0; i<editDlgCols.length; i++)
    {
      // Updates any DdfMgrColumn's that are configured on the control point itself or the
      // Standard proxy extension properties
      updateControlPointProps(editDlgCols[i], discoveryPointLeaf, row);
    }
  }
  /**
   * This is part of the 'toRow' logic for the point manager. This updates
   * any properties that are tied to columns for BSimples that are under the
   * driver's readParameters, writeParameters, or pointId
   */
  protected void pointManagerToRow(Object discovery, MgrEditRow row)
  {
    // Casts the discovery object to a BIDdfDiscoveryObject
    BIDdfDiscoveryObject ddfDiscoveryObject = (BIDdfDiscoveryObject)discovery;

    // Gets the Niagara AX Type for the proxy extension's read parameters structure, write parameters structure, and point id
    Type readParametersType = getProxyExtReadParametersType(row);

    Type writeParametersType = getProxyExtWriteParametersType(row);

    Type pointIdType = getProxyExtPointIdType(row);

    Type discoveryObjectType = ddfDiscoveryObject.getType();

    // Checks if the discovery object's type is the read parameters structure Type
    // Ideally, the discovery object will be a BIDdfPointDiscoveryLeaf.
    // However, to make it easy in some limited circumstances, we allow for the
    // Discovery object to be a read parameters, write parameters, or point id
    if (readParametersType == discoveryObjectType)
      updateDatabaseIdColumns(BDdfProxyExt.readParameters,discovery,row);

    // Else, checks if the discovery object's type is the same as the write parameters structure Type
    else if (writeParametersType == discoveryObjectType)
      updateDatabaseIdColumns(BDdfProxyExt.writeParameters,discovery,row);

    // Else, checks if the discovery object's type is the same Niagara Ax type as the proxy ext's point id
    else if (pointIdType == discoveryObjectType)
      updateDatabaseIdColumns(BDdfProxyExt.pointId,discovery,row);

    // This is the ideal case because the BIDdfPointDiscoveryLeaf specifies
    // All three -- the readParameters, writeParameters, and pointId
    else if (discoveryObjectType.is(BIDdfPointDiscoveryLeaf.TYPE))
    {
      BIDdfPointDiscoveryLeaf pointLeaf = (BIDdfPointDiscoveryLeaf)discovery;

      // Updates any DdfMgrPropPath columns that are for props that are under the driver's readParameters structure
      updateDatabaseIdColumns(BDdfProxyExt.readParameters, pointLeaf, row);

      // Updates any DdfMgrPropPath columns that are for props that are under the driver's writeParameters structure
      updateDatabaseIdColumns(BDdfProxyExt.writeParameters, pointLeaf, row);

      // Updates any DdfMgrPropPath columns that are for props that are under the driver's pointId structure
      updateDatabaseIdColumns(BDdfProxyExt.pointId, pointLeaf, row);

      // Updates the standard properties that are directly under the control point (facets)
      // And the standard properties that are directly under the proxy (device facets, conversion, tuning policy name)
      updateControlPointProps(pointLeaf, row);
    }
    // else, the discovery object type is not from the ddf!
    // so there is nothing that we can do to help make things easier in this case
    // TODO: we might want to print a warning message to the workbench standard output window
  }

  /**
   * Looks at the row's proxy extension and determines
   * the Type of its customized writeParameters property.
   */
  protected Type getProxyExtWriteParametersType(MgrEditRow row)
  {
    // Gets the row's corresponding database component (or soon-to-be database component)
    BComponent targetComponent = row.getTarget();

    // This method should only be called when processing for the point manager so the
    // target will be a control point, unless the driver developer makes his or her own
    // point manager with an MgrLearn that extends from DdfMgrLearn
    if (targetComponent instanceof BControlPoint)
    {
      BControlPoint targetCp = (BControlPoint)targetComponent;

      // Verifies that the target point's proxy extends BDdfProxyExt. It should unless the
      // Driver developer is doing something fishy or highly customized.
      if (targetCp.getProxyExt() instanceof BDdfProxyExt)
      {
        BDdfProxyExt targetProxyExt = (BDdfProxyExt)targetCp.getProxyExt();
        return  targetProxyExt.getWriteParameters().getType();
      }
      else // If the control point's proxy is not a BDdfProxyExt then we cannot help any further
        return null;
    }
    else // Else, if the point manager row's target is not a BControlPoint then we cannot help any futher
      return null;
  }

  /**
   * Looks at the row's proxy extension and determines
   * the Type of its customized readParameters property.
   */
  protected Type getProxyExtReadParametersType(MgrEditRow row)
  {
    // NOTE: See the getProxyExtWriteParametersType method for
    // lot's of comments in code. It's comments are applicable
    // here, if you need to understand the code for this method.
    BComponent targetComponent = row.getTarget();

    if (targetComponent instanceof BControlPoint)
    {
      BControlPoint targetCp = (BControlPoint)targetComponent;
      if (targetCp.getProxyExt() instanceof BDdfProxyExt)
      {
        BDdfProxyExt targetProxyExt = (BDdfProxyExt)targetCp.getProxyExt();
        return  targetProxyExt.getReadParameters().getType();
      }
      else
        return null;
    }
    else
      return null;
  }

  /**
   * Looks at the row's proxy extension and determines
   * the Type of its customized pointId property.
   */
  protected Type getProxyExtPointIdType(MgrEditRow row)
  {
    // NOTE: See the getProxyExtWriteParametersType method for
    // lot's of comments in code. It's comments are applicable
    // here, if you need to understand the code for this method.
    BComponent targetComponent = row.getTarget();

    if (targetComponent instanceof BControlPoint)
    {
      BControlPoint targetCp = (BControlPoint)targetComponent;

      if (targetCp.getProxyExt() instanceof BDdfProxyExt)
      {
        BDdfProxyExt targetProxyExt = (BDdfProxyExt)targetCp.getProxyExt();

        return  targetProxyExt.getPointId().getType();
      }
      else
        return null;
    }
    else
      return null;
  }

/////////////////////////////////////////////////////////////////////////////
// DdfMgrLearn - support for the 'toRow' method (point mgr & Device Mgr mode)
/////////////////////////////////////////////////////////////////////////////

  /**
   * Updates the database columns of an MgrEditRow that are for a particular BDdfIdParams property on the
   * row's target. Updates these columns with the values in the given discovery object.
   *
   * @param idProperty - the driver's proxy's readParameters, writeParameters, or pointId property; or the driver's deviceId or pingParameters properties
   *
   * @param discoverValue - The BIDdfDiscoveryObject for the given row
   *
   * @param row - MgrEditRow representing a database object that is in the station or that may be added to the station.
   */
  protected void updateDatabaseIdColumns(Property idProperty, Object discoverValue, MgrEditRow row)
  {
    // Gets all of the discovery columns
    MgrColumn[] disCols = getColumns(); // We are on the MgrLearn so getColumns yields the discovery columns

    // Prepares a very important table that will soon help us introspect and match up
    // Discovery columns to database columns
    Hashtable<String, MgrColumn> disColsPropPathStrings = new Hashtable<>(disCols.length);

    // Hashes each discovery column that is a DdfMgrPropPath or DdfMgrProp
    // To a prop path string: "{id}."+propPath or prop where "{id}" is the name of the id property (readParameters, writeParameters, pointId, deviceId, or pingParameters)
    for (int i=0; i<disCols.length; i++)
    {
      if (disCols[i] instanceof DdfMgrPropPath)
        disColsPropPathStrings.put( // Hashes the DdfMgrPropPath to a string key that is a version of the prop path as a dot-notation (for example, readParameters.someDriverProperty)
            makePropPathHelperString((DdfMgrPropPath)disCols[i],""),
            disCols[i]);

      else if (disCols[i] instanceof DdfMgrProp)
        disColsPropPathStrings.put( // Hashes the DdfMgrProp to a string key that indicates the properties dot-notation (for example, readParameters.someDriverProperty)
            idProperty.getName()+"."+((DdfMgrProp)disCols[i]).getProp().getName(),
            disCols[i]);
    }

    // Loops through the database columns and gets all DdfMgrPropPaths for any columns that are for BSimples under the corresponding {id} on the database component
    DdfMgrPropPath[] databaseIdCols = getDatabaseColumnsForProp(idProperty.getName());

    for (int i=0; i<databaseIdCols.length; i++)
    {
      // Constructs a special prop path string for the id property's child property, the string starts with "{id}."+childPropName where "{id}" is the name of the id property (readParameters, writeParameters, pointId, deviceId, or pingParameters)
      String dbColPropPathString = makePropPathHelperString(databaseIdCols[i],idProperty.getName());

      // Looks up the corresponding discovery column from the important dbColPropPathString table that we previously built
      // We are looking for the DdfMgrProp or DdfMgrPropPath with the matching dot notation String key
      MgrColumn disCol = disColsPropPathStrings.get(dbColPropPathString);

      if (disCol!=null) // If a corresponding discovery column exists
      {                 // Then this transfers the discovery column value into the database column
        if (databaseIdCols[i].isEditable())
          row.setCell(databaseIdCols[i],(BValue)disCol.get(discoverValue));
      }
    }
  }

  /**
   * Converts the prop path in the given DdfMgrPropPath to a single string using
   * dot notation to separate the prop path.
   *
   * @param ddfMgrPropPath the DdfMgrPropPath to convert to string
   *
   * @param fromProp the property within the prop path to start collecting at, or empty string to
   * collect the entire prop path. If not null, then the return value starts at and includes the fromProp
   *
   *
   * @return
   */
  protected String makePropPathHelperString(DdfMgrPropPath ddfMgrPropPath, String fromProp)
  {
    String helperString = fromProp;

    Property[] propPath = ddfMgrPropPath.getPropPath();

    boolean afterProp = (fromProp.length()>0)?false:true;

    for (int i=0; i<propPath.length; i++)
    {
      if (afterProp)
      {
        if (helperString.length()>0)
          helperString += ".";

        helperString+=propPath[i].getName();
      }
      else
      {
        if (propPath[i].getName().equals(fromProp))
          afterProp=true;
      }
    }
    return helperString;
  }

  /**
   * Gets a list of DdfMgrPropPath columns from the database columns that are
   * likely to be child columns of the given property.
   */
  protected DdfMgrPropPath[] getDatabaseColumnsForProp(String propName)
  {
    Array<DdfMgrPropPath> databaseColumnsForProp = new Array<>(DdfMgrPropPath.class);

    MgrColumn[] databaseColumns = getManager().getModel().getColumns();

    for (int i=0; i<databaseColumns.length; i++)
    {
      if (databaseColumns[i] instanceof DdfMgrPropPath)
      {
        Property[] dbColPropPath = ((DdfMgrPropPath)databaseColumns[i]).getPropPath();
        for (int p=0; p<dbColPropPath.length; p++)
        {
          if (dbColPropPath[p].getName().equals(propName))
          {
            databaseColumnsForProp.add((DdfMgrPropPath)databaseColumns[i]);
          }
        }
      }
    }
    return databaseColumnsForProp.trim();
  }

/////////////////////////////////////////////////////////////////////////////
// DdfMgrLearn - support for the 'toRow' method (Device Mgr mode)
/////////////////////////////////////////////////////////////////////////////

  protected void deviceManagerToRow(Object discovery, MgrEditRow row)
  {
    BIDdfDiscoveryObject ddfDiscoveryObject = (BIDdfDiscoveryObject)discovery;

    // Gets the exact Type of the device's customized deviceId property
    Type deviceIdType = getDeviceIdType(row);

    // Gets the exact Type of the device's customized pingParameters property
    Type pingParametersType = getPingParametersType(row);

    // Processes further if the discovery object's type is the exact same as the
    // deviceId type.
    // Ideally, the discovery object will be a BIDdfDeviceDiscoveryLeaf.
    // However, to make it easy in some limited circumstances, we allow for the
    // Discovery object to be a deviceId or pingParameters structure

    // Checks if the device discovery object is an instance of the driver's deviceId class
    if (deviceIdType!=BDdfIdParams.TYPE && // If the developer has actually redefined the deviceId class
        ddfDiscoveryObject.getType().is(deviceIdType) ) // And if the developer's discovery object type is an instance of their deviceId class
      updateDatabaseIdColumns(BDdfDevice.deviceId, discovery, row);

    // Checks if the device discovery object is an instance of the driver's pingParameters class
    if (pingParametersType != BDdfIdParams.TYPE && // If the developer has acutally redefined the pingParams class
        ddfDiscoveryObject.getType().is(pingParametersType) ) // And if the developer's discovery object type is an instance of their pingParams class
      updateDatabaseIdColumns(BDdfDevice.pingParameters, discovery, row);

    // Checks if the device discovery object is an instance of BIDdfDeviceDiscoveryLeaf, which
    // Defines both the 'deviceId' and the 'pingParameters'
    if (ddfDiscoveryObject.getType().is(BIDdfDeviceDiscoveryLeaf.TYPE))
    {
      updateDatabaseIdColumns(BDdfDevice.deviceId, discovery, row);
      updateDatabaseIdColumns(BDdfDevice.pingParameters, discovery, row);
      updateDatabaseIdColumns(BDdfDevice.communicator, discovery, row);
    }

    if (discovery instanceof BIDdfDiscoveryLeaf) // Sets the default name for the new device. The core manager code will assign suffixes as necessary to make this unique.
    {
      BIDdfDiscoveryLeaf leaf = (BIDdfDiscoveryLeaf)discovery;
      if (leaf.getDiscoveryName()!=null)
          row.setDefaultName(leaf.getDiscoveryName());
    }
  }

  protected Type getDeviceIdType(MgrEditRow row)
  {
    BComponent targetComponent = row.getTarget();

    if (targetComponent instanceof BDdfDevice)
      return ((BDdfDevice)targetComponent).getDeviceId().getType();
    else
      return null;
  }

  protected Type getPingParametersType(MgrEditRow row)
  {
    BComponent targetComponent = row.getTarget();

    if (targetComponent instanceof BDdfDevice)
      return ((BDdfDevice)targetComponent).getPingParameters().getType();
    else
      return null;

  }

  /**
   * This is the default 'isExisting' test for the DdfMgrLearn when it is used for a device manager.
   *
   * @param discoveryLeaf the discovery object, cast as BIDdfDiscoveryLeaf
   *
   * @param databaseDevice the database component, cast as BDdfDevice
   *
   *
   * @return true if the discovery leaf is a BIDdfDeviceDiscoveryLeaf and its deviceId and pingParameters
   * structures are 'equivalent' to the corresponding ones on the databaseDevice OR if the discovery leaf
   * is not a BIDdfDeviceDiscoveryLeaf then this  return true if it is equivalent to the device's deviceId
   * structure. This method returns false otherwise.
   */
  public boolean isExistingDevice(BIDdfDiscoveryLeaf discoveryLeaf, BDdfDevice databaseDevice)
  {
    // Device discovery leaf - defines deviceId and pingParams for the database device
    if (discoveryLeaf instanceof BIDdfDeviceDiscoveryLeaf)
    {
      BIDdfDeviceDiscoveryLeaf deviceDiscoveryLeaf = (BIDdfDeviceDiscoveryLeaf)discoveryLeaf;

      return deviceDiscoveryLeaf.getDeviceId().isExisting(databaseDevice.getDeviceId(),  CONTEXT_DEVICE_MANAGER ) &&
             deviceDiscoveryLeaf.getPingParameters().isExisting( databaseDevice.getPingParameters(), CONTEXT_DEVICE_MANAGER );
    }

    // Device Id - defines only the deviceId for the database device
    else if (discoveryLeaf instanceof BDdfDeviceId)
    {
      return ((BDdfDeviceId)discoveryLeaf).isExisting(databaseDevice.getDeviceId(), CONTEXT_DEVICE_MANAGER);
    }
    else
      return false;
  }

  /**
   * This is the default 'isExisting' test for the DdfMgrLearn when it is used for a point manager.
   *
   * @param discoveryLeaf the discovery object, cast as BIDdfDiscoveryLeaf
   *
   * @param proxyPoint the database component, cast as BControlPoint
   *
   *
   * @return true if the discovery leaf is a BDdfPointDiscoveryLeaf and its pointId, readParameters,
   * and writeParameters structures are 'equivalent' to the corresponding ones on the proxy extension
   * that is under the given point OR if the discovery leaf is not a BDdfPointDiscoveryLeaf then this
   * return true if it is equivalent to the proxy extension's pointId, readParameters, or writeParameters
   * structure. This method returns false otherwise.
   */
  public boolean isExistingPoint(BIDdfDiscoveryLeaf discoveryLeaf, BControlPoint proxyPoint)
  {
    if (discoveryLeaf==null || proxyPoint == null) // Sanity check
      return false;

    else if (proxyPoint.getProxyExt() instanceof BDdfProxyExt) // Sanity check
    {
      BDdfProxyExt dbProxyExt = (BDdfProxyExt)proxyPoint.getProxyExt(); // Gets the BDdfProxyExt

      if (discoveryLeaf instanceof BDdfPointDiscoveryLeaf) // Ideal scenario
      {
        // Point exists if the proxy's pointId, readParameters, and writeParameters are existing matches to
        // the same from the discovery leaf
        BDdfPointDiscoveryLeaf pointDiscoveryLeaf = (BDdfPointDiscoveryLeaf)discoveryLeaf;

        return pointDiscoveryLeaf.getPointId().isExisting( dbProxyExt.getPointId(), CONTEXT_POINT_MANAGER ) &&
               pointDiscoveryLeaf.getReadParameters().isExisting( dbProxyExt.getReadParameters(), CONTEXT_POINT_MANAGER ) &&
               pointDiscoveryLeaf.getWriteParameters().isExisting( dbProxyExt.getWriteParameters(), CONTEXT_POINT_MANAGER );
      }
      else if (discoveryLeaf instanceof BDdfIdParams)
      {
        BDdfIdParams discoveryLeafId = (BDdfIdParams)discoveryLeaf;

        // This is 'pushing it' but we'll assume that if the discovery leaf is an existing match to the proxy's
        // point id then the discovery leaf matches the database point
        if (discoveryLeafId.isExisting(dbProxyExt.getPointId(), CONTEXT_POINT_MANAGER))
        {
          return true;
        }

        // This is 'pushing it' but we'll assume that if the discovery leaf is an existing match to the proxy's
        // read parameters then the discovery leaf matches the database point
        else if (discoveryLeafId.isExisting(dbProxyExt.getReadParameters(), CONTEXT_POINT_MANAGER))
        {
          return true;
        }

        // This is 'pushing it' but we'll assume that if the discovery leaf is an existing match to the proxy's
        // write parameters then the discovery leaf matches the database point
        else if (discoveryLeafId.isExisting(dbProxyExt.getWriteParameters(), CONTEXT_POINT_MANAGER))
        {
          return true;
        }

        // Else, the discovery leaf is not a BDdfPointDiscoveryLeaf, it is neither an existing match to the database
        // component's pointId, readParameters, nor writeParameters, so there is nothing else that we can do but
        // conclude that the discovery leaf is not represented in the database
        else
          return false;
      }

      // Else, the discovery leaf is not even a BDdfIdParams so there is nothing else that we can do but
      // conclude that the discovery leaf is not represented in the database
      else
        return false;
    }
    else // Else, the control point's proxy does not extend from devDriver so there is nothing automatic that we can do.
      return false;
  }



  /**
   * Determines an icon to use for the given discovery object. This method is called for point
   * managers if the discovery object itself does not specify an icon.
   */
  protected BImage getPointIconDefault(Object discovery)
  {
    if (discovery instanceof BIDdfDiscoveryLeaf)       // Chooses an icon based on the default database type
      return getPointIconDefault((BIDdfDiscoveryLeaf)discovery);
    else if (discovery instanceof BIDdfDiscoveryGroup) // Returns the same icon as the point folder icon.
      return pointGroupIcon;
    else                                                // If nothing else, this returns stringElementIconRo -- a gray rectangle.
      return stringElementIconRo;
  }

  /**
   * Returns an icon that looks like the icon for the discovery object's
   * corresponding control point's icon, except that read-only discovery
   * points get an icon with a rectangle (same color though as the control
   * point's icon). Writable discovery points get an icon with a circle (
   * exactly the same as the control point's icon)
   */
  protected BImage getPointIconDefault(BIDdfDiscoveryLeaf discovery)
  {
    TypeInfo[] databaseTypes = discovery.getValidDatabaseTypes();
    if (databaseTypes!=null && databaseTypes.length>0)
    {
      Type defaultType = databaseTypes[0].getTypeSpec().getResolvedType();
      if (defaultType!=null)
      {
        // Checks all writables before the corresponding non-writatable point because
        // writable control points subclass their corresponding non-writable counterparts.
        // In other words, writable control points are also instances of their
        // corresponding non-writable type.
        if (defaultType.is(BNumericWritable.TYPE))
          return numericElementIcon;
        else if (defaultType.is(BNumericPoint.TYPE))
          return numericElementIconRo;
        else if (defaultType.is(BBooleanWritable.TYPE))
          return booleanElementIcon;
        else if (defaultType.is(BBooleanPoint.TYPE))
          return booleanElementIconRo;
        else if (defaultType.is(BEnumWritable.TYPE))
          return enumElementIcon;
        else if (defaultType.is(BEnumPoint.TYPE))
          return enumElementIconRo;
        else if (defaultType.is(BStringWritable.TYPE))
          return stringElementIcon;
        else if (defaultType.is(BStringPoint.TYPE))
          return stringElementIconRo;
        else
          return null; // <- defaultType is not one of the std control point types.
      }
      else
        return null; // <- defaultType is not a valid baja type at all
    }
    else
      return null; // <- Driver developer did not specify any control point types for the discovery leaf
  }

  /**
   * Gets an icon for a device manager's discovery object. This method is
   * called if the discovery object does not customize the icon itself.
   */
  protected BImage getDeviceIconDefault(Object discovery)
  {
    if (discovery instanceof BIDdfDiscoveryGroup) // Returns the same icon as the device folder icon.
      return deviceGroupIcon;
    else                                           // If nothing else, this returns the default Niagara AX device icon
      return deviceIcon;
  }

  /**
   * Gets an icon for the discovery object. This method is called if the
   * discovery object does not customize the icon itself.
   */
  protected BImage getIconDefault(Object discovery)
  {
    if (getManager() instanceof BDdfDeviceManager)
      return getDeviceIconDefault(discovery); // Creates a default icon that is reasonable for the device manager
    else if (getManager() instanceof BDdfPointManager)
      return getPointIconDefault(discovery);  // Creates a default icon that is reasonable for the point manager
    else // This won't happen unless a driver developer puts a DdfMgrLearn on their own manager that does not extend BDdfDeviceManager or BDdfPointManager
      return null;
  }

/////////////////////////////////////////////////////////////////////////
//Attributes
/////////////////////////////////////////////////////////////////////////

  /**
   * Access is restricted not to be rude but to comply with API stability
   * (this field was added 12/3/2008 much later than the release of this API).
   */
  private JobBarSubscriber jobBarSubscriber;

/////////////////////////////////////////////////////////////////////////
// Attributes - Reasonable default discovery icons
/////////////////////////////////////////////////////////////////////////
  /**
   * The default discovery icon for BIDdfDeviceDiscoveryLeaf discovery objects.
   */
  public static final BImage deviceIcon = BImage.make("module://icons/x16/device.png");

  /**
   * The default discovery icon for BIDdfDiscoveryGroup objects on the Ddf Point Manager.
   */
  public static final BImage pointGroupIcon = BImage.make("module://icons/x16/pointFolder.png");

  /**
   * The default discovery icon for BIDdfDiscoveryGroup objects on the Ddf Device Manager.
   */
  public static final BImage deviceGroupIcon = BImage.make("module://icons/x16/deviceFolder.png");

  /**
   * The default discovery icon for BIDdfDiscoveryLeaf discovery objects on the point manager whose
   * default database type is NumericWritable.
   */
  public static final BImage numericElementIcon = BImage.make("module://icons/x16/control/numericPoint.png");

  /**
   * The default discovery icon for BIDdfDiscoveryLeaf discovery objects on the point manager whose
   * default database type is BooleanWritable.
   */
  public static final BImage booleanElementIcon = BImage.make("module://icons/x16/control/booleanPoint.png");

  /**
   * The default discovery icon for BIDdfDiscoveryLeaf discovery objects on the point manager whose
   * default database type is EnumWritable.
   */
  public static final BImage enumElementIcon = BImage.make("module://icons/x16/control/enumPoint.png");

  /**
   * The default discovery icon for BIDdfDiscoveryLeaf discovery objects on the point manager whose
   * default database type is StringWritable.
   */
  public static final BImage stringElementIcon = BImage.make("module://icons/x16/control/stringPoint.png");

  /**
   * The default discovery icon for BIDdfDiscoveryLeaf discovery objects on the point manager whose
   * default database type is NumericPoint.
   */
  public static final BImage numericElementIconRo = BImage.make("module://icons/x16/statusNumeric.png");

  /**
   * The default discovery icon for BIDdfDiscoveryLeaf discovery objects on the point manager whose
   * default database type is BooleanPoint.
   */
  public static final BImage booleanElementIconRo = BImage.make("module://icons/x16/statusBoolean.png");

  /**
   * The default discovery icon for BIDdfDiscoveryLeaf discovery objects on the point manager whose
   * default database type is EnumPoint.
   */
  public static final BImage enumElementIconRo = BImage.make("module://icons/x16/statusEnum.png");

  /**
   * The default discovery icon for BIDdfDiscoveryLeaf discovery objects on the point manager whose
   * default database type is StringPoint.
   */
  public static final BImage stringElementIconRo = BImage.make("module://icons/x16/statusString.png");

}
