/*
 * @copyright 2005 Tridium Inc.
 */
package com.tridium.ddf.identify;

import javax.baja.sys.BInterface;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;

import com.tridium.ddf.IDdfFacetConst;

/**
 * The BDdfAutoDiscoveryJob makes use of this class to allow
 * it to automatically, yet generically loop through all possible
 * combinations of a driver's discovery request in order to
 * retrieve all discovery objects for a driver's Ddf Device
 * Manager or Ddf Point Manager
 * @author lperkins
 *
 */
public interface BIDdfDiscoverParams
  extends BInterface, IDdfFacetConst
{

  /**
   * The implementing class needs to return a Type that will be the
   * discovery leaves in the ddf manager.
   *
   * The discovery children could hypothetically be an array of discovery
   * groups. The discovery groups could contain other discovery groups or
   * ultimately, discovery leaves. The columns that are automatically
   * generated in the discovery table for the user interface manager will
   * be introspect'ed from this type.
   *
   * @return
   */
  public Type getDiscoveryLeafType();

  /**
   * The implementing class needs to return a Type that represents
   * the discover request from the driver whose discoveryParams can be an
   * instance of this object's class. If the class supports more than
   * one discover request type, then this should return the type that is
   * the most appropriate for the particular instance of the implementing
   * class.
   *
   * This is fundamental to developer driver framework's auto-discovery features.
   *
   * @return review method description.
   */
  public Type getDiscoverRequestType();
  
  /**
   * This method is included for future extensibility. We recommend
   * that the developer return an array of size one, with the one item
   * being the same Type that is returned by the getDdfDiscoverRequestType
   * method.
   *
   * This method is used in the driver's auto-discovery feature. It allows
   * the auto-discovery mechanism to transmit multiple different discover messages
   * for the same hypothetical discovery i.d. We envision that the
   * multiple flavors of discover messages might be used for different sub-device
   * types.
   *
   * @return review method description.
   */
  public Type[] getDiscoverRequestTypes();

  /**
   * The implementing class must return an instance of itself
   * that makes sense to be the "first" identifiable entity
   * of the BDdfIdParams class.
   *
   * In the most simple scenario where an entity is identified
   * by a number between 0 and 9, an instance of BDdfIdParams
   * that represents zero would be a good example.
   *
   * @return a member of the implementing class representing the
   * first identifiable instance of the class.
   */
  public abstract BIDdfDiscoverParams getFirst();
  
  /**
   * The implementing class must return an instance of itself
   * that makes sense to be the "last" identifiable entity
   * of the BDdfIdParams class.
   *
   * In the most simple scenario where an entity is identified
   * by a number between 0 and 9, an instance of BDdfIdParams
   * that represents nine would be a good example.
   *
   * @return a member of the implementing class representing the
   * last identifiable instance of the class.
   */
  public abstract BIDdfDiscoverParams getLast();

  /**
   * The implementing class must return the next instance of itself.
   * The developer needs to determine what this would be.
   *
   * If the current instance is already the logical "last" entity
   * then this method should act in a circular fashion and return
   * the first entity.

   * In the most simple scenario where an entity is identified
   * by a number between 0 and 9, an instance of BDdfIdParams
   * that represents five would return another instance that represents
   * six from this method. Furthermore an instance that represents
   * nine should return another instance that represents zero.
   *
   * @return a member of the implementing class representing the
   * first identifiable instance of the class.
   */
  public abstract BIDdfDiscoverParams getNext();

  /**
   * The implementing class must evaluate the current instance of itself
   * against the given instance of its class and determine whether the
   * current instance is before the given instance.
   *
   * In other words, this method tests if the current instance
   * is logically "after" the given instance.
   *
   * In the most simple scenario where an entity is identified
   * by a number between 0 and 9, an instance of BDdfIdParams
   * that represents five would return true if passed another instance
   * that represents four. Conversely, the instance that represents
   * five should return false if this method is passed an instance
   * that represents six. The border scenario of an instance that
   * represents 0 should return false if passed an instance that
   * represents nine. In fact, the instance representing zero should
   * return false for any instance passed to this method.
   *
   * @return true if the current instance is logically after the
   * given instance.
   */
  public abstract boolean isAfter(BIDdfDiscoverParams anotherId);
  
  /**
   * This method needs to loop from this instance to the given instance
   * of the driver's device discover parameters, count the number of
   * iterations, and return that value.
   * 
   * Implementations may return a hard-coded value if the protocol
   * dictates a fixed number of requests.
   * 
   * @param to
   * 
   * @return the number of iterations through the getNext method from
   * this instance to the given instance. The value return should be
   * inclusive -- meaning that it includes a count for this item and
   * for the given 'to' item. If the protocol makes this unclear at
   * compile time then this should return Integer.MAX_VALUE.
   */
  public int countTo(BIDdfDiscoverParams to);

  ////////////////////////////////////////////////////////////////
  // Niagara AX Type
  ////////////////////////////////////////////////////////////////
  
  public static final Type TYPE = Sys.loadType(BIDdfDiscoverParams.class);
}
