/*
 * @copyright 2005 Tridium Inc.
 */
package com.tridium.ddf.discover.auto;

import javax.baja.control.BControlPoint;
import javax.baja.control.ext.BAbstractProxyExt;
import javax.baja.driver.BDevice;
import javax.baja.driver.BDeviceExt;
import javax.baja.driver.point.BPointDeviceExt;
import javax.baja.job.BJob;
import javax.baja.job.BJobState;
import javax.baja.job.JobCancelException;
import javax.baja.sys.BComponent;
import javax.baja.sys.BValue;
import javax.baja.sys.BajaRuntimeException;
import javax.baja.sys.Context;
import javax.baja.sys.Property;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import javax.baja.nre.util.Array;
import javax.baja.util.BFolder;

import com.tridium.ddf.BDdfDevice;
import com.tridium.ddf.BDdfNetwork;
import com.tridium.ddf.BDdfPointDeviceExt;
import com.tridium.ddf.DdfLexicon;
import com.tridium.ddf.comm.req.BIDdfDiscoverRequest;
import com.tridium.ddf.comm.req.BIDdfPingRequest;
import com.tridium.ddf.comm.req.BIDdfReadRequest;
import com.tridium.ddf.comm.req.IDdfPingable;
import com.tridium.ddf.comm.req.IDdfReadable;
import com.tridium.ddf.comm.req.util.DdfRequestUtil;
import com.tridium.ddf.discover.BDdfDiscoveryJob;
import com.tridium.ddf.discover.BIDdfDiscoveryHost;
import com.tridium.ddf.discover.BIDdfDiscoveryObject;
import com.tridium.ddf.discover.IDdfDiscoverer;
import com.tridium.ddf.identify.BDdfIdParams;
import com.tridium.ddf.identify.BIDdfDiscoverParams;
import com.tridium.ddf.point.BDdfProxyExt;

/**
 * This is a "one-size-fits-all" discovery job that takes full
 * advantage of the structure of developer driver to automatically
 * discover the devices under a network or the points under a
 * device. This will be automatically used in most drivers, unless
 * the driver developer takes special steps to customize the
 * implementation of the discovery.
 *
 * @author lperkins
 *
 */
public class BDdfAutoDiscoveryJob
  extends BDdfDiscoveryJob
{
  /*-
   class BDdfAutoDiscoveryJob
   {
     properties
     {
       discoveryPreferences : BDdfAutoDiscoveryPreferences
         -- These are the discovery preferences, passed in automatically by
         -- the ddf device manager or ddf point manager.
         default{[new BDdfAutoDiscoveryPreferences()]}
     }
   }
   -*/
/*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
/*@ $com.tridium.ddf.discover.auto.BDdfAutoDiscoveryJob(1916409288)1.0$ @*/
/* Generated Thu Oct 25 11:30:22 EDT 2007 by Slot-o-Matic 2000 (c) Tridium, Inc. 2000 */

////////////////////////////////////////////////////////////////
// Property "discoveryPreferences"
////////////////////////////////////////////////////////////////

  /**
   * Slot for the <code>discoveryPreferences</code> property.
   * These are the discovery preferences, passed in automatically
   * by the ddf device manager or ddf point manager.
   * @see com.tridium.ddf.discover.auto.BDdfAutoDiscoveryJob#getDiscoveryPreferences
   * @see com.tridium.ddf.discover.auto.BDdfAutoDiscoveryJob#setDiscoveryPreferences
   */
  public static final Property discoveryPreferences = newProperty(0, new BDdfAutoDiscoveryPreferences(),null);

  /**
   * Get the <code>discoveryPreferences</code> property.
   * @see com.tridium.ddf.discover.auto.BDdfAutoDiscoveryJob#discoveryPreferences
   */
  public BDdfAutoDiscoveryPreferences getDiscoveryPreferences() { return (BDdfAutoDiscoveryPreferences)get(discoveryPreferences); }

  /**
   * Set the <code>discoveryPreferences</code> property.
   * @see com.tridium.ddf.discover.auto.BDdfAutoDiscoveryJob#discoveryPreferences
   */
  public void setDiscoveryPreferences(BDdfAutoDiscoveryPreferences v) { set(discoveryPreferences,v,null); }

////////////////////////////////////////////////////////////////
// Type
////////////////////////////////////////////////////////////////

  public Type getType() { return TYPE; }
  public static final Type TYPE = Sys.loadType(BDdfAutoDiscoveryJob.class);

/*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

  public BDdfAutoDiscoveryJob(){} // Niagara AX requires all Types be instatiatable by default


  public BDdfAutoDiscoveryJob(BIDdfDiscoveryHost databaseParent)
  {
    this.discoveryHost=databaseParent;

    BFolder discoveryFolder = databaseParent.getDiscoveryFolder();
    if (discoveryFolder!=null)
    {
      setDiscoveryFolder(discoveryFolder.getSlotPathOrd());
    }
  }

////////////////////////////////////////////////////////////////
// BSimpleJob
////////////////////////////////////////////////////////////////

  public void run(Context cx) throws Exception
  {
    clearRootDiscoveryObjects();

    autoLearn();

    BIDdfDiscoveryObject[] discoveryObjects = getRootDiscoveryObjects();

    if (discoveryObjects!=null && discoveryObjects.length<1)
    {
      if (discoveryHost instanceof BDdfNetwork)
        throw new BajaRuntimeException(DdfAutoDiscoveryLexicon.noDevicesFound);
      else
        throw new BajaRuntimeException(DdfAutoDiscoveryLexicon.noPointsFound);
    }
  }

////////////////////////////////////////////////////////////////
// BDdfAutoDiscoveryJob
////////////////////////////////////////////////////////////////

  protected Type[] getDiscoveryRequestTypes()
  {
    BDdfIdParams minDiscoveryId = getDiscoveryPreferences().getMin();

    if (minDiscoveryId instanceof BIDdfDiscoverParams)
      return ((BIDdfDiscoverParams)minDiscoveryId).getDiscoverRequestTypes();
    else
        throw new BajaRuntimeException(DdfLexicon.ddfDiscoverParamsNeedsImplemented(minDiscoveryId.getType()));
  }

  protected void autoLearn()
    throws Exception
  {
    // Gets all message types that can be used to discover
    Type[] discoverReqTypes = getDiscoveryRequestTypes();

    // Loops through all discover request types that can be used to discover
    for (int i=0; i<discoverReqTypes.length; i++)
    {
      if (discoverReqTypes[i].is(BIDdfDiscoverRequest.TYPE))
        // Proceeds by sending out a discover request for all possible BDdfIdParams's in the range provided in the discovery parameters
        learnFromRequestType(discoverReqTypes[i],100/discoverReqTypes.length);
      else
        discoveryHost.getDdfCommunicator().getLog().error(
            DdfAutoDiscoveryLexicon.typeMustImplementDiscoveryRequest(discoverReqTypes[i]));
    }
  }

  /**
   *
   * @param requestType
   * @param progressValueOfMethod is the absolute percentage that one pass through this method is worth on
   * the job's progress bar.
   */
  protected void learnFromRequestType(Type requestType, int progressValueOfMethod)
  {
    BIDdfDiscoverParams minDiscoverId = (BIDdfDiscoverParams)getDiscoveryPreferences().getMin().newCopy();

    BIDdfDiscoverParams maxDiscoverId = (BIDdfDiscoverParams)getDiscoveryPreferences().getMax().newCopy();

    // Gets the total number of discovery request that will be attempted, when all is said-and-done. We need this
    // To update the dob progress bar (percent complete)
    int numRequestsToTry=minDiscoverId.countTo(maxDiscoverId);

    // Loops from the min BDdfIdParams to the max BDdfIdParams provided in the discovery parameters
    for (BIDdfDiscoverParams currentDiscoverId = minDiscoverId,             // This is the id to discover in the following loop iteration.
         previousDiscoverId = minDiscoverId;                    // This is the id that was discovered in the previous loop iteration.
           !previousDiscoverId.isAfter(currentDiscoverId) &&       // Loops while the previousDiscoverId is less than or equal to the currentDiscoverId
           !currentDiscoverId.isAfter(maxDiscoverId);              // AND while the currentDiscoverId is less than or equal to the maxDiscoverId
         previousDiscoverId=currentDiscoverId,                  // At the end of loop iterations, this sets previousDiscoverId to currentDiscoverId
         currentDiscoverId = currentDiscoverId.getNext())       // And then sets currentDiscoverId to the next discover id
    {
      if (getJobState()==BJobState.canceling) // This check is done in case the user cancels the job
        throw new JobCancelException();
      else // If user did not click cancel
      { // Proceeds to transmit the discover request to discover the currentDiscoverId
        learn( (BIDdfDiscoverRequest)requestType.getInstance(), currentDiscoverId );

        // Updates the job progress bar on the client side
        progress(getProgress()+ (int)      // Increments the current progress by the percent value of this loop iteration
            ( (float)progressValueOfMethod // The percent value of this loop iteration is equal to the 'progressValueOfMethod'
             /(float)numRequestsToTry ));     // Divided by the number of times that this method will

        // NOTE: The 'progress' call above moves 0 in the event that numRequestToTry is Integer.MAX_VALUE
      }
    }

    // If the count from min to max is unknown at compile time, then we report full completion
    // At the return of this method.
    if (numRequestsToTry == Integer.MAX_VALUE)
      progress(getProgress() +  progressValueOfMethod);
  }
  /**
   * Gets all of the database components under the databaseParent so
   * that each can be examined and possibly updated by the discovery
   * response, if it is determined that they could be updated anyway
   * from the discovery response.
   *
   * @return the array of devices (if being used with a device manager) or
   * the array of control points (if being used with a point manager)
   */
  private BComponent[] getDatabaseObjects()
  {
    if (discoveryHost instanceof BDdfNetwork)
    {
      BDdfNetwork network = (BDdfNetwork)discoveryHost;

      BDevice[] dbDevices = network.getDevices();
      return dbDevices;
    }
    else if (discoveryHost instanceof BDdfPointDeviceExt)
    {
      Array<BControlPoint> controlPointsArray = new Array<>(BControlPoint.class);

      BDdfDevice device = (BDdfDevice)
      ((BDdfPointDeviceExt)discoveryHost).getDevice();

      BDeviceExt deviceExts[] = device.getDeviceExts();

      for (int i=0; i<deviceExts.length; i++)
      {
        if (deviceExts[i] instanceof BPointDeviceExt)
        {
          BPointDeviceExt ptDvExt = (BPointDeviceExt)deviceExts[i];
          controlPointsArray.addAll(ptDvExt.getPoints());
        }
      }
      return controlPointsArray.trim();
    }
    else
      throw new IllegalStateException("Not Implemented For Database Parent Type: "+discoveryHost.getType().getTypeName());
  }

  /**
   * Gets an array of objects to receive callbacks when the discovery request
   * for the given devicePingId either times-out or receives a response.
   * @param devicePingId
   * @return an array containing a default call back item for this job, plus all database items
   * for which the pingId can update.
   */
  private IDdfPingable[] getPingableSource(final BDdfIdParams devicePingId)
  {
    Array<IDdfPingable> pingableSource = new Array<>(IDdfPingable.class);

    // Configures all database devices with the same nextId for update if we receive a response
    BComponent[] dbDevices = getDatabaseObjects();

    for (int i=0; i<dbDevices.length; i++)      // Loops through database devices
    {
      if (dbDevices[i] instanceof BDdfDevice)  // Looks at each BDdfDevice
      {
        BDdfDevice ddfDevice = (BDdfDevice)dbDevices[i];

        BDdfIdParams deviceId = ddfDevice.getDeviceId();  // Gets the deviceId

        if ((deviceId.getType().equals(devicePingId.getType())) && // If the deviceId type is the same as the devicePingId type and
            (ddfDevice.getDeviceId().equals(devicePingId)))  // If the device id matches the devicePingId
          pingableSource.add(ddfDevice);             // Then this adds the ddf device as a pingable source, so that it will be updated from the ping response
      }
    }
    return pingableSource.trim();
  }

  /**
   * Gets an array of objects to receive callbacks when the discovery request
   * for the given readReqId either times-out or receives a response.
   * @param readParameters
   * @return an array containing a default call back item for this job, plus all database points
   * (under the same device) that the readParameters can read.
   */
  private IDdfReadable[] getReadableSource(final BDdfIdParams readParameters)
  {
    //log().message(DdfAutoDiscoveryLexicon.pointsNotFound(readParameters));
    Array<IDdfReadable> readableSource = new Array<>(IDdfReadable.class);

    // Adds all database points with the equivalent readParametersreadParameters
    BComponent[] dbPoints = getDatabaseObjects();

    for (int i=0; i<dbPoints.length; i++)      // Loops through database points
    {
      if (dbPoints[i] instanceof BControlPoint)  // Looks at each BControlPoint
      {
        BAbstractProxyExt proxy = ((BControlPoint)dbPoints[i]).getProxyExt();

        if (proxy instanceof BDdfProxyExt)      // Sanity check
        {
          BDdfIdParams requestId = ((BDdfProxyExt)proxy).getReadParameters();  // Gets the read parameters on a proxy

          if ((requestId.getType().equals(readParameters.getType())) && // If the proxy's read parameters' Type is the same as the read parameters' type (of the discovery) AND...
               (requestId.equivalent(readParameters)))  // If the proxy's read parameters are equivalent to the readParameters (for the discovery)
          {
            readableSource.add((IDdfReadable)proxy);             // Then this adds the proxy as a readable source, so that it will incidentally be updated from the discover response
          }
        }
      }
    }
    return readableSource.trim();
  }


  /**
   * Transmits a given, empty, default instance of BIDdfRequest request
   * @param discoverReq an empty, default instance of a particular
   * @param discoverId
   */
  protected void learn(BIDdfDiscoverRequest discoverReq, final BIDdfDiscoverParams discoverId)
  {
    discoverReq.setResponseTimeout(getDiscoveryPreferences().getTimeout());

    discoverReq.setRemainingRetryCount(getDiscoveryPreferences().getRetryCount());

    discoverReq.setDiscoverParameters((BDdfIdParams)discoverId);

    // Defines what happens when the discover request times out or gets its response
    discoverReq.setDiscoverer(new IDdfDiscoverer()
      {
        /**
         * Gives the developer direct access to this auto discovery job.
         */
        public BJob getJob()
        {
          return BDdfAutoDiscoveryJob.this;
        }
        /**
         * The ddf calls this method if the 'discoverReq receives' no discover response.
         */
        public void discoverFail(String reason)
        {
          log().message(reason);
        }
        /**
         * The ddf calls this method if the 'discoverReq receives' a discover response.
         *
         * The ddf passes the 'discoveryObjects' from the discover response.
         */
        public void discoverOk(BIDdfDiscoveryObject[] discoveryObjects)
        {
          BComponent discoveryFolder = discoveryHost.getDiscoveryFolder();

          if (discoveryFolder==null)
            discoveryFolder=BDdfAutoDiscoveryJob.this;

          for (int i=0; i<discoveryObjects.length; i++)
            discoveryFolder.add("d?",(BValue)discoveryObjects[i]);
        }

        /**
         * The ddf passes this context to the discover response's 'parseDiscoveryObjects'
         * method to tell the developer whether this job is discovering devices or
         * points. This Context only important if the developer is using the same request
         * type for both a device learn and a point learn. Such a scenario is probably
         * unlikely.
         */
        public Context getDiscoverContext()
        {
          if (discoveryHost instanceof BDdfNetwork)
            return CONTEXT_DISCOVERING_DEVICES;
          else // It must be an instanceof BDdfPointDeviceExt, I'd venture to say.
            return CONTEXT_DISCOVERING_POINTS;
        }

      }
    );
    // If discovering devices
    if (discoveryHost instanceof BDdfNetwork)
    { if (discoverReq instanceof BIDdfPingRequest)
      {
        // Device id is also the discoverId...think about it...if discovering on behalf of the network then we are discovering devices
        discoverReq.setDeviceId((BDdfIdParams)
            ((BDdfIdParams)discoverId).newCopy());

        // Adds the objects that will receive callbacks upon successful response
        ((BIDdfPingRequest)discoverReq).setPingableSource(
            getPingableSource((BDdfIdParams)
                ((BDdfIdParams)discoverId).newCopy()));
      }
    }
    else if (discoveryHost instanceof BDdfPointDeviceExt)
    {
      // Device id comes from the device, not the discover id
      discoverReq.setDeviceId( (BDdfIdParams)
          ((BDdfDevice)
              ((BDdfPointDeviceExt)discoveryHost).getDevice())
              .getDeviceId().newCopy());

      if (discoverReq instanceof BIDdfReadRequest)
      {
        ((BIDdfReadRequest)discoverReq).setReadParameters((BDdfIdParams)
            ((BDdfIdParams)discoverId).newCopy()); // The read parameters structure is also the discoverId...think about it...if discovering on behalf of the point device ext then we are discovering points

        // Adds the objects that will receive callbacks upon successful response
        ((BIDdfReadRequest)discoverReq).setReadableSource(
            getReadableSource((BDdfIdParams)discoverId));
      }
    }
    //else
      //throw new IllegalStateException("Not Implemented For Database Parent Type: "+discoveryHost.getType().getTypeName());

    // TODO: Enhance - In multiple transaction environments we should be able to hammer out
    // All requests at once and allow the responses to arrive in a true asynchronous manner.
    DdfRequestUtil.communicateSync(discoveryHost.getDdfCommunicator(), discoverReq);
  }

  /**
   * Gets the Ddf Network or Ddf Point Device Extension that is hosting
   * the discovery job.
   * @return the Ddf Network or Ddf Point Device Extension that is hosting
   * the discovery job, cast as a BIDdfDiscoveryHost
   */
  public BIDdfDiscoveryHost getDiscoveryHost()
  {
    return discoveryHost;
  }


  private BIDdfDiscoveryHost discoveryHost = null; // This will be a BDdfNetwork or a BDdfDevice
}
