/*
 * @copyright 2005 Tridium Inc.
 */
package com.tridium.ddf.comm.req;

import javax.baja.sys.BInterface;
import javax.baja.sys.BRelTime;
import javax.baja.sys.BSimple;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;

import com.tridium.ddf.IDdfFacetConst;
import com.tridium.ddf.comm.IDdfDataFrame;
import com.tridium.ddf.comm.rsp.BIDdfResponse;
import com.tridium.ddf.comm.rsp.DdfResponseException;
import com.tridium.ddf.identify.BDdfIdParams;

/**
 * BIDdfRequest represents serializable data to be transmitted out of a BDdfCommunicator
 * in a station. A BDdfCommunicator can be a BDdfSerialCommunicator, a BDdfTcpCommunicator,
 * , or some other object that extends BDdfCommunicator.
 * 
 * @author lperkins
 *
 */
public interface BIDdfRequest
  extends BInterface, IDdfFacetConst
{
  /**
   * All requests are addressed to an actual device.
   *
   * Occasionally some protocols have global requests. In those
   * cases, there are two scenarios that we have always
   * encountered:
   *
   * 1.) The protocols send requests to a reserved address on the
   * field-bus. In that case, the deviceId here will be a custom-instance
   * of BDdfIdParams that identifies the particular reserved device id.
   *
   * 2.) The protocols define a special request that needs no device-id
   * because the request itself applies specifically globally. In that
   * case, the deviceId can be an instance of BDdfIdParams or
   * some other customized instance of BDdfIdParams that essentially
   * means that the device address is null.
   *
   * @return
   */
  public BDdfIdParams getDeviceId();
  /**
   * The ddf calls this method automatically as it
   * instantiates requests to transmit on behalf of devices or points
   * under devices.
   */
  public void setDeviceId(BDdfIdParams deviceId);

  /**
   * The ddf uses the return value from this method
   * to match up requests to responses. Implementing this method to return
   * a value other than null is especially important for drivers that use
   * the ddf multiple-transaction-manager to match reqeusts to responses.
   *
   * For the simplest master-slave, serial protocols, this
   * can return null. These types of drivers should just use an
   * ddf single-transaction-manager anyway, which always matches the most
   * recently received frame with the most recently transmitted request.
   *
   * For more complicated drivers that support multiple outstanding transactions,
   * a BString that encodes the hex string that the request expects in the response
   * will be adequate.
   *
   * @return a BSimple with the same hashCode that you design to also be in the
   * corresponding BSimple tag of the BIDdfResponse.
   */
  public BSimple getTag();

  /**
   * Driver developers should follow their driver's protocol documentation
   * and construct the appropriate byte array for the request here. Please
   * note that in most cases, the ddf will have already
   * automatically called setDeviceId and passed in the information necessary
   * to address the device on the field-bus to which your request will
   * ultimately be transmitted. Therefore, you can call getDeviceId, and
   * cast the result into the property type for your driver's device's
   * deviceId property (that you defined in your device's slotomatic header).
   *
   * Please note that read requests (BIDdfReadRequest) can also call getReadParameters
   * for a copy of the proxy's (IDdfReadable's) readParameters property value.
   *
   * Please note that ping requests (BIDdfPingRequest) can also call getPingParameters
   * for a copy of the device's (IDdfPingable's) pingParameters property value.
   *
   * Please note that discovery requests (BIDdfPingRequest) can also call getDiscoveryId
   * for a copy of the (IDdfPingable's) discoverId property value that is between the min
   * and max discoveryId as defined on your driver's discovery preferences.
   *
   * Please note that write requests (BIDdfWriteRequest) can also call getWriteParameters
   * for a copy of the proxy's (IDdfWritable's) writeParameters property value.
   *
   * @return a byte array following the instuctions above
   */
  public byte[] toByteArray();


  /**
   * The ddf calls this method to find out long to wait
   * after transmitting the request before timing out. The easiest implementation
   * would be for you to return the value was most recently passed to the
   * setResponseTimeout method.
   */
  public BRelTime getResponseTimeout();
  /**
   * By default, the ddf calls this automatically before
   * calling the toByteArray method, and passes in the value that response
   * timeout value from the receiver component under your driver's
   * communicator.
   */
  public void setResponseTimeout(BRelTime v);

  public static final int USE_DEFAULT_RESPONSE_TIMEOUT = -1;
  public static final int USE_DEFAULT_RETRY_COUNT = -1;

  /**
   * Gets the remaining number of retries for this message. The ddf
   * driver framework will call setRemainingRetryCount with a decremented
   * count, as necessary, during the retry processing.
   *
   * If BIDdfRequest.USE_DEFAULT_RETRY_COUNT is initially returned then
   * the ddf will initally call setMaxRetryCount and pass in the
   * default value from the ddf communicator's transmitter through which
   * this request is transmitted. After that, it will subsequently decrement
   * the remaining retry count by calling setRemainingRetryCount with a descremented
   * value, for each failed transmission attempt, until this method returns
   * zero.
   */
  public int getRemainingRetryCount();

  /**
   * The ddf communicator will call this to decrement the number of remaining
   * retries, as appropriate, during its attempts to successfully transmit this
   * request.
   */
  public void setRemainingRetryCount(int v);

  /**
   * After transmitting this request, the BDdfCommunicator will pass frames that it receives
   * here. If you implement the getTag method then the ddf communicator will only pass data frames
   * whose tag's hashcode matches your request tag's hashcode. If your request returns null from
   * the getTag method then all received data frames will be passed here, until the request times
   * out or returns a BIDdfResponse from this method.
   *
   *  This request needs to take one of the following steps:
   *   1. Ignore the frame and return null. The transaction remains open.
   *   2. Collect the frame and return a BIDdfMultiFrameResponse whose <i>isComplete</i> method returns <i>false</i>. In which case, you need to implement your own collection
   *      mechanism. For example, this could be as simple as putting them all in a Vector in the BIDdfMultiFrameResponse. In
   *      this scenario, the transaction remains open. The transaction response timeout will be reset to the <i>responseTimeout</i> of the
   *      returned <i>BIMultiFrameResponse</i> thereby allowing more time for the remainder of the transaction to complete.
   *   3. Return a BIDdfResponse for the data frame and any previously collected frames that you determine together make up a completed response.
   *      This completes the transaction. Alternatively, if you have any previously collected frames then you may return the instance of
   *      BIDdfMultiFrameResponse as long as its <i>isComplete</i> method subsequently returns <i>true</i>. Both of these scenarios
   *      complete the transaction.  
   *   4. Throw an DdfResponseException or subclass there-of to indicate the the frame
   *   forms a complete message but indicates an error condition in the device preventing
   *   a successful response. This completes the transaction.
   *
   *   WARNING: In scenario's 2 and 3, please copy the frame's bytes as the frame's byte array could be a direct reference to an internal
   *   buffer in the receiver.
   *   
   * @param iDdfDataFrame
   * 
   * @return BIDdfResponse but please review the full description for this method.
   * 
   * @throws DdfResponseException but please review the full description for this method.
   */
  public BIDdfResponse processReceive(IDdfDataFrame iDdfDataFrame)
    throws DdfResponseException;
  
  ////////////////////////////////////////////////////////////////
  // Niagara AX Type
  ////////////////////////////////////////////////////////////////
  public static final Type TYPE = Sys.loadType(BIDdfRequest.class);

}
