/*
 * @copyright 2005 Tridium Inc.
 */
package com.tridium.ddf.comm.multipleTransaction;

import java.util.Enumeration;
import java.util.Hashtable;

import javax.baja.sys.BComponent;
import javax.baja.sys.BFacets;
import javax.baja.sys.BInteger;
import javax.baja.sys.BSimple;
import javax.baja.sys.Property;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import javax.baja.util.Queue;

import com.tridium.ddf.comm.IDdfDataFrame;
import com.tridium.ddf.comm.defaultComm.BDdfTransactionMgr;
import com.tridium.ddf.comm.defaultComm.DdfDefaultCommLexicon;
import com.tridium.ddf.comm.req.BIDdfRequest;
import com.tridium.ddf.comm.req.util.DdfRequestUtil;
import com.tridium.ddf.comm.rsp.BDdfResponse;
import com.tridium.ddf.comm.rsp.BIDdfMultiFrameResponse;
import com.tridium.ddf.comm.rsp.BIDdfResponse;
import com.tridium.ddf.comm.rsp.DdfResponseException;
import com.tridium.ddf.comm.rsp.IDdfTransmitAckResponse;

/**
 * BDdfMultipleTransactionMgr goes under BDdfMultipleTransactionCommunicator in the Niagara AX
 * Nav tree. This allows more than one outstanding request on the field-bus at a time. This
 * uses tags (if defined on the requests and responses) to match requests to received frames. If
 * no tags are defined then requests receive all incoming data frames while the requests are
 * outstanding. The requests themselves can filter through the incoming data frames if they
 * do not define a tag.
 *
 * Rather than extending this directly, the devSerialDriver module provides a very reasonable
 * implementation of this. If your driver is serial then you can save time by using the 
 * transaction mgr from the following communicator:
 * <ul>
 * com.tridium.ddfSerial.comm.multipleTransaction.BDdfSerialMutCommunicator
 * </ul>
 * 
 * The devIpDriver module provides a very reasonable implementation of this. If your
 * driver communicates over IP then you can save time by using the multiple
 * transaction manager from either of the following communicators:
 * <ol>
 * <li>com.tridium.ddfIp.tcp.comm.multipleTransaction.BDdfTcpMutCommunicator <b>OR</b>
 * <li>com.tridium.ddfIp.udp.comm.multipleTransaction.BDdfUdpMutCommunicator
 * </ol>
 * 
 *
 * @author    lperkins
 * @creation  Oct 10, 2006
 * @version   $Revision$ $Date$
 * @since     Niagara 3.0
 */
public class BDdfMultipleTransactionMgr
  extends BDdfTransactionMgr
  implements BIDdfMultipleTransactionMgr
{
  /*-
   class BDdfMultipleTransactionMgr
   {
     properties
     {
       outstandingCount : int
         -- This is the number of request messages that have been transmitted but the driver is still awaiting
         -- a response for.
         default{[0]}
        slotfacets{[BFacets.make(BFacets.MIN,BInteger.make(0))]}
       expiredCount : int
         -- This is the number of request messages that were transmitted and timed out. The driver may still
         -- be waiting for a response for these.
         default{[0]}
        slotfacets{[BFacets.make(BFacets.MIN,BInteger.make(0))]}
       maxExpired : int
         -- This is the maximum number of expired messages for which the ddf transaction manager
         -- Will listen for late replies. Expired requests will be stored in a FIFO queue of this
         -- Length. When a new outstanding reply fails to receive a response, and if the expired
         -- count equals this value, then the oldest item in the expired queue will be purged to
         -- Make room for the new outstanding response. If this value is decreased while the station
         -- Is running, then the expired queue size might not take effect until the next outstanding
         -- Request times out. Therefore, do not be alarmed if after reducing this number, the
         -- Corresponding expiredCount stays higher. It will take until the next outstanding request
         -- Times out before the queue size will be shortened. After that time, the expiredCount will
         -- Be reduced to a number equal to or less than the new maxExpired value.
         default{[50]}
        slotfacets{[BFacets.make(BFacets.MIN,BInteger.make(1))]}
     }
     actions
     {
     }
   }
   -*/
/*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
/*@ $com.tridium.ddf.comm.multipleTransaction.BDdfMultipleTransactionMgr(3644528167)1.0$ @*/
/* Generated Thu Oct 25 11:30:22 EDT 2007 by Slot-o-Matic 2000 (c) Tridium, Inc. 2000 */

////////////////////////////////////////////////////////////////
// Property "outstandingCount"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>outstandingCount</code> property.
   * This is the number of request messages that have been
   * transmitted but the driver is still awaiting a response
   * for.
   * @see com.tridium.ddf.comm.multipleTransaction.BDdfMultipleTransactionMgr#getOutstandingCount
   * @see com.tridium.ddf.comm.multipleTransaction.BDdfMultipleTransactionMgr#setOutstandingCount
   */
  public static final Property outstandingCount = newProperty(0, 0,BFacets.make(BFacets.MIN,BInteger.make(0)));
  
  /**
   * Get the <code>outstandingCount</code> property.
   * @see com.tridium.ddf.comm.multipleTransaction.BDdfMultipleTransactionMgr#outstandingCount
   */
  public int getOutstandingCount() { return getInt(outstandingCount); }
  
  /**
   * Set the <code>outstandingCount</code> property.
   * @see com.tridium.ddf.comm.multipleTransaction.BDdfMultipleTransactionMgr#outstandingCount
   */
  public void setOutstandingCount(int v) { setInt(outstandingCount,v,null); }

////////////////////////////////////////////////////////////////
// Property "expiredCount"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>expiredCount</code> property.
   * This is the number of request messages that were transmitted
   * and timed out. The driver may still be waiting for
   * a response for these.
   * @see com.tridium.ddf.comm.multipleTransaction.BDdfMultipleTransactionMgr#getExpiredCount
   * @see com.tridium.ddf.comm.multipleTransaction.BDdfMultipleTransactionMgr#setExpiredCount
   */
  public static final Property expiredCount = newProperty(0, 0,BFacets.make(BFacets.MIN,BInteger.make(0)));
  
  /**
   * Get the <code>expiredCount</code> property.
   * @see com.tridium.ddf.comm.multipleTransaction.BDdfMultipleTransactionMgr#expiredCount
   */
  public int getExpiredCount() { return getInt(expiredCount); }
  
  /**
   * Set the <code>expiredCount</code> property.
   * @see com.tridium.ddf.comm.multipleTransaction.BDdfMultipleTransactionMgr#expiredCount
   */
  public void setExpiredCount(int v) { setInt(expiredCount,v,null); }

////////////////////////////////////////////////////////////////
// Property "maxExpired"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>maxExpired</code> property.
   * This is the maximum number of expired messages for
   * which the ddf transaction manager Will listen for late replies. Expired requests will be stored in a FIFO queue of this Length. When a new outstanding reply fails to receive a response, and if the expired count equals this value, then the oldest item in the expired queue will be purged to Make room for the new outstanding response. If this value is decreased while the station Is running, then the expired queue size might not take effect until the next outstanding Request times out. Therefore, do not be alarmed if after reducing this number, the Corresponding expiredCount stays higher. It will take until the next outstanding request Times out before the queue size will be shortened. After that time, the expiredCount will Be reduced to a number equal to or less than the new maxExpired value.
   * @see com.tridium.ddf.comm.multipleTransaction.BDdfMultipleTransactionMgr#getMaxExpired
   * @see com.tridium.ddf.comm.multipleTransaction.BDdfMultipleTransactionMgr#setMaxExpired
   */
  public static final Property maxExpired = newProperty(0, 50,BFacets.make(BFacets.MIN,BInteger.make(1)));
  
  /**
   * Get the <code>maxExpired</code> property.
   * @see com.tridium.ddf.comm.multipleTransaction.BDdfMultipleTransactionMgr#maxExpired
   */
  public int getMaxExpired() { return getInt(maxExpired); }
  
  /**
   * Set the <code>maxExpired</code> property.
   * @see com.tridium.ddf.comm.multipleTransaction.BDdfMultipleTransactionMgr#maxExpired
   */
  public void setMaxExpired(int v) { setInt(maxExpired,v,null); }

////////////////////////////////////////////////////////////////
// Type
////////////////////////////////////////////////////////////////
  
  public Type getType() { return TYPE; }
  public static final Type TYPE = Sys.loadType(BDdfMultipleTransactionMgr.class);

/*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/
////////////////////////////////////////////////////////////////
// BComplex
////////////////////////////////////////////////////////////////
  public boolean isParentLegal(BComponent parent)
  {
    return parent instanceof BDdfMultipleTransactionCommunicator;
  }
  
////////////////////////////////////////////////////////////////
// BIDdfTransactionMgr
////////////////////////////////////////////////////////////////
  public void stopTransactionMgr()
  {
    try
    {
      super.stopTransactionMgr();
    }
    finally
    {
      // Gets all of the tags for all of the requests that have yet to receive a response
      Enumeration<BSimple> outstandingTags = outstandingRequestTags.keys();
      
      // Sanity check...
      if (outstandingTags!=null)
      {
        // Loops through all tags for all requests that have yet to receive a response
        while (outstandingTags.hasMoreElements())
        {
          // Looks at the next tag for a request that has yet to receive a response
          BSimple outstandingTag = outstandingTags.nextElement();
          
          // There could be one or more outstanding requests that map to the particular tag object.
          // This gets a table that contains all the requests for the particular tag
          Hashtable<BIDdfRequest, BIDdfRequest> outstandingRequestsForTag = outstandingRequestTags.get(outstandingTag);
          
          // Sanity check.
          if (outstandingRequestsForTag!=null)
          {
            // Gets all of the requests for the particular tag. Note that the requests are all
            // stored in the 'outstandingRequestsForTag' hashtable. They are keys in the table.
            // Being keys, they also happen to map to themselves. This is done to all outstanding
            // requests to be easily retrieved during the response process (but of course, this
            // is the cleanup function, not the response process so we really do not benefit from
            // this mapping in this function)
            Enumeration<BIDdfRequest> outstandingRequests = outstandingRequestsForTag.keys();

            // Loops through the one or more requests for the one particular tag
            while (outstandingRequests.hasMoreElements())
            {
              // Looks at the next request for the one particular tag
              BIDdfRequest outstandingRequest = outstandingRequests.nextElement();
              
              // Prints a trace to indicate that the request is now considered as a timeout
              if (getDdfCommunicator().getLog().isTraceOn())
                getDdfCommunicator().getLog().trace(
                    DdfDefaultCommLexicon.requestTimedOut(outstandingRequest));
              
              // Removes the request from the outstandingRequestTags
              removeIdentifiableRequest(outstandingRequest, outstandingRequestTags);
              
              // Allows the message to take further action for the timeout to call readFail, etc.
              DdfRequestUtil.processTimeout(outstandingRequest);
            }
          }
        }
      }
    }
  }
  
  
////////////////////////////////////////////////////////////////
// BDdfTransactionMgr
////////////////////////////////////////////////////////////////
  
  /**
   * This method is called by the receiver for every raw frame of data that it receives.
   */
  protected void frameReceived(IDdfDataFrame ddfReceiveFrame)
  {
    boolean frameConsumed = false;
    if (matchOutstandingRequest(ddfReceiveFrame))
      frameConsumed=true;
    if( matchExpiredRequest(ddfReceiveFrame))
      frameConsumed=true;
    if (!frameConsumed)
      routeToUnsolicited(ddfReceiveFrame);
  }

  /**
   * This method is called by the BDdfTransactionMgr at the start of a
   * request/response transaction.
   */
  protected void beginTransaction(BIDdfRequest req)
  {
    try
    {
      addIdentifiableRequest(req,outstandingRequestTags);
    }
    finally
    {
      updateCounts();
    }
  }  
  
  /**
   * This method is called on the transaction manager thread, just moments
   * after the checkOutstandingTimeout action is invoked by means of a
   * BDdfScheduler.INSTANCE.schedule that occurred when the request was last transmitted.
   */
  protected void doCheckOutstandingTimeout(BIDdfRequest ddfRequest)
  {
    try
    {
      // If the ddfRequest is still outstanding
      if (containsIdentifiableRequest(ddfRequest,outstandingRequestTags))
      {
        synchronized(ddfRequest)
        { 
          // If it has any retries left
          if (ddfRequest.getRemainingRetryCount()>0)
          { 
            // Then lets try again one time
            ddfRequest.setRemainingRetryCount(ddfRequest.getRemainingRetryCount()-1);
            
            try
            {
              getDdfCommunicator().getDdfTransmitter().forceTransmit(ddfRequest);
              
              getDdfCommunicator().getDdfTransmitter().setRetransmissionCount(
                  getDdfCommunicator().getDdfTransmitter().getRetransmissionCount()+1); // Updates the retransmission count
            }
            catch (Exception e)
            {
              if (getDdfCommunicator().getLog().isTraceOn())
                getDdfCommunicator().getLog().trace("RetransmissionError - "+e.toString());
            }
            // And let's schedule another check for timeout
            scheduleToCheckForTimeout(ddfRequest);
          }
          else // There are no (remaining) timeouts permitted. This message is timing out
          {
            if (getDdfCommunicator().getLog().isTraceOn())
              getDdfCommunicator().getLog().trace(
                  DdfDefaultCommLexicon.requestTimedOut(ddfRequest));
            
            // Removes the request from the outstandingRequestTags
            removeIdentifiableRequest(ddfRequest, outstandingRequestTags);
            
            // Adds the outstanding request to the expiredRequestTags
            addExpiredRequest(ddfRequest);
            
            // Allows the message to take further action for the timeout to call readFail, etc.
            DdfRequestUtil.processTimeout(ddfRequest);
          }
        }
      }
    }
    catch (Exception e)
    {
      getDdfCommunicator().getLog().error(e.toString(),e);
    }
    finally
    {
      updateCounts();
    }
  }
  
////////////////////////////////////////////////////////////////
// BDdfMultipleTransactionMgr
////////////////////////////////////////////////////////////////

  private void addExpiredRequest(BIDdfRequest expiredReq)
  {
    int maxExpiredRequestsAllowed = getMaxExpired() - 1; // We subtract one because we need room to add the expiredReq that was passed to this method.
    
    // Keeps the 'expiredRequestTagsFifoHelper'  less than the
    // 'maxExpired' value on the property sheet, minus one. I use a
    // while loop instead of an if statement in case the user changes the
    // maxExpired property. The change will take effect the next time
    // an outstanding request becomes expired.
    while (expiredRequestTagsFifoHelper.size()>maxExpiredRequestsAllowed)
    {
      BIDdfRequest purgeRequest = (BIDdfRequest)expiredRequestTagsFifoHelper.dequeue();
      removeIdentifiableRequest(purgeRequest, expiredRequestTags);
    }
    
    // Adds the expiredReq to our helper queue (which helps us keep track of
    // the number of expired requests for which the ddf will
    // listen for a late response)
    expiredRequestTagsFifoHelper.enqueue(expiredReq);
    addIdentifiableRequest(expiredReq, expiredRequestTags);
  }

  /**
   * This method is called to add the given request to the given hash table. The caveat though
   * is that the given hash table is actually a hash table of hash tables. The inner hash tables
   * are hashed on request tags. Therefore, this method adds the given request to the inner
   * hashtable of requests that share the same tag. The inner hashtable of requests is hashed
   * to the given hashtable by request tag.
   *
   *  This allows constant lookup into the given hashtable by request tag. The resulting inner
   *  hashtable contains an indexed list of requests for the same tag.
   *
   * @param req
   * @param tagsToRequests
   */
  private static void addIdentifiableRequest(BIDdfRequest req, Hashtable<BSimple, Hashtable<BIDdfRequest, BIDdfRequest>> tagsToRequests)
  {
    BSimple reqTag = req.getTag(); // Gets the message tag for the request
    if (reqTag == null)
    { 
      // This is a sanity check, the multiple transaction communicator relies heavily on request tags
      // To function property. Although I could alternatively set req to BString.DEFAULT if req.getTag()
      // Is null, I think that the driver developer needs to think about it further and determine whether
      // Or not to make his or her req.getTag() method return BString.DEFAULT. By returning something so
      // Generic, the benefits of the indexing in this class are minimized.
      throw new NullPointerException(DdfDefaultCommLexicon.mutTagCannotBeNull(req));
    }
    synchronized(tagsToRequests) // All of the operations in the idsToRequests table must be done exclusively by one thread at a time
    {
      Hashtable<BIDdfRequest, BIDdfRequest> requestsForTag = tagsToRequests.get(reqTag); // Gets all other outstanding requests for the same tag (if there are any)
      if (requestsForTag==null) // If no table exists for the reqTag
      {
        requestsForTag = new Hashtable<>(); // Creates a table for the reqTag (Uses a table to allow for more than one outstanding request with the same tag)
        tagsToRequests.put(reqTag,requestsForTag); // Puts the request into the tagsToRequests table
      }
      requestsForTag.put(req,req); // This makes it be constant look-up time to retrieve the request from the requestsForTag hashtable
    }
  }
  
  private static void removeIdentifiableRequest(BIDdfRequest req, Hashtable<BSimple, Hashtable<BIDdfRequest, BIDdfRequest>> tagsToRequests)
  {
    BSimple reqTag = req.getTag(); // Gets the messagetag for the request
    synchronized(tagsToRequests) // All of the operations in this block must be done exclusively by one thread at a time
    {
      Hashtable<BIDdfRequest, BIDdfRequest> requestsForTag = tagsToRequests.get(reqTag); // Gets all requests for the reqTag
      if (requestsForTag!=null) // If no table exists for the reqTag then that might be of concern!
      {
        requestsForTag.remove(req); // Removes the request for the table of requests for the particular reqTag
        if (requestsForTag.size()<1) // If req was the last request then this removes the table of requests for the particular tag from the table of tags-to-requests
          tagsToRequests.remove(reqTag);
      }
    }
  }

  private static int countIdentifiableRequests(Hashtable<BSimple, Hashtable<BIDdfRequest, BIDdfRequest>> tagsToRequests)
  {
    int count = 0;
    Enumeration<Hashtable<BIDdfRequest, BIDdfRequest>> allRequestsForAllTags = tagsToRequests.elements();

    while (allRequestsForAllTags.hasMoreElements())
    {
      Hashtable<BIDdfRequest, BIDdfRequest> requestsForSomeTag = allRequestsForAllTags.nextElement();
      count += requestsForSomeTag.size();
    }
    return count;
  }

  private static boolean containsIdentifiableRequest(BIDdfRequest req, Hashtable<BSimple, Hashtable<BIDdfRequest, BIDdfRequest>> tagsToRequests)
  {
    BSimple reqTag = req.getTag(); // Gets the message tag for the request
    Hashtable<BIDdfRequest, BIDdfRequest> requestsForTag = tagsToRequests.get(reqTag); // Gets all requests for the message Id
    if (requestsForTag!=null) // If no table exists for the reqTag
      return requestsForTag.containsKey(req);
    else
      return false;
  }

  private boolean matchExpiredRequest(IDdfDataFrame ddfReceiveFrame)
  {
    BSimple frameId = ddfReceiveFrame.getFrameTag();                      // Gets the message tag for this IDdfDataFrame
    Hashtable<BIDdfRequest, BIDdfRequest> requestsForSameTag = expiredRequestTags.get(frameId);       // Gets any expired requests that share the same tag
    boolean frameConsumed=false;
    
    if (requestsForSameTag!=null)                                                   // If there are any
    {
      Enumeration<BIDdfRequest> requests = requestsForSameTag.keys();                             // Loops through them
      while (requests.hasMoreElements())
      {
        BIDdfRequest ddfExpiredRequest = requests.nextElement();
        BIDdfResponse ddfRsp = expiredFrameReceived(ddfReceiveFrame,ddfExpiredRequest);                  // Calls expiredFrameReceived for each request passing the frame
        if (ddfRsp!=null)
        {
          frameConsumed=true;
          if (ddfRsp instanceof IDdfTransmitAckResponse)
          {
            transmitRspAckBytes((IDdfTransmitAckResponse)ddfRsp);
          }

          // NOTE: We do not care about BIDdfMultiFrameResponse's in this method, although
          // We checked for them in the matchOutstandingRequest(IDdfDataFrame) method.
          // That is because the request is no longer outstanding at this point. Instead,
          // It is long expired! Therefore, we do not need to reSchedule it for response
          // Timeout checking.
        }
      }
    }
    return frameConsumed;
  }

  private boolean matchOutstandingRequest(IDdfDataFrame ddfReceiveFrame)
  {
    BSimple frameTag = ddfReceiveFrame.getFrameTag();                      // Gets the tag for this IDdfDataFrame
    Hashtable<BIDdfRequest, BIDdfRequest> requestsForSameTag = outstandingRequestTags.get(frameTag);   // Gets any outstanding messages that share the same tag
    boolean frameConsumed=false;
    
    if (requestsForSameTag!=null)                                                   // If there are any
    {
      Enumeration<BIDdfRequest> requests = requestsForSameTag.keys();                             // Loops through them
      while (requests.hasMoreElements())
      {
        BIDdfRequest ddfRequest = requests.nextElement();
        synchronized(ddfRequest) // Locks the ddfRequest so that if its Ticket expires, then
        { 
          // The checkOutstandingTimeout action will see that the request received its response,
          // Albeit -- at the very last possible millisecond! Please note that the doCheckOutstandingTimeout
          // Method of BDdfTransactionManager likewise locks on the request.
          final BIDdfResponse ddfRsp = frameReceived(ddfReceiveFrame,ddfRequest);                            // Call ddfFrameRecieved for each request passing the frame
          if (ddfRsp!=null)
          {
            frameConsumed=true;
            if (ddfRsp instanceof IDdfTransmitAckResponse)
            {
              transmitRspAckBytes((IDdfTransmitAckResponse)ddfRsp);
            }
            if (ddfRsp instanceof BIDdfMultiFrameResponse)
            { 
              // If a multiframe response was received, then this cancels the current
              // response-timeout Ticket and schedules a new one for the request. This
              // Causes the framework to wait for a fresh responseTimeout interval for
              // The next frame of the BIDdfMultiFrameResponse before retrying the transaction
              // Or possibly timing out.
              reScheduleToCheckForTimeout(ddfRequest);
            }
          }
        }
      }
    }
    return frameConsumed;
  }
  

  private void updateCounts()
  {
    setOutstandingCount(countIdentifiableRequests(outstandingRequestTags));
    setExpiredCount(countIdentifiableRequests(expiredRequestTags));
  }
  

  /**
   * This method is called on every outstanding ddfReceiveFrame for each raw data frame that is
   * received and has a matching tag.
   *
   * @param outstandingRequest
   * @param ddfResponse
   */
  protected BIDdfResponse expiredFrameReceived(IDdfDataFrame ddfReceiveFrame, BIDdfRequest expiredRequest)
  {
    try
    {
      BIDdfResponse ddfResponse = expiredRequest.processReceive(ddfReceiveFrame);
      
      if (ddfResponse!=null && isCompletedResponse(ddfResponse))
        expiredFrameReceived(expiredRequest,ddfResponse);
      
      return ddfResponse;
    }
    catch (DdfResponseException ere)
    {
      expiredResponseError(expiredRequest, ere);
      
      return NACK_RESPONSE;
    }
    catch (Exception e)
    {
      expiredResponseError(expiredRequest, new DdfResponseException(e));
      
      return NACK_RESPONSE;
    }
  }

  /**
   * This method is called on every outstanding ddfReceiveFrame for each raw data frame that is
   * received and has a matching tag.
   *
   * @return the BIDdfResponse from the outstanding request, if the outstanding request
   * processes the frame and returns a BIDdfResponse. Returns null otherwise.
   */
  protected BIDdfResponse frameReceived(IDdfDataFrame ddfReceiveFrame, BIDdfRequest outstandingRequest)
  {
    try
    {
      BIDdfResponse ddfResponse = DdfRequestUtil.processReceive(outstandingRequest,ddfReceiveFrame);
      
      if (ddfResponse!=null && isCompletedResponse(ddfResponse))
        frameReceived(outstandingRequest,ddfResponse);
      
      return ddfResponse;
    }
    catch (DdfResponseException ere)
    {
      if (ere instanceof IDdfTransmitAckResponse)
      {
        transmitRspAckBytes((IDdfTransmitAckResponse)ere);
      } 
      
      outstandingResponseError(outstandingRequest, ere);
      return NACK_RESPONSE;
    }
    catch (Exception e)
    {
      outstandingResponseError(outstandingRequest, new DdfResponseException(e));
      return NACK_RESPONSE;
    }
  }
  
  /**
   * This method is called when an outstanding data frame matches up as the response to a BIDdfRequest.
   *
   * @param outstandingRequest
   * @param ddfResponse
   */
  protected void frameReceived(BIDdfRequest outstandingRequest, BIDdfResponse ddfResponse)
  {
    try
    {
      if (getDdfCommunicator().getLog().isTraceOn())
        getDdfCommunicator().getLog().trace(DdfDefaultCommLexicon.receivedResponseForRequest(outstandingRequest));
      
      removeIdentifiableRequest(outstandingRequest, outstandingRequestTags);
      DdfRequestUtil.processResponse(outstandingRequest, ddfResponse);
    }
    finally
    {
      updateCounts();
    }
  }

  protected void outstandingResponseError(BIDdfRequest outstandingRequest, DdfResponseException errorResponse)
  {
    try
    {
      if (getDdfCommunicator().getLog().isTraceOn())
      {
        getDdfCommunicator().getLog().trace(
            DdfDefaultCommLexicon.receivedErrorResponseForRequest(
                outstandingRequest,errorResponse), errorResponse);
      }
      
      removeIdentifiableRequest(outstandingRequest, outstandingRequestTags);
      DdfRequestUtil.processErrorResponse(outstandingRequest, errorResponse);
    }
    finally
    {
      updateCounts();
    }

  }

  protected void expiredResponseError(BIDdfRequest expiredRequest, DdfResponseException errorResponse)
  {
    try
    {
      if (getDdfCommunicator().getLog().isTraceOn())
      {
        getDdfCommunicator().getLog().trace(
            DdfDefaultCommLexicon.receivedLateResponseForRequest(expiredRequest));
        
        getDdfCommunicator().getLog().trace(
            DdfDefaultCommLexicon.receivedErrorResponseForRequest(
                expiredRequest,errorResponse));
      }
      removeIdentifiableRequest(expiredRequest, expiredRequestTags);
      DdfRequestUtil.processErrorResponse(expiredRequest, errorResponse);
    }
    finally
    {
      updateCounts();
    }
  }

  /**
   * This method is called when an outstanding data frame matches up as the response to a previously timed-out BIDdfRequest.
   *
   * @param outstandingRequest
   * @param ddfResponse
   */
  protected void expiredFrameReceived(BIDdfRequest expiredRequest, BIDdfResponse ddfResponse)
  {
    try
    {
      if (getDdfCommunicator().getLog().isTraceOn())
        getDdfCommunicator().getLog().trace(DdfDefaultCommLexicon.receivedLateResponseForRequest(expiredRequest));
      
      removeIdentifiableRequest(expiredRequest, expiredRequestTags);
      DdfRequestUtil.processLateResponse(expiredRequest,ddfResponse);
    }
    finally
    {
      updateCounts();
    }
  }
  
////////////////////////////////////////////////////////////////
// Attributes
////////////////////////////////////////////////////////////////

  /**
   * This maps the tags of outstanding requests (requests that have
   * been transmitted but have not yet timed out) each to a vector
   * of the corresponding requests for the tag. This is necessary
   * since this is a multiple transaction manager, it is quite
   * possible that the driver could attempt to have more than one
   * outstanding request for the same request tag.
   */
  Hashtable<BSimple, Hashtable<BIDdfRequest, BIDdfRequest>> outstandingRequestTags = new Hashtable<>();
  
  /**
   * This is just like the outstandingRequestTags hashtable except
   * that tags are mapped to their outstanding requests in this
   * table after they time-out. This table allows for the multiple
   * transaction manager to process late responses. This maximum
   * size of this hashtable is governed by a property on this
   * object.
   */
  Hashtable<BSimple, Hashtable<BIDdfRequest, BIDdfRequest>> expiredRequestTags = new Hashtable<>();
  
  /**
   * This Queue is used in addition to the expiredRequestTags hashtable.
   * This Queue is used to allow the source code for this class to easy
   * count the number of expired requests in order to enforce the
   * maximum size, as governed by a property on this object.
   */
  Queue expiredRequestTagsFifoHelper = new Queue();
  
  // The NACK_RESPONSE is a special case that let's us treat a request as though it
  // Got a response, even though it didn't get a successful response. In this case
  // It got either a NACK or there was an java exception processing the response. My
  // Thinking here is that in either case, we do not want to route the particular
  // Receive frame to the unsolicited handler. In either case, the response clearly
  // Cared about the frame, even though the response probably detected something about
  // The frame that indicated (according to the particular driver's protocol) a less
  // Than ideal response.
  private static final BDdfResponse NACK_RESPONSE = new BDdfResponse();
}
