/*
 * @copyright 2005 Tridium Inc.
 */
package com.tridium.ddf.comm.defaultComm;

import javax.baja.util.Lexicon;

import com.tridium.ddf.comm.req.BIDdfRequest;
import com.tridium.ddf.comm.rsp.DdfResponseException;
import com.tridium.ddf.comm.rsp.IDdfTransmitAckResponse;

/**
 * This class contains methods and attributes that help clean up other areas
 * of the source code that are used to display localized text.
 * 
 * @author lperkins
 */
public class DdfDefaultCommLexicon
{
  private DdfDefaultCommLexicon(){} // Access provided through public static methods
  
  public static final Lexicon LEX = Lexicon.make(DdfDefaultCommLexicon.class);
  
  
  
  
  // This message is placed in various error logs and sometimes stdout
  // If a request receives a response indicating that an error situation
  // Exists in the device that prevents it from responding successfully
  // The text in parameter {0} describes the response error.
  // ResponseError = Response Error
  public static final String RESPONSE_ERROR = "ResponseError";
  
  public static final String responseError(DdfResponseException ere){
    return LEX.getText(RESPONSE_ERROR, new Object[]{
        ere.getLocalizedMessage()});
  }
  
  
  
  // This message is placed in various error logs and sometimes stdout
  // If a request receives a response but the developer's parsing logic
  // Throws an exception while parsing the value for a particular data point.
  // The text in parameter {0} describes the response error.
  // ParseError = Parse Error
  public static final String PARSE_ERROR = "ParseError";
  
  public static final String parseError(Exception e){
    return LEX.getText(PARSE_ERROR, new Object[]{
        e.getLocalizedMessage()});
  }
  
  
  
  
  // This text is placed into standard output and into the driver's communicator's log
  // If trace is enabled at the start of any field-bus request-response transaction
  // BeginTransaction = Begin Transaction
  public static final String BEGIN_TRANSACTION = "BeginTransaction";
  
  public static final String beginTransaction(BIDdfRequest ddfReq){
    return LEX.getText(BEGIN_TRANSACTION, new Object[]{Integer.toString(ddfReq.hashCode())});
  }

  
  
  // This text is placed into standard output and into the driver's communicator's log
  // If trace is enabled at the start of any field-bus request-response transaction if
  // The response timeout is zero. This essentially means the transaction is a send-only
  // Transaction
  // SendOnlyTransactionNotification = NOTE: This is a send-only transaction.
  public static final String SEND_ONLY_TRANSACTION_NOTIFICATION = "SendOnlyTransactionNotification";
  
  public static final String sendOnlyTransactionNotification = LEX.getText(SEND_ONLY_TRANSACTION_NOTIFICATION);
  
  
  
  // This text is placed into standard output and into driver's communicator's log if
  // Trace is enabled, whenever a request times out. The value in parameter {0} is a
  // Unique integer identifying the request.
  // RequestTimedOut = Request {0} Timed Out
  public static final String REQUEST_TIMED_OUT = "RequestTimedOut";
  
  public static String requestTimedOut(BIDdfRequest ddfReq){
    return LEX.getText(REQUEST_TIMED_OUT,
        new Object[]{Integer.toString(ddfReq.hashCode())});
  }
  
  
  
  // This text is placed into various faultCause properties whenever a request times out.
  // Unique integer identifying the request.
  // RequestTimeOut = Request Timeout
  public static final String REQUEST_TIME_OUT = "RequestTimeOut";
  
  public static final String requestTimeOut = LEX.getText(REQUEST_TIME_OUT);
  
  
  
  // This text is placed into standard output and into driver's communicator's log if
  // Trace is enabled, whenever a request is received for a request. A summary of the request
  // Will be placed into paramater {0}.
  // ReceivedResponseForRequest = Received Response for Request: {0}
  public static final String RECEIVED_RESPONSE_FOR_REQEUST = "ReceivedResponseForRequest";
  
  public static final String receivedResponseForRequest(BIDdfRequest req)
  {
    return LEX.getText(RECEIVED_RESPONSE_FOR_REQEUST,new Object[]{
        Integer.toString(req.hashCode())});
  }
  
  // This text is placed into standard output and into driver's communicator's log if
  // Trace is enabled, whenever a response is received for a request after the request
  // Has timed-out.  A summary of the request
  // Will be placed into paramater {0}. This lexicon key is not applicable to all drivers.
  // ReceivedLateResponseForRequest = Received Late Response for Request: {0}
  public static final String RECEIVED_LATE_RESPONSE_FOR_REQEUST = "ReceivedLateResponseForRequest";
  public static final String receivedLateResponseForRequest(BIDdfRequest req)
  {
    return LEX.getText(RECEIVED_LATE_RESPONSE_FOR_REQEUST,new Object[]{
        Integer.toString(req.hashCode())});
  }
  
  // This text is placed into standard output and into driver's communicator's log if
  // Trace is enabled, whenever a response is received for a request but the response
  // Indicates an error condition in the device that is preventing a successful rsponse.
  // A summary of the request Will be placed into paramater {0}. A summary of the error
  // will be placed in parameter {1}.
  // ReceivedErrorResponseForRequest = Received Error Response for Request: {0}
  public static final String RECEIVED_ERROR_RESPONSE_FOR_REQEUST = "ReceivedErrorResponseForRequest";
  public static final String receivedErrorResponseForRequest(BIDdfRequest req, DdfResponseException ere)
  {
    return LEX.getText(RECEIVED_ERROR_RESPONSE_FOR_REQEUST,
        new Object[]{Integer.toString(req.hashCode()),ere.getLocalizedMessage()});
  }

  
  // UnsupportedStringFormat = Unrecognized syntax "{1}" for point "{0}" 
  public static final String UNSUPPORTED_STRING_FORMAT = "UnsupportedStringFormat";
  public static String unsupportedStringFormat(String point, String value){
    return LEX.getText(UNSUPPORTED_STRING_FORMAT, new Object[]{point,value});
  }

  public static final String ACK_REPLY_TRANSMIT = "AckReplyTransmit";
  public static String ackReplyTransmit(IDdfTransmitAckResponse ackResponse){
    return LEX.getText(ACK_REPLY_TRANSMIT);
  }
  
  
  public static final String ACK_REPLY_TRANSMIT_ERROR = "AckReplyTransmitError";
  public static String ackReplyTransmitError(IDdfTransmitAckResponse ackResponse){
    return LEX.getText(ACK_REPLY_TRANSMIT_ERROR);
  }
  
  // MutTagCannotBeNull = The request tag used in a multiple transaction communicator cannot be null. Please
  // Fix {0}.getTag() and make it return something other than null. Even BString.DEFAULT
  // Would be better than null.
  public static final String MUT_TAG_CANNOT_BE_NULL = "MutTagCannotBeNull";
  public static String mutTagCannotBeNull(BIDdfRequest requestWithNullTag){
    return LEX.getText(MUT_TAG_CANNOT_BE_NULL, new Object[]{requestWithNullTag.getType().getTypeSpec()});
  }
  
  // ReceiveError = An unexpected error occured while trying to receive a data frame.
  public static final String RECEIVE_ERROR = "ReceiveError";
  public static final String receiveError = LEX.getText(RECEIVE_ERROR);
  
  // This text is placed in that station trace log if the driver's receiver thread receives a java interrupt. It is normal for this to happen
  // If the station is shutting down (although an interrupt does not have to occur while the station is shuttin down, one will occur if java
  // Thinks that it needs to get the receive thread's attention). The text passed to parameter {0} is the name of the interrupted thread.
  // Interrupted = Thread {0} Interrupted
  public static final String INTERRUPTED = "Interrupted";
  public static String interrupted(){
    return LEX.getText(INTERRUPTED, new Object[]{Thread.currentThread().getName()});
  }
  
  // This text is placed into the station's standard output if the user attempts to place an instance of BDdfRawTransmitRequest onto
  // The communicator. That is forbidden. Directly using the BDdfRawTransmitRequest is forbidden.
  // InternalDdfTransmitActRequest = "This is an internal BDdfRawTransmitRequest. It should only be forced-transmitted and not placed into the communicator normally.";
  public static final String INTERNAL_DEV_TRANSMIT_ACK_REQUEST_KEY = "InternalDdfTransmitActRequest";
  public static final String internalDdfTransmitAckRequest = LEX.getText("internalDdfTransmitActRequest");
  
  public static final String NO_COMM_AVAILABLE_WHILE_INVALID_STATUS = "NoCommunicationsWhileInvalidStatus";
  public static final String noCommunicationsAvailableWhileInvaldStatus = LEX.getText(NO_COMM_AVAILABLE_WHILE_INVALID_STATUS);
  
}
