/*
 * @copyright 2005 Tridium Inc.
 */
package com.tridium.ddf.clock;

import java.util.LinkedList;
import java.util.ListIterator;

import javax.baja.sys.Action;
import javax.baja.sys.BComponent;
import javax.baja.sys.BRelTime;
import javax.baja.sys.BSingleton;
import javax.baja.sys.BValue;
import javax.baja.sys.Clock;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import javax.baja.sys.Clock.Ticket;


/**
 * This class is a BSingleton whose INSTANCE provides a subset
 * of the functionality of that standard Niagara AX Clock.schedule.
 * 
 * The benefit of using this INSTANCE instead of Clock.schedule
 * is that this INSTANCE uses its own thread and reduces the
 * possibility of applications or appliances scheduling any actions
 * on the same thread that the Ddf transaction manager relies
 * heavily upon.
 * 
 * @author lperkins
 *
 */
public class BDdfScheduler
  extends BSingleton
{
  
////////////////////////////////////////////////////////////////
// Private constructor only - Access via BDdfScheduler.INSTANCE 
////////////////////////////////////////////////////////////////
  /**
   * This class is a BSingleton. Only one instance exists. To gain access to it, use
   * BDdfScheduler.INSTANCE.
   */
  private BDdfScheduler()
  { 
    // This linked list will store the outstanding items that this instance is
    // Scheduling. The 'schedule' method places items into the list sorted (ascending) by
    // Each one's 'long' tick count representing the prefered moment for the item
    // To be processed. This is an 'insertion sort' alorithm whereby the list
    // Is sorted because we always insert in sorted order. 
    outstandingTickets = new LinkedList<>();

    // This thread checks the outstandingTickets periodically and processes those
    // Whose tick count has past and therefore need processed
    ddfSchedulerThread = new DdfSchedulerThread();
    
    ddfSchedulerThread.start();
  }

////////////////////////////////////////////////////////////////
// DdfSchedulerThread API 
////////////////////////////////////////////////////////////////
  
  /**
   * This is convenience method that calls the other 'schedule' method that takes a long, relMillis
   * as an argument
   * 
   * @param host a component on which to invoke an action in some amount of time
   * 
   * @param relTime the BRelTime amount of time in which to invoke the action
   * 
   * @param action the action to invoke on the component
   * 
   * @param arg the argument to pass to action that will be invoked on the component (null is acceptable
   * if the action takes no arguments)
   * 
   * 
   * @return the Ticket for the scheduled item
   */
  public synchronized Ticket schedule(BComponent host, BRelTime relTime, Action action, BValue arg)
  {
    return schedule(host,relTime.getMillis(),action,arg);
  }
  
  /**
   * This schedules an action to be invoked on a given host after a relative amount of time has passed.
   * 
   * The action is invoked on the INSTANCE's own scheduler thread and is therefore, safe from other
   * developers that might otherwise call the standard Niagara AX Clock.schedule method.
   * 
   * The caller needs to ensure that the action is implemented to return quickly. Any of the action's
   * work needs to be performed on another thread. For example, the BDdfTransactionMgr does this by placing a
   * work item onto its own transaction processing thread. Its own transaction processing thread
   * performs the real work.
   *  
   * @param host a component on which to invoke an action in some amount of time
   * 
   * @param relTime the number of milliseconds from now, in which to invoke the action
   * 
   * @param action the action to invoke on the component
   * 
   * @param arg the argument to pass to action that will be invoked on the component (null is acceptable
   * if the action takes no arguments)
   * 
   * 
   * @return the Ticket for the scheduled item
   */
  public synchronized Ticket schedule(BComponent host, long relMillis, Action action, BValue arg)
  {
    DdfSchedulerTicket newTicket = new DdfSchedulerTicket(host,action,arg,Clock.ticks() + relMillis);
    
    // The 'insertLocation' will determine where in the 'oustandingTickets' this new ticket
    // Will be placed
    int insertLocation = -1;
    
    int numTickets = outstandingTickets.size();
    
    // If there are any outstanding tickets
    if (numTickets>0)
    { 
      // Loops through the outstanding tickets until the end of list or until a suitable insert location is found
      ListIterator<DdfSchedulerTicket> walkTickets = outstandingTickets.listIterator();
      while (walkTickets.hasNext() && insertLocation==-1)
      { 
        // Gets the next ticket of the iteration
        int nextIndex = walkTickets.nextIndex(); // Gets the index of the subsequent call to 'next'
        DdfSchedulerTicket nextTicket = walkTickets.next();
        
        // If an outstanding ticket's 'serviceTicks' is after the newTicket's 'serviceTicks'
        // Then the newTicket needs to be inserted just before the outstanding ticket in order to
        // Maintain a sorted list of outstanding tickets. I do a '>' instead of '>=' because I'm
        // Thinking that if two tickets have 'equal' times then the first one to have been
        // Scheduled should prevail
        if (nextTicket.serviceTicks>newTicket.serviceTicks)
          insertLocation = nextIndex; // NOTE: By assigning a value to 'insertLocation', the loop will naturally terminate when it evaluates its condition
      }
    }
    
    // If the 'newTicket' is not to be inserted into the list then this appends it to the list
    if (insertLocation==-1)
      outstandingTickets.addLast(newTicket);
    else
    // Else, the 'newTicket' gets inserted into its proper location to maintain an ascending, sorted list (sorted on serviceTicks)
      outstandingTickets.add(insertLocation,newTicket);
    
    //  Wakes up the 'outerRun' method to force it to 'check' and schedule its next check, in light of the 'newTicket'
    notifyAll(); 
    
    return newTicket;
  }
  
  /**
   * This method is called by the DdfSchedulerTicket.cancel method. This method
   * removes the ticket from the outstanding tickets.
   * 
   * Access is restricted to 'protected' access. To cancel a ticket call
   * the 'cancel' method on the ticket itself. The ticket's 'cancel' method calls
   * this method to release the ticket's resources to the garbage collector.
   *  
   * @param ticketToCancel
   */
  protected synchronized void cancelTicket(DdfSchedulerTicket ticketToCancel)
  {
    outstandingTickets.remove(ticketToCancel);
  }  
  
  /**
   * Invokes the action on all tickets whose time has come, or passed, to be processed. All such
   * tickets are removed from the internal 'outstandingTickets' list.
   */
  protected synchronized void check()
  {
    long clockTicks = Clock.ticks();
    
    // Loops through the outstanding tickets until the end of list or until a suitable insert location is found
    ListIterator<DdfSchedulerTicket> walkTickets = outstandingTickets.listIterator();
    
    while (walkTickets.hasNext())
    { 
      // NOTE: The outstandingTickets list is 'insertion-sorted' ascending
      // On the 'timeWhenScheduledMillis + relMillis' of each ticket
      // Therefore, this loop 'breaks' as soon as it finds a
      // Ticket whose service ticks are after the clockTicks. Once
      // Such a ticket is located, we can conclude that all subsequent
      // Tickets will have an even greater serviceTicks
      DdfSchedulerTicket nextTicket = walkTickets.next();
      
      if (nextTicket.serviceTicks<=clockTicks)
      {
        walkTickets.remove();
        nextTicket.invokeAction(); // NOTE: The developer needs to ensure that the action returns quickly. Any work needs to be pawned off onto another thread. 
      }
      else
        break;
    }
  }
  
  /**
   * This method is called from the 'run' method of the BDdfScheduler's
   * own thread.
   * 
   * @throws InterruptedException
   */
  protected void outerRun()
    throws InterruptedException
  {
    while (true)
    { 
      try
      {
        synchronized(this)
        { 
          // If there are any outstanding tickets
          if (outstandingTickets.size()>0)
          { 
            // NOTE: The 'outstandingTickets' are sorted ascending on serviceTicks, therefore, the
            // First item is the ticket that should be processed next. This gets the first item in the
            // 'outstandingTickets' list.
            DdfSchedulerTicket earliestTicket = outstandingTickets.getFirst();
            
            long ticksBeforeWaiting = Clock.ticks();
            
            // If the ticket to process next needs to be processed at some time in the future
            if (earliestTicket.serviceTicks>ticksBeforeWaiting)
            { 
              // Computes the shortest time interval to wait before the
              // Next outstanding ticket needs to be serviced.  
              long shortestWait = earliestTicket.serviceTicks - ticksBeforeWaiting;
              
              if (shortestWait<100) // Cannot safely schedule for less than 100 millis in Java
                shortestWait = 100;
              
              wait(shortestWait);
            }
            // else, the earliestTicket.serviceTicks has past, this falls through and calls 'check()', which invokes the action on all expired tickets
          }
          else // Else, there are no outstanding tickets
          {    // So this waits indefinitely for one
            wait();
          }
          
          // Invokes the corresponding action on any tickets whose
          // serviceTime has passed
          check();
        }
      }
      catch (InterruptedException ie)
      { 
        // Rethrows the interrupted exception so that the thread stops.
        // The BDdfScheduler does not interrupt itself, therefore if this happens
        // Then that probably means the JVM is trying to shut down
        throw ie;
      }
      catch (Exception e)
      {
        System.err.println("BDdfScheduler Detected Exception: "+e);
        e.printStackTrace();
      }
    }
  }
  
////////////////////////////////////////////////////////////////
// DdfSchedulerThread
////////////////////////////////////////////////////////////////
  
  /**
   * This class implements the BDdfScheduler's own thread of execution.
   * @author lperkins
   *
   */
  public class DdfSchedulerThread
    extends Thread
  {
    DdfSchedulerThread()
    {
      super("Ddf:Scheduler");   
    }
    /**
     * This method is called on its own thread of execution. It
     * calls the outerRun() method on the outer instance.
     */
    public void run()
    {
      try
      {
        outerRun();
      }
      catch(InterruptedException ie)
      {
      }
    }
  }
  
////////////////////////////////////////////////////////////////
// Attributes
////////////////////////////////////////////////////////////////
  
  // Kicks off a DdfSchedulerThread when this class is instantiated
  private final DdfSchedulerThread ddfSchedulerThread;
  
  // Stores the outstanding tickets in a Linked List for efficient insertion
  private final LinkedList<DdfSchedulerTicket> outstandingTickets;
  
  /**
   * This is the BSingleton INSTANCE.
   */
  public static final BDdfScheduler INSTANCE = new BDdfScheduler();

////////////////////////////////////////////////////////////////
// Type
////////////////////////////////////////////////////////////////
  
  public Type getType() { return TYPE; }
  public static final Type TYPE = Sys.loadType(BDdfScheduler.class);
}
