/**
 * Copyright 2005 Tridium, Inc. All Rights Reserved.
 */
package com.tridium.basicdriver.util;


import javax.baja.sys.BComponent;
import javax.baja.sys.Flags;
import javax.baja.sys.Property;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import javax.baja.util.Queue;
import com.tridium.basicdriver.BBasicNetwork;
import com.tridium.basicdriver.UnsolicitedMessageListener;
import com.tridium.basicdriver.message.ReceivedMessage;

/**
 * The receive handler for processing unsolicited received messages.
 * By default, this handler is not "hooked-up" in basic driver. Not
 * all drivers will need to process unsolicited messages. However,
 * those that do, will need to process them in a customized fashion.
 * Therefore, this class cannot quite do all of the work. However, it
 * does handle queueing them up and processing them on its own thread.
 *
 * The driver developer needs to follow these steps to hook this into
 * their driver:
 *
 * 1. Make a class that extends this class.
 *    - Descendant class name should start with a 'B'
 *    - Descendant class needs declared in driver's module-include.xml
 *    - Implement the method processUnsolicitedMessage(ReceivedMessage)
 *      and handle the unsolicited messages in a manner that satisfies
 *      the requirements of the driver.
 *
 * 2. Add a slot to the driver's network whose slot type is the same as
 *    the descendant class created in step 1.
 *    - Since this is a BComponent, it already gets the started and stopped
 *      callbacks. No further set-up is required to "hook it up."
 *
 * 3. That's it!
 *
 * @author    Andy Saunders (Original R2 code)
 * @creation  29 Feb 00
 * @author    Lenard Perkins (Moved Into BasicDriver)
 * @creation  04 Apr 05
 * @version   $Revision: 1$ $Date: 11/13/02 12:47:14 PM$
 * @since     Niagara 3.0 andi 1.0
 */
public abstract class BBasicUnsolicitedReceive
  extends BComponent
  implements Runnable, UnsolicitedMessageListener
{
  /*-
  class BBasicUnsolicitedReceive
  {
    properties
    {
      unsolicitedMessageCount : int
        -- This is the number of unsolicited messages that have been received by the driver since station startup
        flags {transient,readonly}
        default{[0]}
    }

  }
  -*/
/*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
/*@ $com.tridium.basicdriver.util.BBasicUnsolicitedReceive(3450892612)1.0$ @*/
/* Generated Thu Feb 14 10:05:07 GMT-05:00 2008 by Slot-o-Matic 2000 (c) Tridium, Inc. 2000 */

////////////////////////////////////////////////////////////////
// Property "unsolicitedMessageCount"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>unsolicitedMessageCount</code> property.
   * This is the number of unsolicited messages that have
   * been received by the driver since station startup
   * @see com.tridium.basicdriver.util.BBasicUnsolicitedReceive#getUnsolicitedMessageCount
   * @see com.tridium.basicdriver.util.BBasicUnsolicitedReceive#setUnsolicitedMessageCount
   */
  public static final Property unsolicitedMessageCount = newProperty(Flags.TRANSIENT|Flags.READONLY, 0,null);
  
  /**
   * Get the <code>unsolicitedMessageCount</code> property.
   * @see com.tridium.basicdriver.util.BBasicUnsolicitedReceive#unsolicitedMessageCount
   */
  public int getUnsolicitedMessageCount() { return getInt(unsolicitedMessageCount); }
  
  /**
   * Set the <code>unsolicitedMessageCount</code> property.
   * @see com.tridium.basicdriver.util.BBasicUnsolicitedReceive#unsolicitedMessageCount
   */
  public void setUnsolicitedMessageCount(int v) { setInt(unsolicitedMessageCount,v,null); }

////////////////////////////////////////////////////////////////
// Type
////////////////////////////////////////////////////////////////
  
  public Type getType() { return TYPE; }
  public static final Type TYPE = Sys.loadType(BBasicUnsolicitedReceive.class);

/*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/


  /**
   * Callback from BComponenet
   */
  public void started()
    throws Exception
  {
    super.started();
    network = (BBasicNetwork)getParent();
    network.getComm().registerListener(this); // Registers to the driver's Comm handler to receive unsolicited messages from the receive handler
    timeToDie = false;
    unsolicitedMessageManager = new Queue();
    myThread = new Thread(this, getName()+':'+getParent().getName());
    myThread.start();
  }

  /**
   * Callback from BComponenet
   */
  public void stopped()
    throws Exception
  {
    super.stopped();
    network.getComm().unregisterListener(this);
    timeToDie = true;
    myThread.interrupt();
  }

  /**
   * Callback from BComponenet
   */
  public boolean isParentLegal(BComponent parent)
  {
    return parent.getType().is(BBasicNetwork.TYPE);
  }

////////////////////////////////////////////////////////////////
//  Required run method
////////////////////////////////////////////////////////////////
  public void run()
  {
    ReceivedMessage unsolicitedMsg;

    while(!timeToDie)
    {
      try
      { // Pulls next message off of Queue, passing -1 causes it to block if there is none
        unsolicitedMsg = (ReceivedMessage)unsolicitedMessageManager.dequeue(-1);
      }
      catch( InterruptedException ie)
      {
        unsolicitedMsg = null;
        try{Thread.sleep(1000);}catch(Exception e){} // Prevents spinning wheels
      }

      if ( unsolicitedMsg != null)
      {
        setUnsolicitedMessageCount( getUnsolicitedMessageCount()+1 );
        if (network.getLog().isTraceOn())
        {
          network.getLog().trace("Received unsolicited message "+unsolicitedMsg);
        }
        // Calls customized method to process the unsolicited message
        try
        {
          processUnsolicitedMessage(unsolicitedMsg);
        }
        catch (Exception e)
        {
          network.getLog().error(myThread.getName(), e);
        }
      }
    }
  }

 /**
  *  Process the unsolicited message. This must be overridden
  * by subclasses in order to customize the behavior.
  *
  *  @param  rcvMessage incoming SerialDriver message
  */
  protected abstract void processUnsolicitedMessage( ReceivedMessage unsolicitedMsg );

  /** Returns an unsolicited listener code object associated with
   * this listener to use for matching unsolicited received messages
   * that should be routed to this listener.
   *
   * This method is called by BasicDriver's Comm object. The hash code of the return
   * value from this method should match the return code from the corresponding method
   * on the driver's ReceivedMessage.
   *
   * BasicDriver is allowing for different handlers to process diffrerent
   * types of unsolicited messages. Therefore, in theory, the driver could
   * have multiple BBasicUnsolicitedReceive components under the network. The
   * Comm handler will route an unsolicited ReceivedMessage to the handler that
   * returns the same unsolicited listener code.
   */
  public Object getUnsolicitedListenerCode()
  { // Reference in Comm.java; lets ReceivedMessage tell Comm to use BBasicUnsolicitedReceive as the default unsolicited receive handler
    return BBasicUnsolicitedReceive.TYPE;
  }

 /**
  * Access point called by the BasicDriver Comm API to place
  * unsolicited messages on the UnsolicitedMessages queue.
  *
  * @param  ReceivedMessage incoming BasicDriver received message
  */
  public void receiveMessage( ReceivedMessage message)
  {
    unsolicitedMessageManager.enqueue( message);
  }

  BBasicNetwork network =null;
  private Queue  unsolicitedMessageManager = null;
  private boolean timeToDie = true;
  private Thread myThread = null;
} // end of class BBasicUnsolicitedReceive
