/*
 * Copyright 2015 Tridium, Inc. All Rights Reserved.
 */
package com.tridium.basicdriver.util;

import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Vector;
import javax.baja.driver.util.BIPollable;
import javax.baja.driver.util.BPollFrequency;
import javax.baja.nre.util.IntHashMap;
import javax.baja.sys.BStruct;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import com.tridium.basicdriver.BBasicNetwork;
import com.tridium.basicdriver.point.BBasicProxyExt;

/**
 * The BBasicPollGroup class is used to coalesce the polling of proxy extensions
 * whose values can all be retrieved in a single poll message. A driver can
 * extend BBasicPollGroup and implement a poll method that polls the data for a
 * group of proxy extensions. Then in the driver's proxy extension (extending
 * BBasicProxyExt), it should override the appropriate method that returns
 * the Niagara AX type of the cusomized BBasicPollGroup for that proxy. All proxies
 * that share the same BBasicPollGroup instance should return the same poll group
 * code Object in the appropriate overridden method.
 */
public abstract class BBasicPollGroup
  extends BStruct
  implements BIBasicPollable
{
  /*-
  class BBasicPollGroup
  {
  }
  -*/
/*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
/*@ $com.tridium.basicdriver.util.BBasicPollGroup(2318694231)1.0$ @*/
/* Generated Mon Jun 06 09:42:50 EDT 2005 by Slot-o-Matic 2000 (c) Tridium, Inc. 2000 */

////////////////////////////////////////////////////////////////
// Type
////////////////////////////////////////////////////////////////

  public Type getType() { return TYPE; }
  public static final Type TYPE = Sys.loadType(BBasicPollGroup.class);

/*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/


  /**
   * Baja says we have to allow an empty constructor.
   */
  public BBasicPollGroup()
  {
    this.code=null;
    mySubscribedProxies = new Vector<>(8);
    myProxies = new Vector<>(8);
    subscribed = false;
  }

  /**
   * Gets the poll group code that is unique amongst
   * all of the proxies that are part of the poll
   * group. This code will be equivalent to that which
   * is returned by the getPollGroupCode method on each
   * of the proxies that are in the group. The driver
   * developer should choose this code carefully, and
   * use it to help the descendant class of this PollGroup
   * to know specifically what to poll for.
   */
  public Object getCode()
  {
    return code;
  }

  /**
   * The descendant class may call this method to get a list of the
   * proxies for this poll group.
   */
  protected final BBasicProxyExt[] getProxyExts()
  {
    BBasicProxyExt[] pexts = new BBasicProxyExt[myProxies.size()];
    myProxies.copyInto(pexts);
    return pexts;
  }

  /**
   * The descendant class may call this method to get a list of the
   * proxies for this poll group that are presently subscribed for
   * polling.
   */
  protected final BBasicProxyExt[] getSubscribedProxyExts()
  {
    BBasicProxyExt[] pexts = new BBasicProxyExt[mySubscribedProxies.size()];
    mySubscribedProxies.copyInto(pexts);
    return pexts;
  }

  /**
   *  Register a proxy point as a point interested in data from this group.
   *  Any points in this list will be updated on the poll cycle.
   *
   * This method should be called from the proxy's started method.
   */
  public void registerProxy(BBasicProxyExt proxy)
  {
    if(!myProxies.contains(proxy))
      myProxies.add(proxy);
  }

  /**
   * Remove a proxy point as a point interested in data from a group.
   * Any points in this list will be updated on the poll cycle.
   *
   * If the given proxy was the last proxy registered to this group then
   * the group itself will unsubsribe from the poll scheduler
   *
   * This method should be called from the proxy's stopped method
   */
  public void unregisterProxy(BBasicProxyExt proxy)
  {
    try
    {
      readUnsubscribed(proxy);  // Removes proxy from subscribed list
    }
    catch(Exception e)
    {}
    myProxies.remove(proxy);    // Removes proxy from registered list
    // GJ 01/15/07: Fix for removing poll group reference from Hashtable
    // Remove from hashtable
    Type pgType = proxy.getPollGroupType();
    Object pgCode = proxy.getPollGroupCode();

    // First lets lookup a HashTable for the particular poll group type
    Hashtable<Object, BBasicPollGroup> codesToGroups = typesToCodes.get(pgType.getId());

    if (codesToGroups!=null) // Sanity check
    {
      // Next lets get the poll group instance that is hashed to the proxy's code
      BBasicPollGroup pg = codesToGroups.get(pgCode);
      if (pg != null) // Sanity check
      {
        codesToGroups.remove(pgCode); // Removes the proxy from the poll group
      }
      // If there are no more groups for the poll group type...
      if (codesToGroups.size()<1) // Removes the codesToGroups hashtable from the
      {                           // typesToCodes hashtable
        typesToCodes.remove(pgType.getId());
      }
    }
  }



   /**
   * This callback is made when the point enters a subscribed
   * state based on the current status and tuning.  The driver
   * should register for changes or begin polling.  Any IO should
   * be done asynchronously on another thread - never block the
   * calling thread.  The result of reads should be to call the
   * readOk() or readFail() method.
   */
  public void readSubscribed(BBasicProxyExt proxy)
    throws Exception
  {
    if(!mySubscribedProxies.contains(proxy))
      mySubscribedProxies.add(proxy);
    if(!subscribed)
    {
      ((BBasicNetwork)proxy.getNetwork()).getPollScheduler().subscribe(this);
      subscribed = true;
    }
    //System.out.println("subscribe: mySubscribedProxes size is now " + mySubscribedProxies.size());
  }

  /**
   * This callback is made when the point exits the subscribed
   * state based on the current status and tuning.  The driver
   * should unregister for changes of cease polling.  Any IO should
   * be done asynchronously on another thread - never block the
   * calling thread.
   */
  public void readUnsubscribed(BBasicProxyExt proxy)
    throws Exception
  {
    mySubscribedProxies.remove(proxy);
    if(mySubscribedProxies.size() == 0)
    {
      ((BBasicNetwork)proxy.getNetwork()).getPollScheduler().unsubscribe(this);
      subscribed = false;
    }
    //System.out.println("unsubscribe: mySubscribedProxes size is now " + mySubscribedProxies.size());
  }

  /**
   * The descendant class should perform fieldbus communication and poll
   * this group of proxy extension. It should call the getProxyExts method
   * and update each registered proxy extension with the new data.
   */
  public abstract void poll();

  /**
   * Loops through all proxy extensions and returns the fastest poll frequency
   * Among them
   */
  public BPollFrequency getPollFrequency()
  {
    BPollFrequency fastestYet = BPollFrequency.slow; // Tracks the fastest poll frequency found among the group's proxies
    Iterator<BBasicProxyExt> i = mySubscribedProxies.iterator();     // Loops through all proxies that are subscribed for polling
    while (i.hasNext())
    {
      BBasicProxyExt proxy = i.next();
      if (proxy instanceof BIPollable)               // If any have a poll frequency
      {
        BPollFrequency fr = ((BIPollable)proxy).getPollFrequency();
        if (fr.compareTo(fastestYet) < 0)            // The BPollFrequencies with lower ordinal values are faster than those with higher ordinal values
          fastestYet = fr;                           // Then let's keep track of the fastest frequency of them all
      }
    }
    return fastestYet;                               // Returns the fastest BPollFrequency from among the proxies, of by default, BPollFrequency.slow if no proxies or none have poll frequencies
  }

  /**
   * This method returns an instance of the customized BBasicPollGroup for the given
   * proxy extension. Only one instance of a customized BBasicPollGroup will exist for
   * all proxy extensions that return the same poll group code. If that one
   * instance does not yet exist, one will be created, hashed, and returned.
   * Otherwise the one instance will be retrieved from a table and returned.
   *
   * This static method can be used as a public lookup to retrieve the instance of
   * BBasicPollGroup that is related to the given proxy, regardless of what driver that
   * proxy is in and regardless of the ultimate BBasicPollGroup type.
   */
  public static BBasicPollGroup getPollGroup(BBasicProxyExt proxy)
  {
    Type pgType = proxy.getPollGroupType();
    Object pgCode = proxy.getPollGroupCode();

    // First lets lookup a HashTable the particular poll group type
    Hashtable<Object, BBasicPollGroup> codesToGroups = typesToCodes.get(pgType.getId());
    if (codesToGroups==null) // Let's create one if we have to
    {
      codesToGroups=new Hashtable<>();
      typesToCodes.put(pgType.getId(), codesToGroups);
    }
    // Next lets get the poll group instance that is hashed to the proxy's code
    BBasicPollGroup pg = codesToGroups.get(pgCode);
    if (pg==null) // Let's create one if we have to
    {
      pg = (BBasicPollGroup)pgType.getInstance();
      pg.code=pgCode;
      codesToGroups.put(pgCode, pg);
    }
    return pg;
  }

  // This hashtable maps all driver poll group Types to a second Hashtable mapping codes to poll group instances
  static HashMap<Integer, Hashtable<Object, BBasicPollGroup>> typesToCodes=new HashMap<>();

  Object code; // This code is like a coalesce key. All proxies that are a member of this particular instance of BBasicPollGroup should return the same code
  Vector<BBasicProxyExt> mySubscribedProxies; // A list of all proxies that are subscribed for polling
  Vector<BBasicProxyExt> myProxies; // A list of all proxies in general that are polled by this poll group
  boolean subscribed = false;
}
