/*
 * Copyright 2002 Tridium, Inc. All Rights Reserved.
 */
package com.tridium.basicdriver.serial;

import java.io.InputStream;
import java.io.OutputStream;
import javax.baja.serial.BISerialHelperParent;
import javax.baja.serial.BISerialPort;
import javax.baja.serial.BISerialService;
import javax.baja.serial.BSerialHelper;
import javax.baja.sys.BBoolean;
import javax.baja.sys.BFacets;
import javax.baja.sys.BRelTime;
import javax.baja.sys.Clock;
import javax.baja.sys.Context;
import javax.baja.sys.Property;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import com.tridium.basicdriver.BBasicNetwork;
import com.tridium.basicdriver.comm.BCommPlugIn;
import com.tridium.basicdriver.comm.CommReceiver;
import com.tridium.basicdriver.message.Message;
import com.tridium.basicdriver.message.ReceivedMessage;
import com.tridium.basicdriver.util.BasicException;

/**
 * SerialComm is used to synchronize access to a
 * serial network and handles the synchronization of the serial
 * communication.
 * @author    AndySaunders
 * @creation  26 Mar 02
 * @version   $Revision: 1$ $Date: 03/26/02 12:47:14 PM$
 * @since     Niagara 3.0 basicdriver 1.0
 */

public class BSerialComm
  extends BCommPlugIn
  implements BISerialHelperParent
{
  /*-
  class BSerialComm
  {
    properties
    {
      interMessageDelay: BRelTime
        -- Specifies the minimum amount of time to wait after a response message is
        -- received before sending the next request message.
        default {[ BRelTime.make(0) ]}
        slotfacets {[ BFacets.make(BFacets.SHOW_MILLISECONDS, BBoolean.TRUE, BFacets.MIN, BRelTime.make(0), BFacets.MAX, BRelTime.SECOND) ]}
        
      serialPortConfig: BSerialHelper
        -- Contains the serial port properties
        default {[ new BSerialHelper() ]}
    }
  }
  -*/
/*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
/*@ $com.tridium.basicdriver.serial.BSerialComm(2081991697)1.0$ @*/
/* Generated Tue Feb 03 16:29:45 EST 2009 by Slot-o-Matic 2000 (c) Tridium, Inc. 2000 */

////////////////////////////////////////////////////////////////
// Property "interMessageDelay"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>interMessageDelay</code> property.
   * Specifies the minimum amount of time to wait after
   * a response message is received before sending the next request message.
   * @see com.tridium.basicdriver.serial.BSerialComm#getInterMessageDelay
   * @see com.tridium.basicdriver.serial.BSerialComm#setInterMessageDelay
   */
  public static final Property interMessageDelay = newProperty(0, BRelTime.make(0),BFacets.make(BFacets.SHOW_MILLISECONDS, BBoolean.TRUE, BFacets.MIN, BRelTime.make(0), BFacets.MAX, BRelTime.SECOND));
  
  /**
   * Get the <code>interMessageDelay</code> property.
   * @see com.tridium.basicdriver.serial.BSerialComm#interMessageDelay
   */
  public BRelTime getInterMessageDelay() { return (BRelTime)get(interMessageDelay); }
  
  /**
   * Set the <code>interMessageDelay</code> property.
   * @see com.tridium.basicdriver.serial.BSerialComm#interMessageDelay
   */
  public void setInterMessageDelay(BRelTime v) { set(interMessageDelay,v,null); }

////////////////////////////////////////////////////////////////
// Property "serialPortConfig"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>serialPortConfig</code> property.
   * Contains the serial port properties
   * @see com.tridium.basicdriver.serial.BSerialComm#getSerialPortConfig
   * @see com.tridium.basicdriver.serial.BSerialComm#setSerialPortConfig
   */
  public static final Property serialPortConfig = newProperty(0, new BSerialHelper(),null);
  
  /**
   * Get the <code>serialPortConfig</code> property.
   * @see com.tridium.basicdriver.serial.BSerialComm#serialPortConfig
   */
  public BSerialHelper getSerialPortConfig() { return (BSerialHelper)get(serialPortConfig); }
  
  /**
   * Set the <code>serialPortConfig</code> property.
   * @see com.tridium.basicdriver.serial.BSerialComm#serialPortConfig
   */
  public void setSerialPortConfig(BSerialHelper v) { set(serialPortConfig,v,null); }

////////////////////////////////////////////////////////////////
// Type
////////////////////////////////////////////////////////////////
  
  public Type getType() { return TYPE; }
  public static final Type TYPE = Sys.loadType(BSerialComm.class);

/*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

  /**
   * This method is called by BSerialHelper when the port name has changed
   * to indicate that a reinitialization of the serial port is 
   * required (i.e. close the old port, and 
   * and then call BSerialHelper.open() to open the new port).
   */
  public final void reopenPort()
  {
    try
    {
      String newPort = getSerialPortConfig().getPortName();
      
      if (newPort.equals(BSerialHelper.noPort))
      {  // 'none' is selected, so stop the network and don't restart
        getNetwork().configFail("No port selected for serial communication.");
        getNetwork().stopComm();
        return;
      }
      restartSerialNetwork();
    }
    catch (Exception e)
    {
      getNetwork().getLog().error("BSerialNetwork caught exception in reopenPort(): ", e);
    }
  }

  /**
   * Restarts the serial communication network.  This method
   * gets called when the serial port gets changed.
   */
  private void restartSerialNetwork()
    throws Exception
  {
    BBasicNetwork net = getNetwork();
    if ((!net.isDisabled()) && (!net.isDown()) && (!net.isFatalFault()))
    {
      if (net.getLog().isTraceOn()) net.getLog().trace(getName() + " *** Restarting serial comm ***");
      net.stopComm();
      net.startComm();
    }
  }


/////////////////////////////////////////////////////////////////
// Comm port management
/////////////////////////////////////////////////////////////////

  /**  
   * Starts the serial transmit/receive drivers. Returns true if successfully started, false otherwise.
   * Opens the serial port and extracts the input and output streams,
   * and also starts the receive driver thread.
   */
  public boolean commStarted()
    throws Exception
  {
    try
    {                       
      BISerialService platSvc = (BISerialService)Sys.getService(BISerialService.TYPE);

      serialPort = getSerialPortConfig().open(getNetwork().getName());
  
      serialPort.enableReceiveTimeout(platSvc.getMinTimeout());

      in = serialPort.getInputStream();
      out = serialPort.getOutputStream();
    }
    catch(Exception e)
    {
      String errMsg = "Error opening and configuring the serial port";
      getNetwork().getLog().error(errMsg, e);
      
      if (in != null)
      {
        try
        {
          in.close();
        }
        catch (Exception x)
        {
          getNetwork().getLog().error("Unable to close serial input stream.", x);
        }
      }

      if (out != null)
      {
        try
        {
          out.close();
        }
        catch (Exception x)
        {
          getNetwork().getLog().error("Unable to close serial output stream.", x);
        }
      }

      if(serialPort != null) serialPort.close();
      
      throw e;
    }

    // Set the receive/transmit drivers streams and
    // start the receive thread
    CommReceiver commRx = getCommReceiver();
    commRx.setInputStream(in);
    getCommTransmitter().setOutputStream(out);
    rxThread = new Thread(commRx, "SerialRcv:" + getNetwork().getName());
    commRx.setAlive(true);
    rxThread.start();
    rxThread.setPriority(Thread.NORM_PRIORITY);

    return true;
  }

  /**
   * Stops the serial transmit/receive drivers.
   * Closes the serial port and the input and output streams.
   */
  public void commStopped()
    throws Exception
  {
    getCommReceiver().setAlive(false);
    if((getCommReceiver()!=null) && (rxThread!=null)) rxThread.interrupt();

    if (in != null)
    {
      try
      {
        in.close();
      }
      catch (Exception e)
      {
        getNetwork().getLog().error("Unable to close serial input stream.", e);
      }
    }

    if (out != null)
    {
      try
      {
        out.close();
      }
      catch (Exception e)
      {
        getNetwork().getLog().error("Unable to close serial output stream.", e);
      }
    }

    if(serialPort != null)
    {
      serialPort.disableReceiveTimeout();
      serialPort.close();
    }
    in  = null;
    out = null;
  }

////////////////////////////////////////////////////////////
//  BasicDriver Api Overrides
////////////////////////////////////////////////////////////

  /**
   * Send a message using the message request/response service to
   * the communication medium.  Block the calling thread
   * until the response is obtained or the transaction times out.
   * Overridden here to enforce the inter message delay.
   *
   * @param msg a network request (in message form) to be
   *    sent to the output stream
   * @param responseTimeout the timeout to wait for a response for
   *    this request.
   * @param retryCount the number of retries to perform if the request
   *    fails (a timeout occurs).
   * @return Message the response received for the sent message
   *    if successful (or null if no response expected),
   *    otherwise an exception is thrown (i.e. timeout).
   */
  public Message transmit(Message msg, BRelTime responseTimeout, int retryCount)
    throws BasicException
  {
    if (msg == null)
      return null;

    // May not need this check - keep for now for legacy purposes
    if(!msg.getResponseExpected())
    {
      transmitNoResponse(msg);
      return null;
    }

    performInterMessageDelay();
    
    return getComm().transmit(msg, responseTimeout, retryCount);
  }

  /**
   * Send a message to the transmit driver and do not expect or wait
   * for a response from the receive driver.
   * Overridden here to enforce the inter message delay.
   *
   * @param msg a message to be sent to the output stream
   */
  public void transmitNoResponse(Message msg)
    throws BasicException
  {
    if(msg == null)
      return;

    if(!getComm().isCommStarted()) throw new BasicException("Communication handler service not started.");

    performInterMessageDelay();
    
    getComm().transmitNoResponse(msg);
  }

  /**
   * This is the access point for the receive driver to
   * pass its received unsolicited messages and/or 
   * response messages up to the communications handler for
   * processing.  Overridden here to enforce the Inter Message
   * Delay.
   *
   * @param msg the response/unsolicited message received 
   *    from the input stream.
   */
  public void receive(ReceivedMessage msg)
  {
    if (msg == null) return; // By default, don't do anything if received a null message.
    
    setReceivedMessageTicks(Clock.ticks());
    
    getComm().receive(msg);
  }

////////////////////////////////////////////////////////////
//  SerialComm methods
////////////////////////////////////////////////////////////

  /**
   * This method is called just before a message transmit in order to 
   * checks to see when the last time a message was received.
   * It waits the necessary inter message delay time if needed before returning.
   */
  protected void performInterMessageDelay()
  {
    long minDelay = ((BSerialNetwork)getNetwork()).getInterMessageDelay().getMillis();
    if (minDelay <= 0L) return; // short circuit
    
    long difference = Clock.ticks() - lastRecvMessageTicks;
    
    if (difference >= minDelay) return;
    
    // Otherwise sleep the appropriate time
    long sleepTime = Math.max((minDelay - difference), MIN_SLEEP_TIME);
    try { Thread.sleep(sleepTime); } catch (Exception e) { e.printStackTrace(); }
  }
  
  /**
   * Should be called with the current time ticks whenever a message
   * is received.
   */
  protected void setReceivedMessageTicks(long ticks)
  {
    lastRecvMessageTicks = ticks;
  }

/////////////////////////////////////////////////////
// Serial port convenience methods
//////////////////////////////////////////////////////
  
  /**
   * Returns the serial port used by this SerialComm
   */
  public BISerialPort getSerialPort()
  {
    return serialPort;
  }

  public String toString(Context cx)
  {
    StringBuffer sb = new StringBuffer();
    BSerialHelper sh = getSerialPortConfig();
    sb.append(sh.getPortName());
    sb.append(' ');
    sb.append(sh.getBaudRate());
    sb.append(',');
    sb.append(sh.getDataBits());
    sb.append(',');
    sb.append(sh.getStopBits());
    sb.append(',');
    sb.append(sh.getParity());
    return sb.toString();
 }
////////////////////////////////////////////////////////////
//  Attributes of SerialComm
////////////////////////////////////////////////////////////
  private static final long MIN_SLEEP_TIME = 10L;
  
  private BISerialPort serialPort;
  private InputStream in;
  private OutputStream out;
  private Thread rxThread;
  private long lastRecvMessageTicks = 0L;

}