/*
 * Copyright 2002 Tridium, Inc. All Rights Reserved.
 */
package com.tridium.basicdriver.message;

import java.io.OutputStream;
import com.tridium.basicdriver.util.BBasicUnsolicitedReceive;

/**
 * The ReceivedMessage class is the super class for all 
 * incoming (response/unsolicited) messages.
 *
 * @author    Scott Hoye
 * @creation  25 Mar 02
 * @version   $Revision: 1$ $Date: 03/25/02 12:47:14 PM$  
 * @since     Niagara 3.0 basicdriver 1.0
 */
public abstract class ReceivedMessage
  extends Message
{

////////////////////////////////////////////////////////////
// Message
////////////////////////////////////////////////////////////

 /**
  * The default implementation of write on a ReceivedMessage
  * is to do nothing. 
  * Can be overridden by subclasses. 
  */
  public void write(OutputStream out) { }

  /**
  * The default implementation returns null.
  * Can be overridden by subclasses.
  */
  public Message toResponse(ReceivedMessage resp)
  {
    return null;
  }

  /**
   * Be careful calling this method because the corresponding method getSuccessfulResponse
   * for ReceivedMessages is not checked in the BasicDriver framework. Thefore, calling
   * setSuccessfulResponse on a ReceivedMessage instance will have no effect unless your
   * driver checks getSuccessfulResponse explicitly.
   * 
   * For example, you could call getSuccessfulResponse from the toResponse(ReceivedMessage)
   * method of your driver's request Message and then do your own processing if you wish.
   * 
   * @see com.tridium.basicdriver.message.Message#setSuccessfulResponse(boolean)
   * @see com.tridium.basicdriver.message.Message#getSuccessfulResponse()
   * @see com.tridium.basicdriver.message.Message#toResponse(ReceivedMessage)
   */
  public void setSuccessfulResponse(boolean b)
  {
    super.setSuccessfulResponse(b);
  }
////////////////////////////////////////////////////////////
//  Access methods
////////////////////////////////////////////////////////////
  
 /**
  * Returns true if it is known that this is an unsolicited
  * received message.  Returns false otherwise.
  */  
  public boolean getUnsolicited() { return unsolicited; }

 /**
  * Sets whether this response is known to be unsolicited.
  */    
  public void setUnsolicited(boolean unsolicited)
  {
    this.unsolicited = unsolicited;
  }
  
 /** 
  * Sets the unsolicited listener code associated with this message.  
  * Used to determine which unsolicited message listener should receive 
  * this response message if it is determined to be unsolicited.
  */
  public void setUnsolicitedListenerCode(Object code) 
  { 
    this.unsolicitedListenerCode = code;
  }
  
 /** 
  * Gets the unsolicited listener code associated with this message.  
  * Used to determine which unsolicited message listener should receive 
  * this response message if it is determined to be unsolicited.
  */
  public Object getUnsolicitedListenerCode() 
  { 
    return unsolicitedListenerCode; 
  }
  
  /**
   * Return a debug string for this message.
   */
  public String toDebugString()
  {
    StringBuffer sb = new StringBuffer();
    sb.append("ReceivedMessage = " + toString());
    sb.append("\n  Tag = " + getTag());
    sb.append("\n  Known Unsolicited = " + getUnsolicited());
    sb.append("\n  Unsolicited Listener Code = " + getUnsolicitedListenerCode());
    return sb.toString();
  }
  
////////////////////////////////////////////////////////////
//  Attributes
////////////////////////////////////////////////////////////
  private boolean unsolicited = false; 
  private Object unsolicitedListenerCode = BBasicUnsolicitedReceive.TYPE; // Asks Basic Driver's Comm to use BBasicUnsolictedReceive as the default unsolicited recieve handler

}
