/*
 * Copyright 2002 Tridium, Inc. All Rights Reserved.
 */
package com.tridium.basicdriver.comm;

import java.io.InputStream;
import com.tridium.basicdriver.message.Message;
import com.tridium.basicdriver.message.ReceivedMessage;

/**
 * CommReceiver is the base receive driver for the communication medium.
 * It should be subclassed to handle receiving complete messages from the 
 * input stream.
 *
 * @author    Scott Hoye
 * @creation  26 Mar 02
 * @version   $Revision: 1$ $Date: 03/26/02 12:47:14 PM$
 * @since     Niagara 3.0 basicdriver 1.0
 */
public abstract class CommReceiver
  extends Comm.CommSupport
  implements Runnable
{

 /**
  * Sets the input stream associated with this
  * receive driver for reading byte[] messages.
  */
  public void setInputStream(InputStream in)
  {
    this.in = in;
  }
  
  /**
  * Gets the input stream associated with this
  * receive driver for reading messages.
  */
  protected InputStream getInputStream()
  {
    return in;
  }
  
  /**
  * Sets the flag to indicate the receive thread should
  * start running (alive = true) or stop running (alive = false).
  */
  public void setAlive(boolean alive)
  {
    this.isAlive = alive;
  }
  
  /**
  * Returns whether this receive thread is
  * running (true) or stopped (false).
  */
  public boolean isAlive()
  {
    return isAlive;
  }

 /**
  * Called just before a byte[] 
  * is written to the output stream.  Subclasses should
  * override this method to initialize the receive
  * driver before each byte[] is sent.
  * The default implementation is no action.
  */
  public void initReceiveState(byte[] msg) { }
  
 /**
  * Called just before a Message
  * is written to the output stream.  Subclasses should
  * override this method to initialize the receive
  * driver before each Message request is sent.
  * The default implementation is no action.
  */
  public void initReceiveState(Message msg) { }

 /**
  * The execution of the CommReceiver is to continuously
  * receive messages from the input stream.  This method loops
  * and calls receive() to allow subclasses to return complete
  * Messages.  It routes any received message to the Comm.  It
  * loops forever until isAlive() returns false.
  */
  public void run()
  {
    while(isAlive())
    {
      try
      {
        ReceivedMessage msg = receive();
        getComm().receive(msg);
      }
      catch(Exception e)
      {
        if (isAlive())
          getComm().getNetwork().getLog().message("Exception in CommReceiver.run()", e);
      }
    }
    getComm().receiveFinal();
  }
  
 /**
  * Should be overridden by subclasses to read bytes
  * from the input stream and form a complete ReceivedMessage
  * suitable to pass back up to the Comm.  Subclasses should
  * make sure that the ReceivedMessage returned contains all
  * necessary information to handle it (i.e. whether it should
  * be treated as unsolicited, any necessary data, tag information,
  * etc.).
  *
  * Important implementation note:  If the driver allows for received
  * messages that need special processing (i.e. unsolicited received
  * messages or out-of-order responses), then a new instance of a 
  * ReceivedMessage should be generated and returned by this method.
  * Otherwise it may be safe to return the same instance of 
  * ReceivedMessage as the result of this method.  Using the same
  * instance can save on memory allocation, but in the case where 
  * the ReceivedMessage may need special processing by the caller
  * (i.e. the result gets passed and processed on a separate thread 
  * from the calling thread), then it may not be safe to return the 
  * same instance.
  */
  protected abstract ReceivedMessage receive() throws Exception;

////////////////////////////////////////////////////////////////
//  Attributes of CommReceiver
////////////////////////////////////////////////////////////////
  private InputStream in;
  private boolean isAlive = false;

}