/*
 * Copyright 2002 Tridium, Inc. All Rights Reserved.
 */
package com.tridium.basicdriver.comm;

import java.util.ArrayList;
import java.util.Hashtable;
import javax.baja.sys.BRelTime;
import com.tridium.basicdriver.BBasicNetwork;
import com.tridium.basicdriver.UnsolicitedMessageListener;
import com.tridium.basicdriver.message.Message;
import com.tridium.basicdriver.message.ReceivedMessage;
import com.tridium.basicdriver.util.BasicException;

/**
 * The Comm class is used to manage message request/response
 * transactions on a basic network and handles the synchronization 
 * of the communication.  It acts as a parent to handle
 * the interaction between the transmit and receive drivers.
 *
 * @author    Scott Hoye
 * @creation  26 Mar 02
 * @version   $Revision: 1$ $Date: 03/26/02 12:47:14 PM$
 * @since     Niagara 3.0 basicdriver 1.0
 */

public abstract class Comm
{

////////////////////////////////////////////////////////////
//  Constructor
////////////////////////////////////////////////////////////

 /**
  * Constructor - initializes the Comm with a specified BBasicNetwork
  * and CommReceiver (receive driver).  At a minimum, these two instances
  * must always be supplied.  Uses the default CommTransmitter 
  * (transmit driver) and CommTransactionManager.
  */
  public Comm(BBasicNetwork basicNetwork, CommReceiver rDriver)
  {
    this.basicNetwork = basicNetwork;
    setCommReceiver(rDriver);
    setCommTransmitter(new CommTransmitter());
    setCommTransactionManager(new CommTransactionManager());
  }

 /**
  * Constructor - initializes the Comm with a specified BBasicNetwork,
  * CommReceiver (receive driver), and CommTransmitter (transmit driver).
  * Uses the default CommTransactionManager.
  */
  public Comm(BBasicNetwork basicNetwork, CommReceiver rDriver, CommTransmitter tDriver)
  {
    this.basicNetwork = basicNetwork;
    setCommReceiver(rDriver);
    setCommTransmitter(tDriver);
    setCommTransactionManager(new CommTransactionManager());
  }

 /**
  * Constructor - initializes the Comm with a specified BBasicNetwork,
  * CommReceiver (receive driver), CommTransmitter (transmit driver), and
  * CommTransactionManager.
  */
  public Comm(BBasicNetwork basicNetwork, CommReceiver rDriver, CommTransmitter tDriver, CommTransactionManager manager)
  {
    this.basicNetwork = basicNetwork;
    setCommReceiver(rDriver);
    setCommTransmitter(tDriver);
    setCommTransactionManager(manager);
  }

////////////////////////////////////////////////////////////
//  API
////////////////////////////////////////////////////////////

  /**
   * Set a transaction manager or override the default transaction manager
   */
  public final void setCommTransactionManager(CommTransactionManager manager)
  {  
    try
    {
      boolean restart = installSupport(this.transactionManager, manager);
      this.transactionManager = manager;  
      if (restart) start();
    }
    catch (Exception e)
    {
      basicNetwork.getLog().error("Error in Comm.setCommTransactionManager()", e);
    }
  }

  /**
   * Set a CommTransmitter (transmit driver)
   */
  public final void setCommTransmitter(CommTransmitter tDriver)
  { 
    try
    {
      boolean restart = installSupport(this.tDriver, tDriver);
      this.tDriver = tDriver;
      if (restart) start();
    }
    catch (Exception e)
    {
      basicNetwork.getLog().error("Error in Comm.setCommTransmitter()", e);
    }
  }

  /**
   * Set a CommReceiver (receive driver)
   */
  public final void setCommReceiver(CommReceiver rDriver)
  {  
    try
    {
      boolean restart = installSupport(this.rDriver, rDriver);
      this.rDriver = rDriver;
      if (restart) start();
    }
    catch (Exception e)
    {
      basicNetwork.getLog().error("Error in Comm.setCommReceiver()", e);
    }
  }

  /**
   * Retrieve a reference to the basic network
   * being used by this communications handler
   */
  public BBasicNetwork getNetwork()
  {
    return this.basicNetwork;
  }

  /**
   * Retrieve a reference to the transaction manager
   * being used by this communications handler
   */
  public CommTransactionManager getCommTransactionManager()
  {
    return transactionManager;
  }

  /**
   * Retrieve a reference to the CommTransmitter (transmit driver)
   * being used by this communications handler
   */
  public CommTransmitter getCommTransmitter()
  {
    return tDriver;
  }

  /**
   * Retrieve a reference to the CommReceiver (receive driver)
   * being used by this communications handler
   */
  public CommReceiver getCommReceiver()
  {
    return rDriver;
  }
  
  /**
   * Check that the specified support is not null and 
   * not installed on another Comm.
   */
  private boolean installSupport(CommSupport old, CommSupport support)
    throws Exception
  {
    if (support == null) throw new NullPointerException();
    if (old == support) return false;
    if (support.comm != null) throw new IllegalArgumentException("Already installed on another Comm");
    boolean restart = isCommStarted(); // Check to see if we need to attempt a restart of the Comm
    if (restart) stop();
    if (old != null) old.setComm(null);
    support.setComm(this);
    return restart;
  }
  
 /**
  * Returns true if this Communication handler
  * has been started and is running, false if not.
  */
  public boolean isCommStarted()
  {
    return commStarted;
  }

/////////////////////////////////////////////////////////////
// Subordinate services management
/////////////////////////////////////////////////////////////

  /**
   * Starts the Comm.  Called by BBasicNetwork
   * whenever this Comm handler should be enabled.
   * Calls the abstract method started().
   */
  public final void start()
    throws Exception
  {
    if (!isCommStarted())// && (basicNetwork != null) && (basicNetwork.isServiceRunning()))
    {
      if ((tDriver != null) && (started()))
      {
        commStarted = true;
      }
    }
  }

  /**
   * Stops the Comm.  Called by BBasicNetwork
   * whenever the Comm handler should be disabled.
   * Calls the abstract method stopped().
   */
  public final void stop()
    throws Exception
  {
    if (isCommStarted())
    {
      commStarted = false;
      if (basicNetwork.getDispatcher().isRunning())
      {
        StopRequest stopReq = new StopRequest();
        basicNetwork.dispatch(stopReq);
        stopReq.stop();
      }
      stopped();
    }
  }

  /**  
   * Starts the transmit/receive drivers. Returns true if successfully started, false otherwise.
   */
  protected abstract boolean started() throws Exception;

  /**
   * Stops the transmit/receive drivers.
   */
  protected abstract void stopped() throws Exception;

  /**
   * Send a message using the message request/response service to
   * the communication medium. Block the calling thread
   * until the response is obtained or the transaction times out.
   * Uses the default response time out and retry count defined
   * at the network level.
   *
   * @param msg a network request (in message form) to be
   *    sent to the output stream
   * @return a Message - the response received for the sent message
   *    if successful (or null if no response expected),
   *    otherwise an exception is thrown (i.e. timeout).
   */
  public final Message transmit(Message msg)
    throws BasicException
  {
    return transmit(msg, basicNetwork.getResponseTimeout(), basicNetwork.getRetryCount());
  }

  /**
   * Send a message using the message request/response service to
   * the communication medium.  Block the calling thread
   * until the response is obtained or the transaction times out.
   *
   * @param msg a network request (in message form) to be
   *    sent to the output stream
   * @param responseTimeout the timeout to wait for a response for
   *    this request.
   * @param retryCount the number of retries to perform if the request
   *    fails (a timeout occurs).
   * @return Message the response received for the sent message
   *    if successful (or null if no response expected),
   *    otherwise an exception is thrown (i.e. timeout).
   */
  public Message transmit(Message msg, BRelTime responseTimeout, int retryCount)
    throws BasicException
  {
    if (msg == null)
      return null;

    // May not need this check - keep for now for legacy purposes
    if(!msg.getResponseExpected())
    {
      transmitNoResponse(msg);
      return null;
    }

    BasicException e = null;
    Message respMsg = null;

    try
    {
      for(int i=0; i<retryCount+1; i++)
      {
        respMsg = processTransmit(msg, responseTimeout);
        if ((respMsg != null) && respMsg.getSuccessfulResponse())
          break;
      }

      if( (respMsg != null) && !respMsg.getSuccessfulResponse() )
      {
        String failMsg = "Unsuccessful response for request message sent.";
        //if (respMsg == null) failMsg = "No response for request message sent.";
        e = new BasicException(failMsg);
      }

    }
    catch(BasicException se)
    {
      e = se;
    }

    if (e!=null)
    {
      if (e instanceof BasicException)
      {
        if (basicNetwork.getLog().isTraceOn()) basicNetwork.getLog().trace("Comm sendRequest exception: ", e);
      }
      else
        throw e;
    }

    return respMsg;
  }

  /**
   * Send a message to the transmit driver and do not expect or wait
   * for a response from the receive driver.
   *
   * @param msg a message to be sent to the output stream
   */
  public void transmitNoResponse(Message msg)
    throws BasicException
  {
    if(msg == null)
      return;

    if(!commStarted) throw new BasicException("Communication handler service not started.");

    tDriver.writeMessage(msg);
    getNetwork().incrementSent();
  }

  /**
   * Send a message to the transmit driver and wait for a response
   * from the receive driver. Block the calling thread
   * until the response is obtained or the transaction times out.
   *
   * @param msg a message to be converted to bytes and
   *    sent to the output stream
   * @param responseTimeout the timeout to wait for a response for
   *    this request message.
   * @return Message the response received for the sent message
   *    if successful, otherwise an exception is thrown (i.e. timeout).
   */
  protected Message processTransmit(Message msg, BRelTime responseTimeout)
    throws BasicException
  {
    if(!commStarted) throw new BasicException("Communication handler service not started.");

    CommTransaction transaction = transactionManager.getCommTransaction(msg);

    synchronized(transaction)
    {

      tDriver.writeMessage(msg);
      getNetwork().incrementSent();

      try
      {
        if (!transaction.isComplete())
        {
          transaction.wait(responseTimeout.getMillis());
          if (!transaction.isComplete())
          {
            if (basicNetwork.getLog().isTraceOn())
            {
              Object reqTag = transaction.getRequestMessage().getTag();
              basicNetwork.getLog().trace("CommTransaction timed out (tag: " + reqTag + ")");
              if (reqTag != Message.DEFAULT_TAG) // indicates not request/immediate response behavior
              {
                try
                {
                  Message req = transaction.getRequestMessage();
                  if(req!=null) basicNetwork.getLog().trace("Failure to get response for sent message: " + req.toDebugString());
                  //if(req!=null) ByteArrayUtil.hexDump(req.getByteArray());
                } catch(Exception e){}
              }
            }
            transaction.setResponseMessage(null);
            basicNetwork.incrementTimeouts();
          }
          transaction.setComplete(true);
        }
      }
      catch ( InterruptedException e) {}
    }

    Message resp = transaction.getResponseMessage();

    transactionManager.freeCommTransaction(transaction);

    return resp;
  }


////////////////////////////////////////////////////////////
//  BasicDriver Api
////////////////////////////////////////////////////////////

  /**
   * This is the access point for the receive driver to
   * pass its received unsolicited messages and/or 
   * response messages up to the communications handler for
   * processing.
   *
   * @param msg the response/unsolicited message received 
   *    from the input stream.
   */
  public void receive(ReceivedMessage msg)
  {
    if (msg == null) return; // By default, don't do anything if received a null message.
    getNetwork().incrementReceived();
    if (basicNetwork.getLog().isTraceOn()) basicNetwork.getLog().trace("**** Received message: " + msg.toDebugString());

    if (msg.getUnsolicited())
      routeToListeners(msg);
    else if (!handleReceivedMessage(msg))
    {
      // Route all other incoming message to UnsolicitedMessageListeners
      routeToListeners(msg);
    }
  }

 /**
  * Method called by receive driver during
  * shutdown to free any pending message requests.
  */
  public synchronized void receiveFinal()
  {
    // Stop all transactions
    getCommTransactionManager().cancelAllOutstandingCommTransactions();
  }

  /**
   * This is the access point for the CommTransmitter to
   * indicate a failure to transmit.  It is responsible
   * for cleaning up the pending transaction.
   *
   * @param msg the message that failed to send.
   * @param e the exception explaining the reason for the failed send.
   */
  public void handleFailedTransmit(Message msg, Exception e)
  {

    basicNetwork.getLog().message("Failure to transmit in CommTransmitter.", e);

    //  Extract the associated transaction from
    //  the manager based on received message's tag.

    // if not req/resp, have no transaction to clean up.
    if(!msg.getResponseExpected())
      return;

    Object tag = msg.getTag();

    CommTransaction transaction = getCommTransactionManager().getCommTransactionMatch(tag);

    synchronized( transaction)
    {
      transaction.setResponseMessage( null);
      transaction.setComplete(true);
      transaction.notify();
    }
  }

  /**
   * Handles processing of a received message 
   * from the CommReceiver.
   * Attempts to find and notify the matching transaction.  
   * Returns true if the matching transaction is found, false 
   * otherwise.
   *
   * @param msg the message received from the input stream
   */
  protected boolean handleReceivedMessage(ReceivedMessage msg)
  {
    //  Extract the associated transaction from
    //  the manager based on received message's tag.
    Object tag = msg.getTag();
    CommTransaction transaction = getCommTransactionManager().getCommTransactionMatch(tag);

    if (transaction == null)
      return false;

    //  If this transaction is in use( should be),
    //  then act appropriately
    synchronized(transaction)
    {
      if(!transaction.isUsed())
      {
        if (basicNetwork.getLog().isTraceOn()) basicNetwork.getLog().trace("Unmatched response received - assuming unsolicited");
        return false;
      }

      transaction.setResponseMessage( transaction.getRequestMessage().toResponse(msg));
      transaction.setComplete(true);
      transaction.notify();
    }
    return true;
  }

////////////////////////////////////////////////////////////
//  UnsolicitedMessageListener support
////////////////////////////////////////////////////////////

  /**
   * Register an UnsolicitedMessageListener (i.e. for handling unsolicited
   * received messages).  Uses its unsolicited listener code to determine 
   * which incoming messages should be routed to it (matches the unsolicited 
   * listener code of the received message with the one registered for the
   * UnsolicitedMessageListener.
   *
   * @param listener the unsolicited message listener to register
   */
  public void registerListener(UnsolicitedMessageListener listener)
  {
    Object unsolicitedListenerCode = listener.getUnsolicitedListenerCode();
    if (unsolicitedListenerCode == null) return;
    synchronized(listeners)
    {

      ArrayList<UnsolicitedMessageListener> v = listeners.get(unsolicitedListenerCode);

      // If there is not an arraylist for this code create one
      if ( v == null)
      {
        v = new ArrayList<>(4);
        listeners.put(unsolicitedListenerCode,v);
      }

      // Check if there is already an appropriate entry.
      for(int i=0 ; i<v.size() ; i++ )
      {
        UnsolicitedMessageListener ld = v.get(i);
        // If listener is same then this is a match - return
        if(listener == ld) return;

      }

      // Add new entry in vector.
      v.add(listener);
    }
  }

  /**
   * Unregisters an UnsolicitedMessageListener (i.e. for handling unsolicited
   * received messages).
   *
   * @param listener the unsolicited message listener to unregister
   */
  public void unregisterListener(UnsolicitedMessageListener listener)
  {
    Object unsolicitedListenerCode = listener.getUnsolicitedListenerCode();
    if (unsolicitedListenerCode == null) return;
    synchronized(listeners)
    {
      // Find vector for specified code.
      ArrayList<UnsolicitedMessageListener> v = listeners.get(unsolicitedListenerCode);
      if ( v == null) return;
      unregister(v, listener);
    }
  }

  /**
   * Unregisters an UnsolicitedMessageListener (i.e. for handling unsolicited
   * received messages) from the given ArrayList of listeners.
   *
   * @param v the ArrayList of listeners from which to remove
   *    the specified unsolicited message listener
   * @param listener the unsolicited message listener to remove from the ArrayList
   */
  private void unregister(ArrayList<UnsolicitedMessageListener> v, UnsolicitedMessageListener listener)
  {
    // Walk arraylist from high to low to allow element removal.
    for(int i=(v.size() - 1) ; i>=0 ; i-- )
    {
      UnsolicitedMessageListener ld = v.get(i);

      // If for different listener skip
      if(listener == ld)
        v.remove(i);
    }
  }

  /**
   * Routes a received message to each listener 
   * registered for this message.  Uses the unsolicitedListenerCode 
   * of the Message to determine which listeners should receive the message.
   *
   * @param msg the message received from the input stream
   */
  protected void routeToListeners(ReceivedMessage msg)
  {
    Object unsolicitedListenerCode = msg.getUnsolicitedListenerCode();
    if (unsolicitedListenerCode == null) return;
    synchronized(listeners)
    {
      ArrayList<UnsolicitedMessageListener> v = listeners.get(unsolicitedListenerCode);

      if(v != null)
      {
        for(int i = 0 ; i < v.size() ; i++)
        {
          UnsolicitedMessageListener ld = v.get(i);

          // Send message to listener
          ld.receiveMessage(msg);
        }
      }
    }
  }
  
////////////////////////////////////////////////////////////////
// CommSupport
////////////////////////////////////////////////////////////////  

  /**
   * Abstract base class for support classes.
   */
  public static abstract class CommSupport
  {
    /**
     * Get the Comm the support instance is installed on.
     */
    public final Comm getComm()
    {
      return comm;
    }

    /**
     * Set the Comm the support instance is installed on.
     */
    void setComm(Comm comm) { this.comm = comm; }
    
    Comm comm;
  }

////////////////////////////////////////////////////////////
//  Attributes of Comm
////////////////////////////////////////////////////////////
  private BBasicNetwork basicNetwork;
  private CommReceiver rDriver = null;
  private CommTransmitter tDriver = null;
  private CommTransactionManager transactionManager = null;
  private boolean commStarted = false;
  private Hashtable<Object, ArrayList<UnsolicitedMessageListener>> listeners = new Hashtable<>(32);
  

////////////////////////////////////////////////////////////
// Inner class StopRequest
////////////////////////////////////////////////////////////
  /**
   * The StopRequest class is used to ensure that all other
   * synchronous requests have completed before service stop
   * completes.
   */
  class StopRequest
    implements Runnable
  {
  
    /**
     * Empty default constructor
     */
    public StopRequest()
    {
    }

    /**
     * Notifies that the stop request has completed
     * to signify that all previous synchronous requests have
     * already passed through.  
     */
    public synchronized void run()
    {
      // Notify requester that response has been received.
      complete = true;

      this.notify();
    }
  
    /**
     * Returns after this StopRequest gets processed.
     */
    public synchronized void stop()
    {
      if(!complete) try { this.wait(); } catch(Exception e) 
      { 
        basicNetwork.getLog().error("Error in Comm.StopRequest.stop()", e);
      }
    }

    private boolean complete = false;
  }
}