/*
 * Copyright 2002 Tridium, Inc. All Rights Reserved.
 */
package com.tridium.basicdriver;

import javax.baja.driver.loadable.BLoadableNetwork;
import javax.baja.driver.point.BTuningPolicyMap;
import javax.baja.driver.util.BPollScheduler;
import javax.baja.log.Log;
import javax.baja.spy.SpyWriter;
import javax.baja.sys.BBoolean;
import javax.baja.sys.BFacets;
import javax.baja.sys.BIService;
import javax.baja.sys.BRelTime;
import javax.baja.sys.Clock;
import javax.baja.sys.Context;
import javax.baja.sys.Flags;
import javax.baja.sys.Property;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import javax.baja.util.ICoalesceable;
import javax.baja.util.IFuture;
import com.tridium.basicdriver.comm.Comm;
import com.tridium.basicdriver.message.Message;
import com.tridium.basicdriver.util.BBasicCoalescingWorker;
import com.tridium.basicdriver.util.BBasicPollScheduler;
import com.tridium.basicdriver.util.BBasicWorker;

/**
 * BBasicNetwork is the base container for Basic Devices.   
 * Manages the basic workers (queues), the basic poll scheduler, 
 * and the communication handler (Comm).
 *
 * @author    Scott Hoye
 * @creation  22 Mar 02
 * @version   $Revision: 1$ $Date: 03/22/02 12:47:14 PM$
 * @since     Niagara 3.0 basicdriver 1.0
 */
public abstract class BBasicNetwork
  extends BLoadableNetwork
  implements BIService
{
  /*-

  class BBasicNetwork
  {
    properties
    {
      tuningPolicies: BTuningPolicyMap 
        -- A container for tuning policies which determines how 
        -- and when proxy points are read and written.
        default {[ new BTuningPolicyMap() ]}

      dispatcher: BBasicWorker
        -- The basic communication dispatch queue worker thread.
        -- Its intended use is for synchronizing access to the
        -- communication handler (Comm).
        flags { hidden }
        default {[ new BBasicWorker() ]}

      worker: BBasicWorker
        -- The basic asynchronous queue worker thread.
        -- The default is a BBasicCoalescingWorker.
        -- Its intended use is for posting asynchronous
        -- requests (such as learns), that when processed 
        -- may post one or more message requests to the
        -- dispatcher.
        flags { hidden }
        default {[ new BBasicCoalescingWorker() ]}

      writeWorker: BBasicCoalescingWorker
        -- The basic asynchronous write (coalescing) queue 
        -- worker thread.  Its intended use is for posting
        -- write requests that should be coalesced and then
        -- handed off to the dispatcher.
        flags { hidden }
        default {[ new BBasicCoalescingWorker() ]}

      pollScheduler: BPollScheduler
        -- The basic poll scheduler
        default {[ new BBasicPollScheduler() ]}

      retryCount: int
        -- Specifies the default number of retries to perform after a null response
        -- to a basic message request
        default {[ 1 ]}
      
      responseTimeout: BRelTime
        -- Specifies the default maximum time to wait for a response
        -- after a basic message request before determining a failure
        default {[ BRelTime.make(500) ]}
        slotfacets {[ BFacets.make(BFacets.SHOW_MILLISECONDS, BBoolean.TRUE) ]}
    }
  }

  -*/
/*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
/*@ $com.tridium.basicdriver.BBasicNetwork(2926235078)1.0$ @*/
/* Generated Wed Feb 23 16:36:43 EST 2005 by Slot-o-Matic 2000 (c) Tridium, Inc. 2000 */

////////////////////////////////////////////////////////////////
// Property "tuningPolicies"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>tuningPolicies</code> property.
   * A container for tuning policies which determines how
   * and when proxy points are read and written.
   * @see com.tridium.basicdriver.BBasicNetwork#getTuningPolicies
   * @see com.tridium.basicdriver.BBasicNetwork#setTuningPolicies
   */
  public static final Property tuningPolicies = newProperty(0, new BTuningPolicyMap(),null);
  
  /**
   * Get the <code>tuningPolicies</code> property.
   * @see com.tridium.basicdriver.BBasicNetwork#tuningPolicies
   */
  public BTuningPolicyMap getTuningPolicies() { return (BTuningPolicyMap)get(tuningPolicies); }
  
  /**
   * Set the <code>tuningPolicies</code> property.
   * @see com.tridium.basicdriver.BBasicNetwork#tuningPolicies
   */
  public void setTuningPolicies(BTuningPolicyMap v) { set(tuningPolicies,v,null); }

////////////////////////////////////////////////////////////////
// Property "dispatcher"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>dispatcher</code> property.
   * The basic communication dispatch queue worker thread.
   * Its intended use is for synchronizing access to the
   * communication handler (Comm).
   * @see com.tridium.basicdriver.BBasicNetwork#getDispatcher
   * @see com.tridium.basicdriver.BBasicNetwork#setDispatcher
   */
  public static final Property dispatcher = newProperty(Flags.HIDDEN, new BBasicWorker(),null);
  
  /**
   * Get the <code>dispatcher</code> property.
   * @see com.tridium.basicdriver.BBasicNetwork#dispatcher
   */
  public BBasicWorker getDispatcher() { return (BBasicWorker)get(dispatcher); }
  
  /**
   * Set the <code>dispatcher</code> property.
   * @see com.tridium.basicdriver.BBasicNetwork#dispatcher
   */
  public void setDispatcher(BBasicWorker v) { set(dispatcher,v,null); }

////////////////////////////////////////////////////////////////
// Property "worker"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>worker</code> property.
   * The basic asynchronous queue worker thread. The default
   * is a BBasicCoalescingWorker. Its intended use is for
   * posting asynchronous requests (such as learns), that
   * when processed may post one or more message requests
   * to the dispatcher.
   * @see com.tridium.basicdriver.BBasicNetwork#getWorker
   * @see com.tridium.basicdriver.BBasicNetwork#setWorker
   */
  public static final Property worker = newProperty(Flags.HIDDEN, new BBasicCoalescingWorker(),null);
  
  /**
   * Get the <code>worker</code> property.
   * @see com.tridium.basicdriver.BBasicNetwork#worker
   */
  public BBasicWorker getWorker() { return (BBasicWorker)get(worker); }
  
  /**
   * Set the <code>worker</code> property.
   * @see com.tridium.basicdriver.BBasicNetwork#worker
   */
  public void setWorker(BBasicWorker v) { set(worker,v,null); }

////////////////////////////////////////////////////////////////
// Property "writeWorker"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>writeWorker</code> property.
   * The basic asynchronous write (coalescing) queue worker
   * thread.  Its intended use is for posting write requests
   * that should be coalesced and then handed off to the
   * dispatcher.
   * @see com.tridium.basicdriver.BBasicNetwork#getWriteWorker
   * @see com.tridium.basicdriver.BBasicNetwork#setWriteWorker
   */
  public static final Property writeWorker = newProperty(Flags.HIDDEN, new BBasicCoalescingWorker(),null);
  
  /**
   * Get the <code>writeWorker</code> property.
   * @see com.tridium.basicdriver.BBasicNetwork#writeWorker
   */
  public BBasicCoalescingWorker getWriteWorker() { return (BBasicCoalescingWorker)get(writeWorker); }
  
  /**
   * Set the <code>writeWorker</code> property.
   * @see com.tridium.basicdriver.BBasicNetwork#writeWorker
   */
  public void setWriteWorker(BBasicCoalescingWorker v) { set(writeWorker,v,null); }

////////////////////////////////////////////////////////////////
// Property "pollScheduler"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>pollScheduler</code> property.
   * The basic poll scheduler
   * @see com.tridium.basicdriver.BBasicNetwork#getPollScheduler
   * @see com.tridium.basicdriver.BBasicNetwork#setPollScheduler
   */
  public static final Property pollScheduler = newProperty(0, new BBasicPollScheduler(),null);
  
  /**
   * Get the <code>pollScheduler</code> property.
   * @see com.tridium.basicdriver.BBasicNetwork#pollScheduler
   */
  public BPollScheduler getPollScheduler() { return (BPollScheduler)get(pollScheduler); }
  
  /**
   * Set the <code>pollScheduler</code> property.
   * @see com.tridium.basicdriver.BBasicNetwork#pollScheduler
   */
  public void setPollScheduler(BPollScheduler v) { set(pollScheduler,v,null); }

////////////////////////////////////////////////////////////////
// Property "retryCount"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>retryCount</code> property.
   * Specifies the default number of retries to perform
   * after a null response to a basic message request
   * @see com.tridium.basicdriver.BBasicNetwork#getRetryCount
   * @see com.tridium.basicdriver.BBasicNetwork#setRetryCount
   */
  public static final Property retryCount = newProperty(0, 1,null);
  
  /**
   * Get the <code>retryCount</code> property.
   * @see com.tridium.basicdriver.BBasicNetwork#retryCount
   */
  public int getRetryCount() { return getInt(retryCount); }
  
  /**
   * Set the <code>retryCount</code> property.
   * @see com.tridium.basicdriver.BBasicNetwork#retryCount
   */
  public void setRetryCount(int v) { setInt(retryCount,v,null); }

////////////////////////////////////////////////////////////////
// Property "responseTimeout"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the <code>responseTimeout</code> property.
   * Specifies the default maximum time to wait for a response
   * after a basic message request before determining a
   * failure
   * @see com.tridium.basicdriver.BBasicNetwork#getResponseTimeout
   * @see com.tridium.basicdriver.BBasicNetwork#setResponseTimeout
   */
  public static final Property responseTimeout = newProperty(0, BRelTime.make(500),BFacets.make(BFacets.SHOW_MILLISECONDS, BBoolean.TRUE) );
  
  /**
   * Get the <code>responseTimeout</code> property.
   * @see com.tridium.basicdriver.BBasicNetwork#responseTimeout
   */
  public BRelTime getResponseTimeout() { return (BRelTime)get(responseTimeout); }
  
  /**
   * Set the <code>responseTimeout</code> property.
   * @see com.tridium.basicdriver.BBasicNetwork#responseTimeout
   */
  public void setResponseTimeout(BRelTime v) { set(responseTimeout,v,null); }

////////////////////////////////////////////////////////////////
// Type
////////////////////////////////////////////////////////////////
  
  public Type getType() { return TYPE; }
  public static final Type TYPE = Sys.loadType(BBasicNetwork.class);

/*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/


  /**
   * Register this component under "BBasicNetwork.class".
   */
  public Type[] getServiceTypes() { return new Type[] { getType() }; }

  /**
   * Default constructor.
   */
  public BBasicNetwork()
  {
    super();
  }

  /**
   * Builds the communication handler (Comm).
   */
  public void serviceStarted()
    throws Exception
  {
    buildComm();
  }
  
  /**
   * Service stop.
   */
  public void serviceStopped()
    throws Exception
  {
  }

  /**
   * Initializes the basic network with a new communication handler (Comm)
   * for both the transmit and receive drivers.  After creating
   * a new Comm, this method calls initComm(Comm comm)
   * with the new Comm to allow subclasses to perform any initialization
   * (i.e. adding custom UnsolicitedMessageListeners for handling unsolicited received 
   * messages).  
   */
  private void buildComm()
    throws Exception
  {
    comm = makeComm();
    initComm(comm);
  }
  
  /**
   * This callback should be overridden by subclasses to provide a hook
   * for making any initialization to the communication handler (Comm). 
   * This method should also be where special custom UnsolicitedMessageListeners are 
   * created (i.e. for handling unsolicited received messages) and registered
   * to the Comm.
   */
  protected void initComm(Comm comm)
    throws Exception
  {
    // Here's where listeners should be created.
    // Listeners should register themselves to comm
    // (i.e. for unsolicited message receive, etc.)
  }
  
  /**
   * This method starts the Communication handler
   * (Comm) if the network is not down/fault/out-of-service
   * and the current Comm is not null.
   */
  public void startComm()
    throws Exception
  {
    if((!isDisabled()) && (!isFatalFault()) && (comm != null))
    {
      if (getLog().isTraceOn()) getLog().trace(getName() + " *** Starting Communication Handler ***");
      comm.start();
      if (getLog().isTraceOn()) getLog().trace(getName() + " *** Started Communication Handler ***");
    }
  }
  
  /**
   * This method stops the Communication handler
   * (Comm) if the current Comm is not null.
   */
  public void stopComm()
    throws Exception
  {
    if((comm != null) && (comm.isCommStarted()))
    {
      if (getLog().isTraceOn()) getLog().trace(getName() + " *** Stopping Communication Handler ***");
      comm.stop();
      if (getLog().isTraceOn()) getLog().trace(getName() + " *** Stopped Communication Handler ***");
    }
  }

  /**
   * Start the basic network.  This starts the Communication handler
   * (Comm) if the network is not down/fault/out-of-service.
   */
  public void started()
    throws Exception
  {                 
    super.started();
    try
    {
      startComm();
    }
    catch (Exception e)
    {
      getLog().error("Could not start communication handler", e);
    }
  }
  
  /**
   * Stop the basic network.  This stops the Communication handler (Comm).
   */
  public void stopped()
    throws Exception
  {
    super.stopped();
    try
    {
      stopComm();
    }
    catch (Exception e)
    {
      getLog().error("Could not stop communication handler", e);
    }
  }
  
  /**
   * Returns true if the communication is enabled and ready (the
   * Communication handler is started and the network is not down/fault/out-of-service),
   * returns false if not.
   */
  public boolean isCommActive()
  { 
    boolean commStart = true;
    if(comm != null) commStart = comm.isCommStarted();
    return (commStart && (!isDisabled()) && (!isFault())); 
  }

  /**
   * Returns the communication handler (Comm)
   */
  public Comm getComm() { return comm; }
  
  /**
   * Subclasses must override to return the custom communication
   * handler (Comm)
   */
  protected abstract Comm makeComm();

  /**
   * Network Property Changed.  Check for down/fault/out of service changes,
   * and stop/start the network as appropriate.
   */
  public void changed(Property prop, Context context)
  {
    super.changed(prop, context);
    if (!isRunning()) return;

    try
    {
      if (prop == status)
      {
        if (!isCommActive() && !isDisabled() && !isFatalFault())
        {
          //Start basic comm services
//          if(Sys.atSteadyState())
            startComm();  
        }
        else if ( isDisabled() )
        {
          //Stop basic comm services
          stopComm();
        }
      }
    }
    catch (Exception e)
    {
      getLog().error("BBasicNetwork caught exception in changed(): ", e);
    }
  }
  
  /**
   * Returns the log currently used by this
   * basic network.  If not overridden by subclasses,
   * the default uses the network name to create the log.
   */
  @SuppressWarnings("deprecation")
  public Log getLog()
  {
    return Log.getLog(getName());
  }
  
////////////////////////////////////////////////////////////////
// BLoadableNetwork Overrides
////////////////////////////////////////////////////////////////

  /**
   * Async tasks are pushed onto the asynchronous worker queue by default.  
   */
  public final IFuture postAsync(Runnable r)
  {
    return post(r);
  }
  
////////////////////////////////////////////////////////////////
// Posting to Worker Queues
////////////////////////////////////////////////////////////////

  /**
   * Post the runnable task to the dispatcher comm queue.
   * This queue is intended to synchronize access to the 
   * communication handler (Comm), thus runnable tasks
   * posted here should attempt to transmit a message
   * via Comm and accept a response if any.
   */
  public final IFuture dispatch(Runnable r)
  {
    return getDispatcher().post(r);
  }
  
  /**
   * Post the runnable task to the asynchronous worker queue.
   * This queue is intended to handle asynchronous operations,
   * such as learns, that may also post message requests to 
   * the dispatcher.
   */
  public final IFuture post(Runnable r)
  {
    return getWorker().post(r);
  }
  
  /**
   * Post the runnable task to the coalescing write worker 
   * queue (asynchronous).  This queue is intended to handle
   * asynchronous operations, such as write requests, that
   * should be coalesced.  These operations should post any
   * message requests to the dispatcher.
   */
  public final IFuture postWrite(Runnable r)
  {
    return getWriteWorker().post(r);
  }  

////////////////////////////////////////////////////////////////
// Basic Send Message Request Helper Methods
////////////////////////////////////////////////////////////////

  /**
   * Send the message (by posting a message request to the 
   * communication dispatcher queue) and wait for and return 
   * a response message.  The calling thread will block and
   * wait for the response Message to be returned.
   *
   * @param msg The Message to send.
   */
  public final Message sendSync(Message msg)
  {
    return sendSync(msg, getResponseTimeout(), getRetryCount());
  }
  
  /**
   * Send the message (by posting a message request to the 
   * communication dispatcher queue) and wait for and return 
   * a response message.  The calling thread will block and
   * wait for the response Message to be returned.
   *
   * @param msg The Message to send.
   * @param responseTimeout The timeout to wait for a
   *    response Message to this request, if one is expected.
   * @param retryCount The number of retries to perform if the
   *    Message to send fails to get a response Message (a 
   *    timeout occurs).
   */
  public Message sendSync(Message msg, BRelTime responseTimeout, int retryCount)
  {
    return processSend(msg, responseTimeout, retryCount);
  }

  /**
   * Send the message (by posting an asynchronous request to the 
   * asynchronous worker queue) and don't wait for any
   * response.  Once processed, the MessageListener passed in
   * will be routed any response Message.
   *
   * @param msg The Message to send.
   * @param listener The listener (most likely the source of
   *    the message request) who will receive any
   *    response Message.
   */
  public final void sendAsync(Message msg, MessageListener listener)
  {
    sendAsync(msg, listener, getResponseTimeout(), getRetryCount());
  }
  
  /**
   * Send the message (by posting an asynchronous request to the 
   * asynchronous worker queue) and don't wait for any
   * response.  Once processed, the MessageListener passed in
   * will be routed any response Message.
   *
   * @param msg The Message to send.
   * @param listener The listener (most likely the source of
   *    the message request) who will receive any
   *    response Message.
   * @param responseTimeout The timeout to wait for a
   *    response Message to this request, if one is expected.
   * @param retryCount The number of retries to perform if the
   *    Message to send fails to get a response Message (a 
   *    timeout occurs).
   */
  public void sendAsync(Message msg, MessageListener listener, BRelTime responseTimeout, int retryCount)
  {
    post(new AsyncMessageRequest(this, msg, listener, responseTimeout, retryCount));
  }
  
  /**
   * Send the message, should be a write message, (by posting 
   * an asynchronous request to the asynchronous write worker 
   * queue) and don't wait for any response.  Once processed, 
   * the MessageListener passed in will be routed any response
   * Message.  Coalescing is determined based on the 
   * MessageListener parameter.
   *
   * @param msg The Message to send.
   * @param listener The listener (most likely the source of
   *    the message request) who will receive any
   *    response Message.
   */
  public final void sendAsyncWrite(Message msg, MessageListener listener)
  {
    sendAsyncWrite(msg, listener, getResponseTimeout(), getRetryCount());
  }
  
  /**
   * Send the message, should be a write message, (by posting 
   * an asynchronous request to the asynchronous write worker 
   * queue) and don't wait for any response.  Once processed, 
   * the MessageListener passed in will be routed any response
   * Message.  Coalescing is determined based on the 
   * MessageListener parameter.
   *
   * @param msg The Message to send.
   * @param listener The listener (most likely the source of
   *    the message request) who will receive any
   *    response Message.
   * @param responseTimeout The timeout to wait for a
   *    response Message to this request, if one is expected.
   * @param retryCount The number of retries to perform if the
   *    Message to send fails to get a response Message (a 
   *    timeout occurs).
   */
  public void sendAsyncWrite(Message msg, MessageListener listener, BRelTime responseTimeout, int retryCount)
  {
    postWrite(new AsyncWriteMessageRequest(this, msg, listener, responseTimeout, retryCount));
  }

  /**
   * Process sending a Message and returning the response Message (or null
   * if a response is not expected or a failure (i.e. timeout) occurs).
   * This method actually posts a send message request to the dispatcher
   * queue, so that when processed, the message will be sent to the 
   * communication handler (Comm).
   *
   * @param msg The Message to send
   * @param responseTimeout The timeout to wait for a
   *    response Message to this request, if one is expected.
   * @param retryCount The number of retries to perform if the
   *    Message to send fails to get a response Message (a 
   *    timeout occurs).
   */
  private Message processSend(Message msg, BRelTime responseTimeout, int retryCount)
  {
    if (!isCommActive() || (msg == null)) return null;

    boolean responseExpected = msg.getResponseExpected();
    
    DispatchRequest req = new DispatchRequest(this, msg, responseTimeout, retryCount);

    if (responseExpected)
    {
      dispatch(req);
      // Get response - will block until response received
      return req.getResponse(0);
    }
    else
    {
      dispatch(req);
      return null;
    }
  }

////////////////////////////////////////////////////////////////
// Basic Statistics
////////////////////////////////////////////////////////////////

  /**
   * Adds basic communication statistics to the spy page,
   * such as total messages sent/received and
   * transmission rates.
   */
  public void spy(SpyWriter out)
    throws Exception
  {
    super.spy(out);
    out.startProps();
    out.trTitle("BasicNetwork", 2);
    
    long elapsedTime = Clock.ticks() - lastRateTicks;
    double sentMessagesPerSecond = (((double)(totalSentMessages - lastSentMessages))/ (((double)elapsedTime)/1000.0));
    double receivedMessagesPerSecond = (((double)(totalReceivedMessages - lastReceivedMessages))/ (((double)elapsedTime)/1000.0));
    lastSentMessages = totalSentMessages;
    lastReceivedMessages = totalReceivedMessages;
    lastRateTicks = Clock.ticks();
    
    out.prop("Comm", comm);
    out.prop("Total Sent Messages", new Long(totalSentMessages));
    out.prop("Sent Messages per Second", new Double(sentMessagesPerSecond));
    out.prop("Total Received Messages", new Long(totalReceivedMessages));
    out.prop("Received Messages per Second", new Double(receivedMessagesPerSecond));
    out.prop("Total Failed Messages (timeouts)", new Long(totalTimeoutMessages));
    out.endProps();
  }

  /**
   * This method should be called when a Message
   * was sent out in order to update the basic statistics.
   */
  public final void incrementSent()
  {
    totalSentMessages++;
  }

  /**
   * This method should be called when a Message
   * was received in order to update the basic statistics.
   */
  public final void incrementReceived()
  {
    totalReceivedMessages++;
  }
  
  /**
   * This method should be called when a Message
   * failed to get a response (timeout) in order 
   * to update the basic statistics.
   */
  public final void incrementTimeouts()
  {
    totalTimeoutMessages++;
  }

  public final long getTotalSentMessages() { return totalSentMessages; }
  public final long getTotalReceivedMessages() { return totalReceivedMessages; }
  public final long getTotalTimeoutMessages() { return totalTimeoutMessages; }

////////////////////////////////////////////////////////////////
// Attributes
////////////////////////////////////////////////////////////////

  private Comm comm = null;
  private long totalSentMessages = 0L;
  private long totalReceivedMessages = 0L;
  private long lastSentMessages = 0L;
  private long lastReceivedMessages = 0L;
  private long lastRateTicks = Clock.ticks();
  private long totalTimeoutMessages = 0L;

////////////////////////////////////////////////////////////////
// Inner Classes
////////////////////////////////////////////////////////////////
  
  /**
   * The DispatchRequest class is used to send a Message
   * and receive a Message response.
   * A DispatchRequest is posted directly to the BasicNetwork's
   * dispatcher queue.
   *
   * @author    Scott Hoye
   * @creation  26 Jul 02
   * @version   $Revision: 1$ $Date: 07/26/02 12:47:14 PM$
   * @since     Niagara 3.0 basicdriver 1.0
   */
  private class DispatchRequest
    implements Runnable
  {
    
    /**
     * Constructor
     *
     * @param basicNet The basic network to use for sending the request.
     * @param msg The message to send.
     * @param responseTimeout The timeout to wait for a response to
     *    this request, if one is expected.
     * @param retryCount The number of retries to perform if the request
     *    fails (a timeout occurs).
     */
    public DispatchRequest(BBasicNetwork basicNet, 
                           Message msg, 
                           BRelTime responseTimeout,
                           int retryCount)
    {
      this.basicNet = basicNet;
      this.msg = msg;
      this.responseTimeout = responseTimeout;
      this.retryCount = retryCount;
    }
  
   /**
    * Calls execute().
    */
    public void run()
    {
      execute();
    }
  
   /**
    * Sends the request to the basic network's Comm
    * for distribution out on the communication medium and waits for
    * a response if one is expected.
    */
    synchronized void execute()
    {
      response = null;
      try
      {
        if( msg == null)
        {
          // DispatchRequest received null message to send.  Not sending null message to the Comm
          return;
        }
        if (!(msg.getResponseExpected()))
          basicNet.getComm().transmitNoResponse(msg);
        else
        {
          response = basicNet.getComm().transmit(msg, responseTimeout, retryCount);
        }
      }
      catch (Exception e)
      {
        basicNet.getLog().error("DispatchRequest caught exception in execute(): ", e);
      }
  
      // Notify requester that response has been received.
      complete = true;
      this.notify();
    }
  
   /**
    * Returns the response as soon as it receives it, or returns null
    * after the given timeout (in milliseconds) occurs.
    */
    synchronized Message getResponse(int timeout)
    {
      if(!complete)
      {
        try
        {
          this.wait(timeout);
        }
        catch(Exception e)
        {
          basicNet.getLog().error("DispatchRequest caught exception in getResponse(): ", e);
        }
      }
      return response;
    }
    
  ////////////////////////////////////////////////////////////////
  //  Attributes
  ////////////////////////////////////////////////////////////////
    Message msg;
    Message response = null;
    boolean complete = false;
    BBasicNetwork  basicNet;
    BRelTime responseTimeout;
    int retryCount;
  
  }

  /**
   * AsyncMessageRequest handles a message to get sent 
   * and handles routing the response (if any) to the source listener.
   * An AsyncMessageRequest should be posted to the BasicNetwork's
   * async worker queue.
   *
   * @author    Scott Hoye
   * @creation  12 Feb 04
   * @version   $Revision: 1$ $Date: 2/12/2004 10:03:14 AM$
   * @since     Niagara 3.0 basicdriver 1.0
   */
  private class AsyncMessageRequest
    implements Runnable
  {                    
  
  ////////////////////////////////////////////////////////////////
  // Constructor
  ////////////////////////////////////////////////////////////////
    
    /**
     * Constructor
     *
     * @param source The source of this request who will also receive
     *    the response to the request (if any).
     * @param req The basic synchronous message request to post to the BasicNetwork's
     *    synchronous basic comm queue.
     */
    public AsyncMessageRequest(BBasicNetwork basicNet, 
                               Message msg, 
                               MessageListener source,
                               BRelTime responseTimeout,
                               int retryCount)
    {                   
      this.basicNet = basicNet;
      this.msg = msg;
      this.source = source;
      this.responseTimeout = responseTimeout;
      this.retryCount = retryCount;
    }           
  
  ////////////////////////////////////////////////////////////////
  // Runnable
  ////////////////////////////////////////////////////////////////
    
    /**
     * Run posts the request to the synchronous basic comm queue
     * and routes any response to the source.
     */
    public void run()
    {
      Message response = basicNet.sendSync(msg, responseTimeout, retryCount);
      if ((source != null) && (msg.getResponseExpected()))
        source.processMessage(response);
    }
                   
  ////////////////////////////////////////////////////////////////
  // Attributes
  ////////////////////////////////////////////////////////////////
    
    MessageListener source;
    Message msg;
    BBasicNetwork  basicNet;
    BRelTime responseTimeout;
    int retryCount;
        
  }
  
  /**
   * AsyncWriteMessageRequest handles a message to get sent 
   * (coalesces requests from the same source) and handles
   * routing the response (if any) to the source.
   * An AsyncWriteMessageRequest should be posted to the BasicNetwork's
   * write worker queue (asynchronous).
   *
   * @author    Scott Hoye
   * @creation  12 Feb 04
   * @version   $Revision: 1$ $Date: 2/12/2004 10:03:14 AM$
   * @since     Niagara 3.0 basicdriver 1.0
   */
  private class AsyncWriteMessageRequest
    extends AsyncMessageRequest
    implements ICoalesceable
  {                    
  
  ////////////////////////////////////////////////////////////////
  // Constructor
  ////////////////////////////////////////////////////////////////
    
    /**
     * Constructor
     *
     * @param source The source of this request who will also receive
     *    the response to the request (if any).
     * @param req The basic synchronous message request to post to the BasicNetwork's
     *    synchronous basic comm queue.
     */
    public AsyncWriteMessageRequest(BBasicNetwork basicNet, 
                                    Message msg, 
                                    MessageListener source,
                                    BRelTime responseTimeout,
                                    int retryCount)
    {                    
      super(basicNet, msg, source, responseTimeout, retryCount);
      this.hashCode = source.hashCode();
    }           
  
  ////////////////////////////////////////////////////////////////
  // ICoalesceable
  ////////////////////////////////////////////////////////////////
    
    /**
     * Hash code is based on the source IBasicResponseListener.
     */
    public int hashCode()
    {                             
      return hashCode;
    }             
    
    /**
     * Equality is based on the source IBasicResponseListener.
     */
    public boolean equals(Object object)
    {                                  
      if (object instanceof AsyncWriteMessageRequest)
      {                               
        AsyncWriteMessageRequest o = (AsyncWriteMessageRequest)object;
        return source == o.source;
      }
      return false;
    }
    
    /**
     * Return this.
     */
    public Object getCoalesceKey()
    {
      return this;
    }
    
    /**
     * Return c - last Invocation wins.
     */
    public ICoalesceable coalesce(ICoalesceable c)
    {                                 
      return c;  
    }
                   
  ////////////////////////////////////////////////////////////////
  // Attributes
  ////////////////////////////////////////////////////////////////
    int hashCode;
        
  }

}
