/*
 * Copyright 2015, Tridium, Inc. All Rights Reserved.
 */
package javax.baja.security;

import java.io.IOException;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import javax.baja.sys.BFacets;
import javax.baja.sys.BObject;
import javax.baja.sys.BasicContext;
import javax.baja.sys.Context;
import javax.baja.user.BUser;
import com.tridium.nre.SupplierWithException;
import com.tridium.nre.security.EncryptionAlgorithmBundle;
import com.tridium.nre.security.EncryptionKeySource;
import com.tridium.nre.security.ISecretBytesSupplier;
import com.tridium.nre.security.NiagaraBasicPermission;

/**
 * Context that provides encoding instructions for BPassword values.
 *
 * Callers that construct PasswordEncodingContext values that contain secret information should
 * ensure that they invoke {@link #close()} when the information is no longer needed.  Try-with-
 * resources is a straightforward way to do that.
 *
 * @since Niagara 4.0
 * @author Matt Boon
 * @creation March 5, 2015
 */
public final class PasswordEncodingContext
  implements Context, AutoCloseable
{
////////////////////////////////////////////////////////////////
// Constructor
////////////////////////////////////////////////////////////////

  public PasswordEncodingContext(Context base,
                                 EncryptionKeySource decryptionKeySource,
                                 EncryptionKeySource encryptionKeySource,
                                 Optional<ISecretBytesSupplier>decryptionKey,
                                 Optional<ISecretBytesSupplier>encryptionKey)
  {
    this(base);
    validateKeyCombination("encryption", encryptionKeySource, encryptionKey);
    validateKeyCombination("decryption", decryptionKeySource, decryptionKey);
    this.encryptionKeySource = encryptionKeySource;
    this.decryptionKeySource = decryptionKeySource;
    this.decryptionKey = decryptionKey;
    this.encryptionKey = encryptionKey;
  }

  public PasswordEncodingContext(Context base)
  {
    this.base = base == null ? new BasicContext() : base;

    this.decryptionKeySource = EncryptionKeySource.keyring;
    this.encryptionKeySource = EncryptionKeySource.keyring;
  }

////////////////////////////////////////////////////////////////
// Utility
////////////////////////////////////////////////////////////////

  /**
   * Return a new context with no base which can be used for encoding BPassword values with
   * a key from the NRE security provider's keyring
   */
  public static PasswordEncodingContext makeKeyring()
  {
    return new PasswordEncodingContext(
        null,
        EncryptionKeySource.keyring,
        EncryptionKeySource.keyring,
        Optional.empty(),
        Optional.empty());
  }

  /**
   * Return a new context with no base which cannot be used for encoding BPassword values
   */
  public static PasswordEncodingContext makeNone()
  {
    return new PasswordEncodingContext(
      null,
      EncryptionKeySource.none,
      EncryptionKeySource.none,
      Optional.empty(),
      Optional.empty());
  }

  /**
   * Return either a PasswordEncodingContext or a Context whose base is one, and
   * update the PasswordEncodingContext using the given consumer.
   */
  public static Context updateContext(Context from, Consumer<PasswordEncodingContext> consumer)
  {
    Objects.requireNonNull(consumer);
    Context toCheck = from;
    while (toCheck != null)
    {
      if (toCheck instanceof PasswordEncodingContext)
      {
        consumer.accept((PasswordEncodingContext)toCheck);
        return from;
      }
      toCheck = toCheck.getBase();
    }
    PasswordEncodingContext result = new PasswordEncodingContext(from);
    consumer.accept(result);
    return result;
  }

  /**
   * Return a context suitable for encoding with the local keyring
   */
  public static Context updateForKeyring(Context from)
  {
    return updateContext(from, pContext ->
      pContext.setEncryptionAndDecryptionKey(EncryptionKeySource.keyring, Optional.empty()));
  }

  /**
   * Return a context which will allow no password values to be encoded with a reversible encryption.
   */
  public static Context updateForNone(Context from)
  {
    // This is safe to call in privileged mode, as it's telling the context to never encrypt or decrypt
    // a password.  The other update methods could be used to reveal passwords or at least encode them
    // with a malicious caller's key (so they could be revealed immediately afterward), so they all
    // need to run with security manager checks in place.
    return AccessController.doPrivileged((PrivilegedAction<Context>)() -> updateContext(from, pContext ->
      pContext.setEncryptionAndDecryptionKey(EncryptionKeySource.none, Optional.empty())));
  }

  /**
   * Return a context suitable for encoding with the given external key
   */
  public static Context updateForExternal(Context from, ISecretBytesSupplier keySupplier)
  {
    Objects.requireNonNull(keySupplier);
    return updateContext(from, pContext ->
      pContext.setEncryptionAndDecryptionKey(EncryptionKeySource.external, Optional.of(keySupplier)));
  }

  /**
   * If the given Context or any of its bases is a PasswordEncodingContext, return it, otherwise
   * return a new PasswordEncodingContext that uses the given context as a base.
   */
  public static PasswordEncodingContext from(Context from)
  {
    Context toCheck = from;
    while (toCheck != null)
    {
      if (toCheck instanceof PasswordEncodingContext)
      {
        return (PasswordEncodingContext)toCheck;
      }
      toCheck = toCheck.getBase();
    }
    return new PasswordEncodingContext(from);
  }

////////////////////////////////////////////////////////////////
// Encryption and Decryption Combined
////////////////////////////////////////////////////////////////

  public PasswordEncodingContext setEncryptionAndDecryptionKey(EncryptionKeySource source)
  {
    checkClosed();
    return setEncryptionAndDecryptionKey(source, Optional.empty());
  }

  public PasswordEncodingContext setEncryptionAndDecryptionKey(EncryptionKeySource source, Optional<ISecretBytesSupplier> key)
  {
    checkClosed();
    validateKeyCombination("encryption/decryption", source, key);
    encryptionKeySource = source;
    this.encryptionKey.ifPresent(ISecretBytesSupplier::close);
    this.decryptionKey.ifPresent(ISecretBytesSupplier::close);
    if (key.isPresent())
    {
      this.encryptionKey = Optional.of(key.get().newCopy());
      this.decryptionKey = Optional.of(key.get().newCopy());
    }
    else
    {
      this.encryptionKey = key;
      this.decryptionKey = key;
    }
    decryptionKeySource = source;
    return this;
  }

////////////////////////////////////////////////////////////////
// Decryption
////////////////////////////////////////////////////////////////

  public Optional<ISecretBytesSupplier>getDecryptionKey()
  {
    checkClosed();
    SecurityManager sm = System.getSecurityManager();
    if (sm != null) sm.checkPermission(GET_CONTEXT_KEY_PERMISSION);
    return decryptionKey;
  }

  public boolean hasDecryptionKey()
  {
    checkClosed();
    return decryptionKey.isPresent();
  }
  
  public EncryptionKeySource getDecryptionKeySource()
  {
    checkClosed();
    return decryptionKeySource;
  }

  public PasswordEncodingContext setDecryptionKey(EncryptionKeySource source, Optional<ISecretBytesSupplier> key)
  {
    checkClosed();
    validateKeyCombination("decryption", source, key);
    decryptionKeySource = source;
    this.decryptionKey.ifPresent(ISecretBytesSupplier::close);
    if (key.isPresent())
    {
      this.decryptionKey = Optional.of(key.get().newCopy());
    }
    else
    {
      this.decryptionKey = key;
    }
    return this;
  }

  public PasswordEncodingContext setDecryptionUndefined()
  {
    checkClosed();
    SecurityManager sm = System.getSecurityManager();
    if (sm != null)
    {
      sm.checkPermission(SET_CONTEXT_KEY_PERMISSION);
    }
    decryptionKeySource = EncryptionKeySource.undefined;
    decryptionKey.ifPresent(ISecretBytesSupplier::close);
    decryptionKey = Optional.empty();
    return this;
  }

////////////////////////////////////////////////////////////////
// Encryption
////////////////////////////////////////////////////////////////

  public Optional<ISecretBytesSupplier>getEncryptionKey()
  {
    checkClosed();
    SecurityManager sm = System.getSecurityManager();
    if (sm != null) sm.checkPermission(GET_CONTEXT_KEY_PERMISSION);
    return encryptionKey;
  }

  public boolean hasEncryptionKey()
  {
    checkClosed();
    return encryptionKey.isPresent();
  }
  
  public EncryptionKeySource getEncryptionKeySource()
  {
    checkClosed();
    return encryptionKeySource;
  }

  public PasswordEncodingContext setEncryptionKey(EncryptionKeySource source, Optional<ISecretBytesSupplier> key)
  {
    checkClosed();
    validateKeyCombination("encryption", source, key);
    encryptionKeySource = source;
    this.encryptionKey.ifPresent(ISecretBytesSupplier::close);
    if (key.isPresent())
    {
      this.encryptionKey = Optional.of(key.get().newCopy());
    }
    else
    {
      this.encryptionKey = key;
    }
    return this;
  }

  public PasswordEncodingContext setEncryptionUndefined()
  {
    checkClosed();
    SecurityManager sm = System.getSecurityManager();
    if (sm != null)
    {
      sm.checkPermission(SET_CONTEXT_KEY_PERMISSION);
    }
    encryptionKeySource = EncryptionKeySource.undefined;
    encryptionKey.ifPresent(ISecretBytesSupplier::close);
    encryptionKey = Optional.empty();
    return this;
  }

  /**
   * Set the EncryptionAlgorithmBundle to use for encoding
   *
   * @param algorithmBundle the algorithm bundle to use for encoding
   * @return this PasswordEncodingContext
   * @since Niagara 4.9
   */
  public PasswordEncodingContext setEncryptionAlgorithmBundle(EncryptionAlgorithmBundle algorithmBundle)
  {
    this.algorithmBundle = Optional.ofNullable(algorithmBundle);
    return this;
  }

  /**
   * Get the EncryptionAlgorithmBundle to use for encoding, or Optional.empty() if not set
   *
   * @return the EncryptionAlgorihtmBundle to use for encoding, or empty
   * @since Niagara 4.9
   */
  public Optional<EncryptionAlgorithmBundle> getEncryptionAlgorithmBundle()
  {
    return algorithmBundle;
  }

////////////////////////////////////////////////////////////////
// Encoding Errors
////////////////////////////////////////////////////////////////

  /**
   * Configure to throw {@link MissingEncodingKeyException} whenever password decoding fails
   * due to missing or invalid key
   */
  public PasswordEncodingContext throwExceptionOnDecodeFailures()
  {
    return setDecodeFailureSupplier(EXCEPTION_DECODE_FAILURE_SUPPLIER);
  }

  /**
   * Configure to return {@link BPassword#DEFAULT} whenever password decoding fails
   * due to missing or invalid key
   */
  public PasswordEncodingContext defaultOnDecodeFailures()
  {
    return setDecodeFailureSupplier(DEFAULT_DECODE_FAILURE_SUPPLIER);
  }

  /**
   * Configure to take a particular action whenever password decoding fails due to missing or
   * invalid key
   */
  public PasswordEncodingContext setDecodeFailureSupplier(SupplierWithException<BPassword, MissingEncodingKeyException> value)
  {
    Objects.requireNonNull(value);
    decodeFailureSupplier = value;
    return this;
  }

  /**
   * return a password value or throw {@link MissingEncodingKeyException} as appropriate for
   * a decode failure
   */
  public BPassword getDecodeErrorPasswordValue()
    throws MissingEncodingKeyException
  {
    return decodeFailureSupplier.get();
  }

  /**
   * Configure to return encoded {@link BPassword#DEFAULT} whenever password encoding fails
   * due to missing or invalid key
   */
  public PasswordEncodingContext defaultOnEncodeFailures()
  {
    return setEncodeFailureSupplier(DEFAULT_ENCODE_FAILURE_SUPPLIER);
  }

  /**
   * Configure to throw {@link MissingEncodingKeyException} whenever password encoding fails
   * due to missing or invalid key
   */
  public PasswordEncodingContext throwExceptionOnEncodeFailures()
  {
    return setEncodeFailureSupplier(EXCEPTION_ENCODE_FAILURE_SUPPLIER);
  }

  /**
   * Configure to take a particular action whenever password encoding fails due to missing or
   * invalid key
   */
  public PasswordEncodingContext setEncodeFailureSupplier(SupplierWithException<String,MissingEncodingKeyException> value)
  {
    Objects.requireNonNull(value);
    encodeFailureSupplier = value;
    return this;
  }

  /**
   * return an encoded password value or throw {@link MissingEncodingKeyException} as appropriate for
   * a decode failure
   */
  public String getErrorEncodedValue()
    throws MissingEncodingKeyException
  {
    return encodeFailureSupplier.get();
  }

////////////////////////////////////////////////////////////////
// Context
////////////////////////////////////////////////////////////////

  @Override
  public Context getBase()
  {
    return base;
  }

  @Override
  public BUser getUser()
  {
    return base.getUser();
  }

  @Override
  public BFacets getFacets()
  {
    return base.getFacets();
  }

  @Override
  public BObject getFacet(String name)
  {
    return base.getFacet(name);
  }

  @Override
  public String getLanguage()
  {
    return base.getLanguage();
  }

////////////////////////////////////////////////////////////////
// AutoCloseable
////////////////////////////////////////////////////////////////

  @Override
  public void close()
  {
    decryptionKey.ifPresent(ISecretBytesSupplier::close);
    encryptionKey.ifPresent(ISecretBytesSupplier::close);
    decryptionKey = Optional.empty();
    encryptionKey = Optional.empty();
    decryptionKeySource = EncryptionKeySource.none;
    encryptionKeySource = EncryptionKeySource.none;
    isClosed = true;
  }

////////////////////////////////////////////////////////////////
// Private
////////////////////////////////////////////////////////////////

  private void checkClosed()
  {
    if (isClosed)
    {
      throw new IllegalStateException("Cannot perform operation on a PasswordEncodingContext that has been closed");
    }
  }

  private static void validateKeyCombination(String kind, EncryptionKeySource source, Optional<ISecretBytesSupplier> key)
  {
    Objects.requireNonNull(kind);
    Objects.requireNonNull(source);
    Objects.requireNonNull(key);
    if (key.isPresent())
    {
      switch(source)
      {
        case keyring:
        case none:
        case undefined:
          throw new IllegalStateException("Cannot set " + kind + " key for EncryptionKeySource of " + source.name());
      }
      SecurityManager sm = System.getSecurityManager();
      if (sm != null) sm.checkPermission(SET_CONTEXT_KEY_PERMISSION);
    }
  }

////////////////////////////////////////////////////////////////
// Attributes
////////////////////////////////////////////////////////////////
  
  private final Context base;
  private boolean isClosed = false;
  private EncryptionKeySource decryptionKeySource = EncryptionKeySource.none;
  private EncryptionKeySource encryptionKeySource = EncryptionKeySource.none;
  private Optional<ISecretBytesSupplier> decryptionKey = Optional.empty();
  private Optional<ISecretBytesSupplier> encryptionKey = Optional.empty();
  private Optional<EncryptionAlgorithmBundle> algorithmBundle = Optional.empty();
  private SupplierWithException<BPassword,MissingEncodingKeyException> decodeFailureSupplier = EXCEPTION_DECODE_FAILURE_SUPPLIER;
  private SupplierWithException<String,MissingEncodingKeyException> encodeFailureSupplier = EXCEPTION_ENCODE_FAILURE_SUPPLIER;

  private static final NiagaraBasicPermission GET_CONTEXT_KEY_PERMISSION = new NiagaraBasicPermission("GET_CONTEXT_KEY");
  private static final NiagaraBasicPermission SET_CONTEXT_KEY_PERMISSION = new NiagaraBasicPermission("SET_CONTEXT_KEY");

  public static final SupplierWithException<BPassword,MissingEncodingKeyException> EXCEPTION_DECODE_FAILURE_SUPPLIER =
    () -> { throw new MissingEncodingKeyException(); };

  public static final SupplierWithException<BPassword,MissingEncodingKeyException> DEFAULT_DECODE_FAILURE_SUPPLIER =
    () -> BPassword.DEFAULT;

  public static final SupplierWithException<String,MissingEncodingKeyException> EXCEPTION_ENCODE_FAILURE_SUPPLIER =
    () -> { throw new MissingEncodingKeyException(); };

  public static final SupplierWithException<String,MissingEncodingKeyException> DEFAULT_ENCODE_FAILURE_SUPPLIER =
    () ->
      {
        try
        {
          return BPassword.DEFAULT.encodeToString();
        }
        catch (IOException wontHappenForDefault)
        {
          return null;
        }
      };
}
