/*
 * Copyright (c) 2017 Tridium, Inc. All Rights Reserved.
 */

package javax.baja.security;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.security.cert.X509Certificate;
import java.util.Objects;
import javax.baja.nre.annotations.NiagaraType;
import javax.baja.nre.annotations.NoSlotomatic;
import javax.baja.rpc.NiagaraRpc;
import javax.baja.rpc.Transport;
import javax.baja.rpc.TransportType;
import javax.baja.sys.BObject;
import javax.baja.sys.BSimple;
import javax.baja.sys.Context;
import javax.baja.sys.LocalizableException;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import javax.baja.util.LexiconModule;

import com.tridium.crypto.core.cert.NX509Certificate;

/**
 * BX509Certificate is a BSimple container for a standard X509Certificate.
 *
 * In at least one case (e.g. com.tridium.saml.idp.BStationServiceProvider#UNINITIALIZED,
 * as of Niagara 4.10u2) distinct instances of this class need to be maintained where
 * the private "cert" field is null.  Because of this restriction, "BSimple interning"
 * should not be used with this class.
 *
 * @author Bill Smith on 1/13/2017
 * @since Niagara 4.5
 */

@NiagaraType
@NoSlotomatic
public final class BX509Certificate
  extends BSimple
{
  /**
   * Creates a BX509Certificate from a supplied X509Certificate.
   *
   * @since Niagara 4.5
   */
  public static BX509Certificate make(X509Certificate cert)
  {
    return new BX509Certificate(cert);
  }

  /**
   * Creates a BX509Certificate from a supplied PEM encoded certificate string.
   *
   * @since Niagara 4.5
   */
  public static BX509Certificate make(String encoded)
  {
    if (encoded == null || encoded.isEmpty())
    {
      return make((X509Certificate) null);
    }

    try
    {
      return make(NX509Certificate.decodeFromString(encoded));
    }
    catch(Exception e)
    {
      throw new IllegalArgumentException("invalid source provided", e);
    }
  }

  /**
   * Private constructor.
   * @since Niagara 4.5
   */
  private BX509Certificate(X509Certificate cert)
  {
    this.cert = cert;
  }

  /**
   * Retrieves the contained X509 certificate or null.
   *
   * @since Niagara 4.5
   */
  public X509Certificate getX509Certificate()
  {
    return cert;
  }

  public String getUsername()
  {
    return extractUserName(cert);
  }

  public static String extractUserName(X509Certificate cert)
  {
    return NX509Certificate.extractCommonName(cert.getSubjectX500Principal());
  }

  /**
   * Is the contained X509 certificate null?
   *
   * @since Niagara 4.9
   */
  public boolean isNull()
  {
    return cert == null;
  }

  /**
   * BX509Certificate is encoded as using {@link java.io.DataOutput#writeUTF(java.lang.String)}.
   *
   * @since Niagara 4.5
   */
  @Override
  public void encode(DataOutput encoder)
    throws IOException
  {
    encoder.writeUTF(encodeToString());
  }

  /**
   * BX509Certificate is decoded using {@link java.io.DataInput#readUTF()}.
   *
   * @since Niagara 4.5
   */
  @Override
  public BObject decode(DataInput decoder)
    throws IOException
  {
    return decodeFromString(decoder.readUTF());
  }

  /**
   * Returns a string encoded representation of this BX509Certificate object.
   *
   * @since Niagara 4.5
   */
  @Override
  public String encodeToString()
    throws IOException
  {
    if (cert == null)
    {
      return "";
    }

    try
    {
      return NX509Certificate.encodeToString(cert);
    }
    catch (Exception e)
    {
      throw new IOException("invalid source provided", e);
    }
  }

  /**
   * Returns BX509Certificate object decoded from the provided string.
   *
   * @since Niagara 4.5
   */
  @Override
  public BObject decodeFromString(String s)
    throws IOException
  {
    try
    {
      return make(s);
    }
    catch (Exception e)
    {
      throw new IOException("invalid source provided", e);
    }
  }

  /**
   * BX509Certificate equality is based on the equality of the contained
   * certificate.
   *
   * @since Niagara 4.5
   */
  @Override
  public boolean equals(Object obj)
  {
    if (obj == null)
    {
      return false;
    }

    if (obj instanceof BX509Certificate)
    {
      BX509Certificate other = (BX509Certificate) obj;
      return Objects.equals(cert, other.cert);
    }
    return false;
  }

  /**
   * BX509Certificate uses the hash code of the the contained certificate
   * or zero if null.
   *
   * @since Niagara 4.5
   */
  @Override
  public int hashCode()
  {
    return cert != null ? cert.hashCode() : 0;
  }

  @NiagaraRpc(
    permissions="unrestricted",
    transports = {
      @Transport(type = TransportType.box)}
  )
  public static final String getSubjectDn(String certificate, Context context)
    throws LocalizableException
  {
    if(certificate == null || certificate.isEmpty())
    {
      // since the user hasn't selected a cert, there is no need to act on it
      return LexiconModule.make("baja").get("x509Certificate.noCertificate", context);
    }

    try
    {
      return make(certificate).getX509Certificate().getSubjectDN().toString();
    }
    catch(Exception e)
    {
      throw new LocalizableException("baja", "x509Certificate.getSubjectDn.error", e);
    }
  }

////////////////////////////////////////////////////////////////
// Type
////////////////////////////////////////////////////////////////

  public static final BX509Certificate DEFAULT = new BX509Certificate(null);

  @Override
  public Type getType() { return TYPE; }
  public static final Type TYPE = Sys.loadType(BX509Certificate.class);

////////////////////////////////////////////////////////////////
// Attributes
////////////////////////////////////////////////////////////////

  private X509Certificate cert;
}
