/*
 * Copyright 2015 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.security;

import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.concurrent.atomic.AtomicReference;
import javax.baja.authn.BAuthenticationScheme;
import javax.baja.authn.BPasswordAuthenticationScheme;
import javax.baja.nre.annotations.NiagaraProperty;
import javax.baja.nre.annotations.NiagaraSlots;
import javax.baja.nre.annotations.NiagaraType;
import javax.baja.space.BComponentSpace;
import javax.baja.sys.BComplex;
import javax.baja.sys.BComponent;
import javax.baja.sys.BFacets;
import javax.baja.sys.BValue;
import javax.baja.sys.BasicContext;
import javax.baja.sys.Context;
import javax.baja.sys.Flags;
import javax.baja.sys.IPropertyValidator;
import javax.baja.sys.Localizable;
import javax.baja.sys.LocalizableRuntimeException;
import javax.baja.sys.Property;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import javax.baja.sys.Validatable;
import javax.baja.user.BPasswordStrength;
import javax.baja.user.BUser;
import javax.baja.user.BUserService;
import javax.baja.util.CannotValidateException;

import com.tridium.nre.security.SecretChars;
import com.tridium.sys.schema.Fw;
import com.tridium.user.BUserPasswordConfiguration;

/**
 * @author Tom Duffy
 * @creation 8/15/2014
 * @since Niagara 4.0
 */
@NiagaraType
@NiagaraSlots(
  properties = {
    @NiagaraProperty(name = "passwordConfig", type="baja:UserPasswordConfiguration", defaultValue = "new BUserPasswordConfiguration()")
  }
)
public class BPasswordAuthenticator
  extends BPasswordCache
{
  //Overwrite the flags
  public static final Property password = newProperty(Flags.OPERATOR, BPassword.DEFAULT, BFacets.make(BFacets.make(BFacets.FIELD_EDITOR,"wbutil:UserPasswordFE"),BFacets.make(BFacets.UX_FIELD_EDITOR,"webEditors:UserPasswordEditor")));
/*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
/*@ $javax.baja.security.BPasswordAuthenticator(1863563759)1.0$ @*/
/* Generated Fri Apr 10 13:37:44 EDT 2015 by Slot-o-Matic (c) Tridium, Inc. 2012 */

////////////////////////////////////////////////////////////////
// Property "passwordConfig"
////////////////////////////////////////////////////////////////
  
  /**
   * Slot for the {@code passwordConfig} property.
   * @see #getPasswordConfig
   * @see #setPasswordConfig
   */
  public static final Property passwordConfig = newProperty(0, new BUserPasswordConfiguration(),null);
  
  /**
   * Get the {@code passwordConfig} property.
   * @see #passwordConfig
   */
  public BUserPasswordConfiguration getPasswordConfig() { return (BUserPasswordConfiguration)get(passwordConfig); }
  
  /**
   * Set the {@code passwordConfig} property.
   * @see #passwordConfig
   */
  public void setPasswordConfig(BUserPasswordConfiguration v) { set(passwordConfig,v,null); }

////////////////////////////////////////////////////////////////
// Type
////////////////////////////////////////////////////////////////
  
  @Override
  public Type getType() { return TYPE; }
  public static final Type TYPE = Sys.loadType(BPasswordAuthenticator.class);

/*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/
  public BPasswordAuthenticator(){}

  public BPasswordAuthenticator(BPassword value)
  {
    setPassword(value);
  }

  ////////////////////////////////////////////////////////////////
// Component
////////////////////////////////////////////////////////////////

  @Override
  public final Object fw(int x, Object a, Object b, Object c, Object d)
  {
    switch(x)
    {
      case Fw.CHANGED:
        BComponentSpace space = getComponentSpace();
        if (space == null || space.fireDirectCallbacks())
        {
          fwChanged((Property)a, (Context)b);

        }
        break;
    }
    return super.fw(x, a, b, c, d);
  }

  @Override
  public void started()
  {
    if (isInUserService())
    {
      convertToPbkdf2Password();
    }
  }

  @Override
  public void changed(Property property, Context context)
  {
    if (!isRunning())
    {
      return;
    }
    if (property == passwordConfig)
    {
      BComplex parent = getParent();
      if (parent instanceof BComponent)
      {
        parent.asComponent().changed(getPropertyInParent(), context);
      }

    }
    if(property.getType() == BPassword.TYPE){
      invalidUserSessions((BUser) getParent());
    }
  }

  private void fwChanged(Property prop, Context cx)
  {
    if (isInUserService())
    {
      if (password.equals(prop) && cx != pwConverted)
      {
        // if using pbkdf2, convert the users password
        convertToPbkdf2Password();
      }
    }
  }

  @Override
  public IPropertyValidator getPropertyValidator(Property property, Context context)
  {
    if (property == password)
    {
      return new IPropertyValidator()
      {
        @Override
        public void validateSet(Validatable validatable, Context context)
        {
          if (context != pwConverted)
          {
            checkPassword((BPassword) validatable.getProposedValue(password), context);
          }
        }
      };
    }
    return null;
  }

  /**
   * Check if the specified password meets password requirements. Checks will
   * only be performed on reversible passwords, calls with an irreversible password
   * will skip chacks and return successfully.
   *
   * @param newPassword the new password to check, must be reversible
   * @param context the context
   * @throws LocalizableRuntimeException if validation fails
   * @throws CannotValidateException if validation fails
   * @since Niagara 4.10
   */
  public void checkPassword(BPassword newPassword, Context context)
  {
    BComplex parent = getParent();
    if (parent instanceof BUser)
    {
      BUser user = (BUser)parent;
      // lease to depth of user password config
      user.lease(1);
      BAuthenticationScheme scheme = user.getAuthenticationScheme();
      if(scheme instanceof BPasswordAuthenticationScheme)
      {
        // lease to depth of global password config
        scheme.lease(1);
        checkPassword(user, (BPasswordAuthenticationScheme) scheme, getPasswordConfig(), newPassword, context);
      }
    }
  }

  /**
   * Check if the specified password meets password requirements. Checks will
   * only be performed on reversible passwords, calls with an irreversible password
   * will skip chacks and return successfully.
   *
   * @param user the user the password is for, or null if a new user
   * @param scheme the authentication scheme for the user, or null if not available
   * @param config the users UserPasswordConfiguration, or null if not available
   * @param newPassword the new password to check, must be reversible
   * @param context the context
   * @throws LocalizableRuntimeException if validation fails
   * @throws CannotValidateException if validation fails
   * @since Niagara 4.10
   */
  public static void checkPassword(BUser user,
                                   BPasswordAuthenticationScheme scheme,
                                   BUserPasswordConfiguration config,
                                   BPassword newPassword,
                                   Context context)
  {
    if (!newPassword.getPasswordEncoder().isReversible())
    {
      return;
    }

    try
    {
      BPasswordStrength strength = scheme.getGlobalPasswordConfiguration().getPasswordStrength();
      try (SecretChars passChars = AccessController.doPrivileged((PrivilegedAction<SecretChars>) newPassword::getSecretChars))
      {
        AtomicReference<Localizable> messageRef = new AtomicReference<>();
        if (!strength.isPasswordValid(passChars.get(), messageRef::set))
        {
          throw new LocalizableRuntimeException("baja", messageRef.get().toString(context));
        }
      }
      if (user != null && scheme.isDuplicatePassword(newPassword, user))
      {
        throw new LocalizableRuntimeException("baja", "user.strongPassword.alreadyUsed");
      }
      if (config != null)
      {
        config.changeIntervalCheck(scheme.getGlobalPasswordConfiguration());
      }
    }
    catch (LocalizableRuntimeException e)
    {
      throw e;
    }
    catch (Exception e)
    {
      throw new CannotValidateException(e);
    }
  }

  private boolean isInUserService()
  {
    if(getParent() != null)
    {
      BValue parent = getParent().getParent();
      return parent instanceof BUserService;
    }
    else
    {
      return false;
    }
  }

  // expired password should no longer keep a user from logging in, but they will be forced
  // to change their password. Return true, expiration will be checked in Tuner/NiagaraAuthenticator
  @Override
  public boolean canLogin()
  {
    return true;
  }

  public boolean convertToPbkdf2Password()
  {
    BPassword oldPw = getPassword();
    if (oldPw.getPasswordEncoder().isReversible())
    {
      getPasswordConfig().passwordModified();
      BPassword newPw = BPassword.make(AccessController.doPrivileged((PrivilegedAction<String>)oldPw::getValue), BPbkdf2HmacSha256PasswordEncoder.ENCODING_TYPE);
      set("password", newPw, pwConverted);
      return true;
    }
    return false;
  }

  public static final Context pwConverted = new BasicContext()
  {
    @Override
    public boolean equals(Object obj) { return this == obj; }
    @Override
    public int hashCode() { return toString().hashCode(); }
    @Override
    public String toString() { return "Context.pwConverted"; }
  };

  public static final BPasswordAuthenticator DEFAULT = new BPasswordAuthenticator();
}
