/*
 * Copyright 2018 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.security;

import java.util.Objects;
import javax.baja.nre.annotations.AgentOn;
import javax.baja.nre.annotations.NiagaraType;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import javax.baja.util.Version;
import com.tridium.nre.security.Aes256PasswordManager;
import com.tridium.nre.security.AliasedAesAlgorithmBundle;
import com.tridium.nre.security.EncryptionAlgorithmBundle;
import com.tridium.nre.security.KeyRingPermission;

/**
 * Password encoder that encrypts with a key of a given alias.
 *
 * @author Patrick sager
 * @creation 1/2/2018
 * @since Niagara 4.6
 */
@NiagaraType(agent = @AgentOn(types = {"baja:Password"}))
public class BAliasedAes256PasswordEncoder
  extends BAbstractAes256PasswordEncoder
{
/*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
/*@ $javax.baja.security.BAliasedAes256PasswordEncoder(119196451)1.0$ @*/
/* Generated Tue Jan 02 15:01:55 EST 2018 by Slot-o-Matic (c) Tridium, Inc. 2012 */

////////////////////////////////////////////////////////////////
// Type
////////////////////////////////////////////////////////////////

  @Override
  public Type getType()
  {
    return TYPE;
  }

  public static final Type TYPE = Sys.loadType(BAliasedAes256PasswordEncoder.class);

/*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

////////////////////////////////////////////////////////////////
// Constructor
////////////////////////////////////////////////////////////////

  public BAliasedAes256PasswordEncoder()
  {
    // default key alias
    keyAlias = Aes256PasswordManager.DEFAULT_AES_KEY_ALIAS;
  }

  public BAliasedAes256PasswordEncoder(String keyAlias)
  {
    this.keyAlias = keyAlias;
  }

////////////////////////////////////////////////////////////////
// BAbstractPasswordEncoder
////////////////////////////////////////////////////////////////

  @Override
  public void parse(String key) throws Exception
  {
    try
    {
      String[] data = getAlgorithmBundle().decode(key);
      Objects.requireNonNull(data);
      keyAlias = data[AliasedAesAlgorithmBundle.ALIAS_INDEX];
      iv = data[AliasedAesAlgorithmBundle.IV_INDEX];
      cipher = data[AliasedAesAlgorithmBundle.CIPHER_INDEX];
    }
    catch (IllegalArgumentException iae)
    {
      // Couldn't parse it, maybe it's not actually encoded
      encode(key);
    }
  }

  @Override
  public String getEncodingType()
  {
    return ENCODING_TYPE;
  }

  @Override
  public String getEncodedValue()
  {
    String[] data = new String[getAlgorithmBundle().getDataElementCount()];
    data[AliasedAesAlgorithmBundle.ALIAS_INDEX] = keyAlias;
    data[AliasedAesAlgorithmBundle.CIPHER_INDEX] = cipher;
    data[AliasedAesAlgorithmBundle.IV_INDEX] = iv;
    return getAlgorithmBundle().encode(data);
  }

  @Override
  protected EncryptionAlgorithmBundle getAlgorithmBundle()
  {
    return ALGORITHM_BUNDLE;
  }

  /**
   * Set the alternate owner of this password encoder. Any caller with KeyRingPermission on the alias
   * of the alternate owner is allowed to decrypt the password even if they do not have the KeyRingPermission
   * for the alias that this password is encrypted with. The caller of setAlternateOwner() must have
   * the KeyRingPermission for the alias that this password is encrypted with.
   *
   * @param alternateOwner the alternate owner
   * @since Niagara 4.6
   */
  public void setAlternateOwner(String alternateOwner)
  {
    SecurityManager sm = System.getSecurityManager();
    if (sm != null)
    {
      sm.checkPermission(new KeyRingPermission(keyAlias));
    }
    this.alternateOwner = alternateOwner;
  }

  @Override
  public String getAlternateOwner()
  {
    return alternateOwner;
  }

  private String alternateOwner;

////////////////////////////////////////////////////////////////
// Constants
////////////////////////////////////////////////////////////////
  public static final AliasedAesAlgorithmBundle ALGORITHM_BUNDLE = AliasedAesAlgorithmBundle.make(256);
  public static final String ENCODING_TYPE = ALGORITHM_BUNDLE.getAlgorithmName();

  public static final Version MIN_VERSION = new Version("4.6.0");
}