/*
 * Copyright 2017 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.naming;

import java.util.ArrayList;
import java.util.Arrays;

/**
 * @author Eric Anderson
 * @creation 6/10/2017
 * @since Niagara 4.3U1
 */
public final class OrdUtil
{
  // private constructor
  private OrdUtil()
  {
  }

  /**
   * Slot and file path ord queries may contain "../" to do relative traversal up the tree.  If
   * there is more than one backup, the ord will contain "/../", which will be replaced by the
   * browser within a URL by removing other sections.  For example, https://127.0.0.1/a/b/c/d/../e/f
   * is converted to https://127.0.0.1/a/b/c/e/f and https://127.0.0.1/a/b/c/d/../../e/f is
   * converted to https://127.0.0.1/a/b/c/f.  This will result in unintended behavior in subsequent
   * ord resolution with that URL.  Therefore, all but the last "../" is replaced with {@code
   * "<schema>:..|"}.  This function is replicated in BajaScript by Ord.replaceBackups.
   *
   * @param ord ord that is searched for "../" backups
   * @return the original ord if no changes are necessary or an updated ord with the necessary
   * replacements
   */
  public static BOrd replaceBackups(BOrd ord)
  {
    OrdQuery[] queries = ord.parse();

    // This list is only instantiated if any replacements are necessary.
    ArrayList<OrdQuery> newQueries = null;
    for (int i = 0; i < queries.length; ++i)
    {
      int backupDepth = queries[i] instanceof Path ? ((Path)queries[i]).getBackupDepth() : 0;
      if (backupDepth > 1)
      {
        if (newQueries == null)
        {
          // Initialize the array list with at least as much room for all of the original queries
          // plus the additional ones need for the current query's backups.  Other queries may also
          // require additional queries for backups but this is at least a good start.
          newQueries = new ArrayList<>(queries.length + backupDepth - 1);
          // Catch the newQueries list up to the current OrdQuery; previous ones did not have a
          // backup depth > 1
          newQueries.addAll(Arrays.asList(queries).subList(0, i));
        }

        Path path = (Path)queries[i];

        // Replace all but one backup with a new path with ".." as the body
        for (int j = 0; j < backupDepth - 1; ++j)
          newQueries.add(path.makePath(".."));

        // Remove all the "/.." from the body of the original OrdQuery.  For example,
        // slot:../../../abc/def becomes slot:../abc/def
        String newBody = queries[i].getBody().replace("/..", "");
        newQueries.add(path.makePath(newBody));
      }
      else if (newQueries != null)
      {
        // A previous replacement has been made so continue to append remaining ord queries.
        newQueries.add(queries[i]);
      }
    }

    // Remake the ord with the new set of ord queries that contains backup replacements.
    if (newQueries != null)
      ord = BOrd.make(newQueries.toArray(new OrdQuery[newQueries.size()]));

    return ord;
  }
}
