/*
 * Copyright 2015 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.bacnet.virtual;

import java.io.IOException;
import java.util.HashMap;
import java.util.StringTokenizer;

import javax.baja.data.BIDataValue;
import javax.baja.sys.*;

import java.util.logging.Level;
import java.util.logging.Logger;

import javax.baja.spy.SpyWriter;
import javax.baja.virtual.BVirtualComponent;

import javax.baja.bacnet.BBacnetDevice;
import javax.baja.bacnet.BBacnetNetwork;
import javax.baja.bacnet.BacnetConst;
import javax.baja.bacnet.datatypes.BBacnetObjectIdentifier;
import javax.baja.bacnet.enums.BBacnetObjectType;
import javax.baja.bacnet.point.BBacnetTuningPolicy;
import javax.baja.bacnet.point.BBacnetTuningPolicyMap;

import com.tridium.bacnet.job.BacnetDiscoveryUtil;

/**
 * BBacnetVirtualObject is the virtual representation of a BACnet
 * object.  It contains the objectId, tuning policy name, priority for
 * writing, and any point facets that configure the display of property
 * data.
 *
 * @author Craig Gemmill
 * @version $Revision$ $Date$
 * @creation 05 Dec 2007
 * @since NiagaraAX 3.3
 */
public class BBacnetVirtualObject
  extends BVirtualComponent
  implements BacnetConst
{
  private static final BFacets UNINITIALIZED_FACETS = BFacets.make("initialized", BBoolean.FALSE);

  /*-
  class BBacnetVirtualObject
  {
    properties
    {
      objectId:BBacnetObjectIdentifier
        flags { hidden, readonly }
        default {[ BBacnetObjectIdentifier.DEFAULT ]}
      
      facets: BFacets
        default {[ UNINITIALIZED_FACETS ]}
        
      tuningPolicyName: String
        -- References the TuningPolicy component by name.
        default {[ "defaultPolicy" ]}
        slotfacets {[ BFacets.make(BFacets.FIELD_EDITOR, "bacnet:VirtualTuningPolicyNameFE") ]}
      
      writePriority:int
        default {[ -1 ]}
      
      prioritizedPoint:boolean
        default {[ false ]}
    }
  }
  -*/
/*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
/*@ $javax.baja.bacnet.virtual.BBacnetVirtualObject(232828283)1.0$ @*/
/* Generated Thu Feb 14 12:54:50 EST 2008 by Slot-o-Matic 2000 (c) Tridium, Inc. 2000 */

////////////////////////////////////////////////////////////////
// Property "objectId"
////////////////////////////////////////////////////////////////

  /**
   * Slot for the <code>objectId</code> property.
   *
   * @see javax.baja.bacnet.virtual.BBacnetVirtualObject#getObjectId
   * @see javax.baja.bacnet.virtual.BBacnetVirtualObject#setObjectId
   */
  public static final Property objectId = newProperty(Flags.HIDDEN | Flags.READONLY, BBacnetObjectIdentifier.DEFAULT, null);

  /**
   * Get the <code>objectId</code> property.
   *
   * @see javax.baja.bacnet.virtual.BBacnetVirtualObject#objectId
   */
  public BBacnetObjectIdentifier getObjectId()
  {
    return (BBacnetObjectIdentifier)get(objectId);
  }

  /**
   * Set the <code>objectId</code> property.
   *
   * @see javax.baja.bacnet.virtual.BBacnetVirtualObject#objectId
   */
  public void setObjectId(BBacnetObjectIdentifier v)
  {
    set(objectId, v, null);
  }

////////////////////////////////////////////////////////////////
// Property "facets"
////////////////////////////////////////////////////////////////

  /**
   * Slot for the <code>facets</code> property.
   *
   * @see javax.baja.bacnet.virtual.BBacnetVirtualObject#getFacets
   * @see javax.baja.bacnet.virtual.BBacnetVirtualObject#setFacets
   */
  public static final Property facets = newProperty(0, UNINITIALIZED_FACETS, null);

  /**
   * Get the <code>facets</code> property.
   *
   * @see javax.baja.bacnet.virtual.BBacnetVirtualObject#facets
   */
  public BFacets getFacets()
  {
    return (BFacets)get(facets);
  }

  /**
   * Set the <code>facets</code> property.
   *
   * @see javax.baja.bacnet.virtual.BBacnetVirtualObject#facets
   */
  public void setFacets(BFacets v)
  {
    set(facets, v, null);
  }

////////////////////////////////////////////////////////////////
// Property "tuningPolicyName"
////////////////////////////////////////////////////////////////

  /**
   * Slot for the <code>tuningPolicyName</code> property.
   * References the TuningPolicy component by name.
   *
   * @see javax.baja.bacnet.virtual.BBacnetVirtualObject#getTuningPolicyName
   * @see javax.baja.bacnet.virtual.BBacnetVirtualObject#setTuningPolicyName
   */
  public static final Property tuningPolicyName = newProperty(0, "defaultPolicy", BFacets.make(BFacets.FIELD_EDITOR, "bacnet:VirtualTuningPolicyNameFE"));

  /**
   * Get the <code>tuningPolicyName</code> property.
   *
   * @see javax.baja.bacnet.virtual.BBacnetVirtualObject#tuningPolicyName
   */
  public String getTuningPolicyName()
  {
    return getString(tuningPolicyName);
  }

  /**
   * Set the <code>tuningPolicyName</code> property.
   *
   * @see javax.baja.bacnet.virtual.BBacnetVirtualObject#tuningPolicyName
   */
  public void setTuningPolicyName(String v)
  {
    setString(tuningPolicyName, v, null);
  }

////////////////////////////////////////////////////////////////
// Property "writePriority"
////////////////////////////////////////////////////////////////

  /**
   * Slot for the <code>writePriority</code> property.
   *
   * @see javax.baja.bacnet.virtual.BBacnetVirtualObject#getWritePriority
   * @see javax.baja.bacnet.virtual.BBacnetVirtualObject#setWritePriority
   */
  public static final Property writePriority = newProperty(0, -1, null);

  /**
   * Get the <code>writePriority</code> property.
   *
   * @see javax.baja.bacnet.virtual.BBacnetVirtualObject#writePriority
   */
  public int getWritePriority()
  {
    return getInt(writePriority);
  }

  /**
   * Set the <code>writePriority</code> property.
   *
   * @see javax.baja.bacnet.virtual.BBacnetVirtualObject#writePriority
   */
  public void setWritePriority(int v)
  {
    setInt(writePriority, v, null);
  }

////////////////////////////////////////////////////////////////
// Property "prioritizedPoint"
////////////////////////////////////////////////////////////////

  /**
   * Slot for the <code>prioritizedPoint</code> property.
   *
   * @see javax.baja.bacnet.virtual.BBacnetVirtualObject#getPrioritizedPoint
   * @see javax.baja.bacnet.virtual.BBacnetVirtualObject#setPrioritizedPoint
   */
  public static final Property prioritizedPoint = newProperty(0, false, null);

  /**
   * Get the <code>prioritizedPoint</code> property.
   *
   * @see javax.baja.bacnet.virtual.BBacnetVirtualObject#prioritizedPoint
   */
  public boolean getPrioritizedPoint()
  {
    return getBoolean(prioritizedPoint);
  }

  /**
   * Set the <code>prioritizedPoint</code> property.
   *
   * @see javax.baja.bacnet.virtual.BBacnetVirtualObject#prioritizedPoint
   */
  public void setPrioritizedPoint(boolean v)
  {
    setBoolean(prioritizedPoint, v, null);
  }

////////////////////////////////////////////////////////////////
// Type
////////////////////////////////////////////////////////////////

  public Type getType()
  {
    return TYPE;
  }

  public static final Type TYPE = Sys.loadType(BBacnetVirtualObject.class);

/*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/


  public BBacnetVirtualObject()
  {
  }

  public BBacnetVirtualObject(String virtualPathName)
  {
    try
    {
      // virtualPathName is a series of one or more tokens separated by semicolons
      // must have at least one token with objectId
      StringTokenizer st = new StringTokenizer(virtualPathName, ";");
      BBacnetObjectIdentifier id = (BBacnetObjectIdentifier)BBacnetObjectIdentifier.DEFAULT
        .decodeFromString(st.nextToken());
      setObjectId(id);

      // additionally, may have up to 1 token each for policy= and priority=
      while (st.hasMoreTokens())
      {
        String s = st.nextToken();
        if (s.startsWith(POLICY_DEF))
        {
          setTuningPolicyName(s.substring(POLICY_DEF_LEN));
        }
        else if (s.startsWith(PRIORITY_DEF))
        {
          try
          {
            int pri = Integer.parseInt(s.substring(PRIORITY_DEF_LEN));
            setWritePriority(pri);
          }
          catch (Exception e)
          {
            log.log(Level.SEVERE, "Invalid priority: " + s + " in virtualPathName for " + virtualPathName, e);
          }
        }
      }
    }
    catch (IOException e)
    {
      log.log(Level.SEVERE, "IOException occurred in BBacnetVirtualObject", e);
    }
  }


////////////////////////////////////////////////////////////////
// Overrides
////////////////////////////////////////////////////////////////

  public String toString(Context cx)
  {
    return getObjectId().toString(cx);
  }

  public void started()
    throws Exception
  {
    discoverFacets();
  }

  public boolean isChildLegal(BComponent child)
  {
    return child instanceof BBacnetVirtualProperty;
  }


////////////////////////////////////////////////////////////////
// Access
////////////////////////////////////////////////////////////////

  /**
   * Get the containing BBacnetDevice.
   * This method will not work for the "local" version.
   *
   * @return the BBacnetDevice in which this object resides.
   */
  public BBacnetDevice device()
  {
    if (getVirtualGateway() == null) return null;
    return ((BBacnetVirtualGateway)getVirtualGateway()).device();
  }

  public void updateStatus()
  {
    SlotCursor<Property> sc = getProperties();
    while (sc.next(BBacnetVirtualProperty.class))
      ((BBacnetVirtualProperty)sc.get()).updateStatus();
  }

  /**
   * Get the BTuningPolicy configured by policyName. If the policyName doesn't
   * map to a valid policy then log a warning and use the defaultPolicy.
   * <p>
   * Note that all virtual components are polled at this time.
   */
  public BBacnetTuningPolicy getPolicy()
  {
    if (cachedPolicy == null)
    {
      String tpName = getTuningPolicyName();
      BBacnetTuningPolicyMap map = getPolicyMap();
      BValue x = map.get(tpName);
      if (x instanceof BBacnetTuningPolicy)
      {
        cachedPolicy = (BBacnetTuningPolicy)x;
      }
      else
      {
        log.warning("TuningPolicy not found: " + tpName);
        cachedPolicy = (BBacnetTuningPolicy)map.getDefaultPolicy();
      }
    }
    return cachedPolicy;
  }


////////////////////////////////////////////////////////////////
// Support
////////////////////////////////////////////////////////////////

  /**
   * Discover the facets to be used for the properties that use them.
   */
  protected void discoverFacets()
  {
    network().postAsync(new Runnable()
    {
      public void run()
      {
        // Discover the facets to be used for this point.
        HashMap<String, BIDataValue> m = BacnetDiscoveryUtil.discoverFacets(getObjectId(), device());
        setFacets(BFacets.make(m));

        // Check if this is a prioritized point.
        int objectType = getObjectId().getObjectType();
        if ((objectType == BBacnetObjectType.ANALOG_OUTPUT)
          || (objectType == BBacnetObjectType.BINARY_OUTPUT)
          || (objectType == BBacnetObjectType.MULTI_STATE_OUTPUT))
          setPrioritizedPoint(true);
        else if ((objectType == BBacnetObjectType.ANALOG_VALUE)
          || (objectType == BBacnetObjectType.BINARY_VALUE)
          || (objectType == BBacnetObjectType.MULTI_STATE_VALUE))
          setPrioritizedPoint(BacnetDiscoveryUtil.checkForPriorityArray(getObjectId(), device()).getBoolean());
      }
    });
  }

  /**
   * Get the containing BBacnetNetwork.
   * This method will not work for the "local" version.
   */
  private BBacnetNetwork network()
  {
    return ((BBacnetVirtualGateway)getVirtualGateway()).network();
  }

  /**
   * Get the BTuningPolicyMap on the parent network.
   */
  private BBacnetTuningPolicyMap getPolicyMap()
  {
    BBacnetTuningPolicyMap map = (BBacnetTuningPolicyMap)
      network().get("tuningPolicies");
    if (map != null)
      return map;
    throw new IllegalStateException("Network missing tuningPolicies property");
  }

////////////////////////////////////////////////////////////////
// Spy
////////////////////////////////////////////////////////////////

  public void spy(SpyWriter out)
    throws Exception
  {
    super.spy(out);
    out.startProps();
    out.trTitle("BacnetVirtualObject", 2);
    out.prop("cachedPolicy", cachedPolicy);
    out.endProps();
  }


////////////////////////////////////////////////////////////////
// Constants
////////////////////////////////////////////////////////////////

  private static Logger log = Logger.getLogger("bacnet.virtual");

  static final String POLICY_DEF = "policy=";
  static final int POLICY_DEF_LEN = 7;
  static final String PRIORITY_DEF = "priority=";
  static final int PRIORITY_DEF_LEN = 9;


////////////////////////////////////////////////////////////////
// Attributes
////////////////////////////////////////////////////////////////

  private BBacnetTuningPolicy cachedPolicy = null;
}
