/*
 * Copyright 2013 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.bacnet.util.worker;

import java.util.ArrayList;
import java.util.List;

import javax.baja.sys.BComponent;
import javax.baja.sys.BIcon;
import javax.baja.sys.Context;
import javax.baja.sys.NotRunningException;
import javax.baja.sys.Property;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import javax.baja.util.CoalesceQueue;
import javax.baja.util.IFuture;
import javax.baja.util.Queue;
import javax.baja.util.ThreadPoolWorker;

/**
 * BBacnetAddressWorkerPool balances incoming work across a
 * pool of thread pools using the BBacnetAddress hashcode() method.
 * <p>
 * Spreading the load by address will decrease the impact
 * to the system as a whole from one device.
 * <p>
 * This will not be an even split, and can be less than optimal depending
 * on the device addresses in a particular installation.
 * It is possible that all messages could end up getting routed
 * to one worker pool.
 *
 * @author Joseph Chandler
 * @version $Revision$ $Date$
 * @creation 26 Aug 2013
 * @since Niagara 3.8 Bacnet 1.0
 */
public class BBacnetAddressWorkerPool
  extends BComponent
  implements IWorkerPool
{
  private static final int DEFAULT_POOLS = 4;
  private static final int DEFAULT_WORKERS_PER_POOL = 2;
  
  /*-
  class BBacnetAddressWorkerPool
  {
    properties
    {
      addressPools: int
        -- The number of address pools should be based on the 
        -- number of bacnet devices and the desired loading factor.
        default {[ DEFAULT_POOLS ]} 

      workersPerAddressPool: int
        -- The number of workers per address pool
        default {[ DEFAULT_WORKERS_PER_POOL ]} 
    }
  }
  -*/
/*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
/*@ $javax.baja.bacnet.util.worker.BBacnetAddressWorkerPool(3378217669)1.0$ @*/
/* Generated Tue Sep 03 16:19:21 EDT 2013 by Slot-o-Matic 2000 (c) Tridium, Inc. 2000 */

////////////////////////////////////////////////////////////////
// Property "addressPools"
////////////////////////////////////////////////////////////////

  /**
   * Slot for the <code>addressPools</code> property.
   * The number of address pools should be based on the
   * number of bacnet devices and the desired loading factor.
   *
   * @see javax.baja.bacnet.util.worker.BBacnetAddressWorkerPool#getAddressPools
   * @see javax.baja.bacnet.util.worker.BBacnetAddressWorkerPool#setAddressPools
   */
  public static final Property addressPools = newProperty(0, DEFAULT_POOLS, null);

  /**
   * Get the <code>addressPools</code> property.
   * The number of address pools should be based on the
   * number of bacnet devices and the desired loading factor.
   *
   * @see javax.baja.bacnet.util.worker.BBacnetAddressWorkerPool#addressPools
   */
  public int getAddressPools()
  {
    return getInt(addressPools);
  }

  /**
   * Set the <code>addressPools</code> property.
   * The number of address pools should be based on the
   * number of bacnet devices and the desired loading factor.
   *
   * @see javax.baja.bacnet.util.worker.BBacnetAddressWorkerPool#addressPools
   */
  public void setAddressPools(int v)
  {
    setInt(addressPools, v, null);
  }

////////////////////////////////////////////////////////////////
// Property "workersPerAddressPool"
////////////////////////////////////////////////////////////////

  /**
   * Slot for the <code>workersPerAddressPool</code> property.
   * The number of workers per address pool
   *
   * @see javax.baja.bacnet.util.worker.BBacnetAddressWorkerPool#getWorkersPerAddressPool
   * @see javax.baja.bacnet.util.worker.BBacnetAddressWorkerPool#setWorkersPerAddressPool
   */
  public static final Property workersPerAddressPool = newProperty(0, DEFAULT_WORKERS_PER_POOL, null);

  /**
   * Get the <code>workersPerAddressPool</code> property.
   * The number of workers per address pool
   *
   * @see javax.baja.bacnet.util.worker.BBacnetAddressWorkerPool#workersPerAddressPool
   */
  public int getWorkersPerAddressPool()
  {
    return getInt(workersPerAddressPool);
  }

  /**
   * Set the <code>workersPerAddressPool</code> property.
   * The number of workers per address pool
   *
   * @see javax.baja.bacnet.util.worker.BBacnetAddressWorkerPool#workersPerAddressPool
   */
  public void setWorkersPerAddressPool(int v)
  {
    setInt(workersPerAddressPool, v, null);
  }

////////////////////////////////////////////////////////////////
// Type
////////////////////////////////////////////////////////////////

  public Type getType()
  {
    return TYPE;
  }

  public static final Type TYPE = Sys.loadType(BBacnetAddressWorkerPool.class);

/*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/


////////////////////////////////////////////////////////////////
// Constructors
////////////////////////////////////////////////////////////////

  /**
   * Default constructor.
   */
  public BBacnetAddressWorkerPool()
  {

  }

////////////////////////////////////////////////////////////////
// Overrides
////////////////////////////////////////////////////////////////

  public boolean isParentLegal(BComponent parent)
  {
    return BBacnetWorkerPool.isLegalParent(parent);
  }

  public void started()
  {
    startPools();
  }

  public void stopped()
  {
    stopPools();
  }

  public void changed(Property p, Context cx)
  {
    if (!isRunning())
      return;

    if (p.equals(addressPools) || p.equals(workersPerAddressPool))
    {
      restartPools();
    }
  }

  private void restartPools()
  {
    synchronized (lock)
    {
      stopPools();
      startPools();
    }
  }

  private void startPools()
  {
    synchronized (lock)
    {
      int numberOfPools = getAddressPools();
      addressWorkers = new ArrayList<>(numberOfPools);
      IWorkerPoolAware parent = (IWorkerPoolAware)getParent();
      int maxQueueSize = parent.getQueue().maxSize();
      String threadName = parent.getWorkerThreadName();

      for (int i = 0; i < numberOfPools; i++)
      {
        Queue queue = new CoalesceQueue(maxQueueSize);
        ThreadPoolWorker worker = new ThreadPoolWorker(queue);
        worker.setMaxThreads(getWorkersPerAddressPool());
        worker.start(threadName + i);
        addressWorkers.add(new AddressWorker(worker, queue));
      }
      parent.stopWorker();
    }
  }

  private void stopPools()
  {
    synchronized (lock)
    {
      if (addressWorkers != null)
      {
        int numberOfPools = addressWorkers.size();
        for (int i = 0; i < numberOfPools; i++)
        {
          AddressWorker aw = addressWorkers.get(i);
          aw.stop();
        }
        addressWorkers.clear();
      }
    }
  }

  public IFuture post(Runnable r)
  {
    if (!isRunning())
      throw new NotRunningException();

    synchronized (lock)
    {
      if (r instanceof IBacnetAddress)
      {
        //Try to keep work from the same device on the same workers.
        IBacnetAddress request = (IBacnetAddress)r;
        int workerIdx = request.getAddress().hash() % addressWorkers.size();
        AddressWorker aw = addressWorkers.get(workerIdx);
        aw.enqueue(r);

      }
      else
      {
        if (addressWorkers.size() > 0)
        {
          AddressWorker defaultWorker = addressWorkers.get(0);
          defaultWorker.enqueue(r);
        }
      }
    }
    return null;
  }

  /**
   * Get the icon.
   */
  public BIcon getIcon()
  {
    return icon;
  }

  private static final BIcon icon = BIcon.std("gears.png");

////////////////////////////////////////////////////////////////
//Attributes
////////////////////////////////////////////////////////////////

  private List<AddressWorker> addressWorkers;
  private Object lock = new Object();

  private static class AddressWorker
  {
    public AddressWorker(ThreadPoolWorker worker, Queue queue)
    {
      this.worker = worker;
      this.queue = queue;
    }

    public void stop()
    {
      if (worker != null)
      {
        worker.stop();
      }
    }

    public void enqueue(Runnable r)
    {
      queue.enqueue(r);
    }

    private ThreadPoolWorker worker;
    private Queue queue;
  }

}