/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/* jshint browser: true, camelcase: false */
/* global $interop_BroadcastChannel_postMessage */

/**
 * API Status: **Private**
 * @module nmodule/workbench/com/tridium/workbench/web/browser/interop/BroadcastChannel
 */
(function (global) {

  'use strict';

  var tabId = randomString(),
      reg = {},
      idCounter = 0;

////////////////////////////////////////////////////////////////
// BroadcastChannel Workbench polyfill
////////////////////////////////////////////////////////////////

  /**
   * Workbench polyfill for the BroadcastChannel API. Uses JS-Java interop to
   * function and will not work outside of Workbench. This will overwrite
   * BroadcastChannel if present, even if one of the Workbench browsers adds
   * support for it in the future.
   *
   * Note that this is *not* a RequireJS module and can be injected
   * synchronously as a script.
   *
   * @class
   * @alias module:nmodule/workbench/com/tridium/workbench/web/browser/interop/BroadcastChannel
   * @param {String} channelName
   */
  var BroadcastChannel = function BroadcastChannel(channelName) {
    this.$channelName = channelName;
    this.$id = tabId + ':' + new Date().getTime().toString(16) + ':' + idCounter++;
    (reg[channelName] || (reg[channelName] = [])).push(this);
  };

  BroadcastChannel.prototype.postMessage = function (data) {
    if (this.$closed) { throw alreadyClosedError(); }

    BroadcastChannel.$interop_postMessage({
      channelName: this.$channelName,
      channelId: this.$id,
      data: JSON.stringify(data)
    });
  };

  BroadcastChannel.prototype.onmessage = function (event) {};

  BroadcastChannel.prototype.close = function () {
    var that = this, channelName = that.$channelName;

    that.$closed = true;

    reg[channelName] = reg[channelName].filter(function (ch) {
      return ch !== that;
    });
  };


////////////////////////////////////////////////////////////////
// Workbench/WebBrowser interop
////////////////////////////////////////////////////////////////

  /**
   * Workbench-to-browser interop.
   * @param {{ channelName: string, channelId: string, event: object }} obj
   */
  BroadcastChannel.$interop_onmessage = function (obj) {
    var channelName = obj.channelName,
        channelId = obj.channelId,
        event = obj.event,
        arr = reg[channelName];

    if (arr) {
      arr.forEach(function (ch) {
        if (ch.$id !== channelId) {
          ch.onmessage(event);
        }
      });
    }
  };

  /**
   * Browser-to-Workbench interop.
   *
   * @param {{ channelName: string, channelId: string, data: string }} data
   */
  BroadcastChannel.$interop_postMessage = function (data) {
    //This function is injected by Workbench as its own data consumer.
    if (typeof $interop_BroadcastChannel_postMessage === 'function') {
      return $interop_BroadcastChannel_postMessage(data);
    }
  };


////////////////////////////////////////////////////////////////
// Support methods
////////////////////////////////////////////////////////////////

  function randomString(length) {
    var s = '';
    for (var i = 0, len = length || 10; i < len; i++) {
      s += Math.floor(Math.random() * 16).toString(16);
    }
    return s;
  }

  function alreadyClosedError() {
    var e = new Error('Already closed');
    e.name = 'InvalidStateError';
    return e;
  }

  global.BroadcastChannel = BroadcastChannel;
}(this));
