/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/*global niagara*/
/*jshint browser: true, node: true */

/**
 * Define baja plug-in for RequireJS. The baja plug-in will be used whenever a 
 * requirement for <code>baja!moduleName:typeName</code> is used in RequireJS. This plug-in
 * is used for importing Type information.
 *
 * For extra network efficiency, it's recommended to provide a comma separated list of
 * of Types to import. For example, <code>baja!foo:Boo,goo:Doo</code>.
 *
 * This plug-in can also be used to acquire the BajaScript namespace (with no TypeSpec specified).
 * For example, <code>baja!</code> is used to access the 'baja' namespace.
 *
 * Unless specified (via the config.autoStart option), this plug-in will auto-start 
 * BajaScript if it's not already started.
 *
 * Unless specified (via the config.autoStop option), this plug-in will set up BajaScript
 * to automatically stop on page shutdown.
 *
 * In RequireJS, {@link http://requirejs.org/docs/api.html#config-moduleconfig} config options
 * can be passed through into a plug-in. This plug-in supports the following options...
 *     var require = {
 *       config: {
 *         baja: {
 *           start: { // Options passed into baja.start...
 *           },
 *           stop: { // Options passed into baja.stop...
 *           },
 *           autoStart: false, // disable the plug-in from auto-starting BajaScript (true by default)
 *           autoStop: false   // disable the plug-in from automatically stopping BajaScript (true by default)
 *         }
 *       }
 *     };
 *
 * @module  plugin/baja
 * @private
 */
define(function () {
 
  'use strict';
  
  var baja,   // Cache BajaScript namespace for quick access
      bajaPromises, //ditto
      startCalled,
      stopSetupCalled,
      emptyObj = {},
      
      isBrowser = require.isBrowser,
      modulePrefix = getEnvModulePrefix() || "/module/";
        
  /**
   * If BajaScript is not already started (and autoStart for this plug-in is false)
   * then start BajaScript. The callback from this will then try to import any necessary
   * TypeSpecs.
   *
   * @ignore
   *
   * @param {String} name the name string passed into the plug-in.
   * @param {Function} onLoad the function callback for the plug-in. This function must be called once
   *                          the plug-in has finished loading.
   * @param {Object} options the config options for the plug-in.
   */   
  function start(name, onLoad, options) {
    // Set up defaults
    var autoStart = baja.def(options.autoStart, true),
        autoStop  = baja.def(options.autoStop,  true),
        shutdown,
        typeSpecs;
    
    function doImportTypes() {
      baja.importTypes(typeSpecs)
        .then(function (types) {
          onLoad(types);
        }, function (err) {
          baja.error(err);
        });
    }

    if (!stopSetupCalled && autoStop) {    
      // If the BajaScript start method hasn't been called then invoke it
      stopSetupCalled = true;

      // Register Stop callbacks for BajaScript
      shutdown = function () {
        baja.stop(options.stop || emptyObj);
      };
      
      if (isBrowser) {
        //if not node, assume browser
        if (window.addEventListener) {
          //NCCB-19190: comment back in when interested in registry inconsistency
          //window.addEventListener('beforeunload', function (event) {
          //  if (baja.registry.isInconsistent()) {
          //    baja.outln('inconsistent registry information ' +
          //      'detected. cancel the confirmation message and call Logan. see NCCB-19190.');
          //    event.returnValue = 'https://bugs.chromium.org/p/chromium/issues/detail?id=587940';
          //  }
          //});
          window.addEventListener("unload", shutdown, false);
          window.addEventListener("pagehide", shutdown, false);
          window.addEventListener("pageshow", function () {
            if (baja.isStopped()) {
              window.location.reload(false);
            }
          });
        }
        else {
          window.onunload = shutdown;
        }
      } else {
        process.on('exit', function () {
          baja.outln('process exiting, shutting down BajaScript');
          baja.stop(options.stop || emptyObj);
        });
      }
    }
    
    // Import the Types if available
    if (name) {
      typeSpecs = name.split(',');
      
      if (autoStart && !baja.isStarted()) {
        baja.started(doImportTypes);
      } else {
        doImportTypes();
      }
    }
    else {
      if (autoStart) {
        if (!startCalled) {
          // Start BajaScript
          startCalled = true;
          baja.start(options.start || emptyObj);
        }
        
        baja.started(function () {
          onLoad(baja);
        });
      } else {
        onLoad(baja);
      }
    }
  }
    
 /**
  * Called when the plug-in is invoked.
  *
  * @ignore
  *
  * @param {String} name the name string passed into the plug-in.
  * @param {Function} req the parent require function.
  * @param {Function} onLoad the function callback for the plug-in. This function must be called once
  *                          the plug-in has finished loading.
  * @param {Object} config the config options for the plug-in.
  */
  function load(name, req, onLoad, config) {
    var options = config &&
                  config.config &&
                  config.config.baja ? config.config.baja : emptyObj,
        disableConnectionReuse = !!options.disableConnectionReuse,          
        entryPoint = options.env || (isBrowser ? "bajaScript/env/browser" : "bajaScript/env/define-node"),
        parentBaja;
        
    function bajaParentRequire() {
      var wnd,
          parentRequire;

      try {
        if (!disableConnectionReuse && isBrowser && !parentBaja) {
          // Try to find RequireJS in a parent window.
          wnd = window;
          while (wnd !== wnd.parent && (wnd = wnd.parent)) {
            // Detect RequireJS.
            if (typeof wnd.require === "function" &&
              typeof wnd.define === "function" &&
              wnd.define.amd) {
              parentRequire = wnd.require;
              break;
            }
          }
        }
      } catch (ignore) {
        //don't reuse connection when there are cross-origin frame problems
      }

      if (parentRequire) {
        // Try to find BajaScript in a parent window and use it if it's available.
        parentRequire([entryPoint], function (_parentBaja) {
          parentBaja = _parentBaja;
          bajaRequire();
        }, bajaRequire);
      }
      else {
        bajaRequire();
      }
    }

    function bajaRequire() {
      // For now, just require these other libs
      req([entryPoint, 'bajaPromises'], function (b, p) {
        baja = b;
        bajaPromises = p;
        baja.$webdev = options.webdev;
        baja.$offline = options.offline;

        var parentComm = parentBaja && parentBaja.comm,
            parentConnection,
            readyToStart = bajaPromises.resolve();

        // See if a connection is available. If so then attempt to reuse it.
        if (parentComm && 
          typeof parentComm.getConnection === "function" && 
          typeof parentComm.reuseConnection === "function") {

          parentConnection = parentComm.getConnection();

          if (parentConnection) {
            // Use the same message numbering for sending out messages.
            baja.comm.incrementRequestId = parentComm.incrementRequestId;

            // Use the same frame numbering for sending out frames.
            baja.comm.incrementFrameId = parentComm.incrementFrameId;

            // Use the same envelope numbering for sending out envelopes.
            baja.$mux.nextEnvelopeId = function () {
              return parentBaja.$mux.nextEnvelopeId();
            };

            //baja.$bsRegStorage = parentBaja.$bsRegStorage;

            // Set the connection object to use.
            readyToStart = baja.comm.reuseConnection(parentConnection);
          }
        }

        return readyToStart
          .then(function () {
            start(name, onLoad, options);
          });
      });
    }
  
    // Require BajaScript if we haven't got a cached copy of it
    if (!baja) {        
      if (!require.specified("bajaPromises")) {
        require.config({
          paths: {
            bajaPromises: options.promises || (modulePrefix + "bajaScript/rc/env/Promises"),
          }
        });
      }

      if (!require.specified("bajaBrowserEnvUtil")) {
        require.config({
          paths: {
            bajaBrowserEnvUtil: options.browserEnvUtil || (modulePrefix + "bajaScript/rc/env/jQueryBrowserEnvUtil")
          }
        });
      }

      if (!options.webdev) {
        // If not in web development mode, require the minified file first
        req(["bajaScript/bs.built.min"], function () {
          bajaParentRequire();
        });
      }
      else {
        bajaParentRequire();
      }
    }
    else {
      // If we already have a cached copy of the BajaScript namespace then 
      // start straight away
      start(name, onLoad, options);
    }
  }

  /**
   * Return the modulePrefix from niagara.env.modulePrefix if it exists
   * @returns {String}
   */
  function getEnvModulePrefix() {
    return typeof niagara !== "undefined" &&
      niagara.env &&
      niagara.env.modulePrefix;
  }
  
  // Export functions for plug-in
  return {
    load: load,
    pluginBuilder: "buildBaja"
  };
});
  
