/**
 * @copyright 2019 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/*jshint browser: true, devel: true */
/*global MutationObserver, WebKitMutationObserver, niagaraAsync */

/**
 * Hooks for running BajaScript inside of a browser.
 *
 * @module baja/env/browser
 * @private
 */
define(["stompbox",
        "bajaBrowserEnvUtil",
        "bajaScript/baja/comm/Callback",
        "bajaScript/alarm",
        "bajaScript/boxcs",
        "bajaScript/bson",
        "bajaScript/coll",
        "bajaScript/comm",
        "bajaScript/comp",
        "bajaScript/ctypes",
        "bajaScript/file",
        "bajaScript/hist",
        "bajaScript/nav",
        "bajaScript/obj",
        "bajaScript/ord",
        "bajaScript/sys",
        "bajaScript/tag",
        "bajaScript/transfer",
        "bajaScript/virt"], function defineBrowser(
          stompbox,
          envUtil,
          Callback,
          baja) {
  "use strict";

  window.stompbox = stompbox.default;

  if (!baja.comm) {
    throw new Error("comm.js required");
  }
  
  ////////////////////////////////////////////////////////////////
  // Callback Promise Implementation
  //////////////////////////////////////////////////////////////// 
  
  envUtil.mixInPromises(Callback);
      
  /**
   * This namespace is for documentation purposes only and will not actually
   * be available to Bajascript apps. It details enhancements/decorations
   * applied to functions in <code>baja.comm</code> when Bajascript is deployed
   * to a web browser environment.
   * 
   * @namespace
   * @name baja.browser.comm
   */
  (function comm() {
    var doStart = baja.comm.start;

    baja.comm.start = function start(obj) {
      var cb = new baja.comm.Callback();
      cb.addOk(function () {
        return doStart({
          started: function () {
            envUtil.domReady(obj.started || baja.ok);
          }
        });
      });
      // manually create the server session to work with
      return stompbox.default.getServer()
        .then(function (server) {
          baja.stompbox = server;
          baja.comm.makeServerSession(cb);
        });
    };
    
    /**
     * In a browser, <code>baja.comm.BoxFrame#send</code> will delegate this method.
     *
     * @private
     * 
     * @param callback The asynchronous callback.
     */   
    baja.comm.BoxFrame.prototype.send = function (callback) {
      return baja.stompbox.handleFrame(this.$body)
        .then(function (result) {
          return callback.ok(result);
        }, function (err) {
          return callback.fail(err);
        });
    };

    ////////////////////////////////////////////////////////////////
    // HTTP Comms
    //////////////////////////////////////////////////////////////// 
        
    /**
     * An Http Error.
     *
     * A HTTP Error happens as a result of problem with HTTP communication.
     * 
     * @class
     * @private
     * @param {XmlHttpRequest} x the object used for comms.
     */
    baja.comm.HttpError = function (x) {
      var t = x.responseText || "Session disconnected",
          status = x.status;
      
      this.name = "HttpError";
      this.message = t + " err: " + x.status;
      this.status = status;
      
      // Indicate to delay any reconnection if a 404 or no status is returned.
      this.delayReconnect = (!status || status === 404);
    };
    
    baja.subclass(baja.comm.HttpError, baja.comm.ServerError);
                    
  }());
    
  /**
   * This namespace is for documentation purposes only and will not actually
   * be available to Bajascript apps. It details enhancements/decorations
   * applied to functions in <code>baja</code> when Bajascript is deployed
   * to a web browser environment.
   * 
   * @namespace
   * @name baja.browser
   */
  (function sysUtil() {
    baja.outln = function (msg) {
      // If BajaScript has stopped then don't output anything else...
      if (baja.isStopping()) {
        return this;
      }
      console.log(msg);
    };
    
    baja.clearOut = function () {
      if (console.clear) { console.clear(); }
    };
    
    baja.error = function (msg) {
      if (baja.isStopping()) {
        return this;
      }
      console.error(msg);
    };

    /**
     * In a browser environment that supports it, swap out the default 
     * setTimeout(0) implementation of baja.runAsync with a 
     * much faster implementation.
     */
    (function setupAsync() {
      var Obs;
      if (typeof niagaraAsync === 'function') {
        baja.runAsync = niagaraAsync;
      } else {
        Obs = (typeof MutationObserver !== 'undefined' && MutationObserver) || 
              (typeof WebKitMutationObserver !== 'undefined' && WebKitMutationObserver);
        
        if (Obs) {
          baja.runAsync = function (fn) {
            var node = document.createTextNode('');
            new Obs(fn).observe(node, { characterData: true });
            node.data = '1';
          };
        }
      }
    }());
  }());
    
  /**
   * This namespace is for documentation purposes only and will not actually
   * be available to Bajascript Apps. It details enhancements/decorations
   * applied to functions in <code>baja.storage</code> when Bajascript is 
   * deployed to a web browser environment.
   * 
   * @namespace
   * @name baja.browser.storage
   */
  (function sysRegistry() { 
    var storage = baja.storage,
        store;

    try {
      store = window.localStorage;
    } catch (ignore) {}
  
    /**
     * In a browser, this overrides <code>baja.storage.removeItem</code>
     * and will clear the web storage content (if the browser supports web 
     * storage).
     * 
     * @name baja.browser.storage.removeItem
     * @function
     * @private
     * @throws {Error} if the item could not be deleted
     */
    storage.removeItem = function (key) { 
      if (store) {
        return store.removeItem(key);
      }
    };
    
    /**
     * In a browser, this overrides <code>baja.storage.setItem</code>
     * and will write a key/value pair to web storage (if the browser supports 
     * web storage).
     * 
     * @name baja.browser.storage.setItem
     * @function
     * @private
     * 
     * @param {String} key
     * @param {String} data
     * @throws {Error} if the item could not be saved
     */
    storage.setItem = function (key, data) {
      // If available, save any cached registry information to web storage   
      if (store) {
        return store.setItem(key, data);
      }
    };
    
    /**
     * In a browser, this overrides <code>baja.storage.getItem</code> and will
     * retrieve a value from web storage (if the browser supports web storage).
     * 
     * @name baja.browser.storage.getItem
     * @function
     * @private
     * 
     * @param {String} key
     * @throws {Error} if the item could not be read
     */
    storage.getItem = function (key) {
      if (store) {
        return store.getItem(key);
      }
    };                     
  }());
 
  return baja; 
});
