/**
 * @copyright 2020 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/* jshint browser: true *//* eslint-env browser */
/**
 * API Status: **Private**
 * @module baja/env/mux/BoxEnvelope
 */
define([ 'bajaScript/env/mux/muxUtils' ], function (muxUtils) {
  'use strict';

  var toBytes = muxUtils.toBytes;

  /**
   * A `BoxEnvelope` contains a payload consisting of zero or more
   * string fragments. These fragments can be sent and received individually.
   * In this way, large payloads which may consist of many or large BoxFrames
   * can be sent without exceeding the maximum WebSocket message size.
   *
   * Think of this class as a transport-layer mechanism. It doesn't make any
   * decisions regarding which BoxFrames batch together into individual network
   * calls. It only handles the work of transmitting large payloads over a
   * protocol (i.e. WebSocket) that restricts the size of individual messages.
   *
   * @class
   * @alias module:baja/env/mux/BoxEnvelope
   * @param {object} params
   * @param {string} params.sessionId - server session ID
   * @param {string} params.destination - BOX destination
   * @param {number} params.envelopeId - envelope ID
   * @param {number} params.fragmentCount - number of fragments in this envelope
   * @param {number} params.birthDate - envelope birth date as a long timestamp
   * @param {number} params.maxMessageSize - the maximum allowable size of a
   * fragment
   * @param {number} params.maxEnvelopeSize - the maximum total allowable size
   * of this envelope
   * @param {boolean} params.unsolicited - true if this is an unsolicited
   * envelope sent down to us from the server
   * @param {Array.<Uint8Array>} [params.fragments] if the envelope is fully formed
   * at the time of creation (i.e. on the client side), populate it with its
   * fragment payloads
   *
   */
  var BoxEnvelope = function BoxEnvelope(params) {
    params = params || {};
    var sessionId = this.$sessionId = params.sessionId;
    this.$destination = params.destination;
    var envelopeId = this.$envelopeId = params.envelopeId;
    this.$birthDate = params.birthDate || +new Date();
    this.$maxMessageSize = params.maxMessageSize || Number.MAX_SAFE_INTEGER;
    this.$maxEnvelopeSize = params.maxEnvelopeSize || Number.MAX_SAFE_INTEGER;
    var unsolicited = this.$unsolicited = params.unsolicited;
    var fragments = this.$fragments = params.fragments || [];
    this.$fragmentCount = params.fragmentCount || fragments.length;
    this.$prefixOverhead = toBytes([ 'F', '2.2', sessionId, envelopeId, 99, 99,
      unsolicited ? 'u' : 's', '' ].join(';')).length;
  };

  /**
   * Creates a new BoxEnvelope from a given payload. The payload will be split
   * into fragments, each one guaranteed smaller than the max message size.
   *
   * @param {object} params
   * @param {number} params.envelopeId
   * @param {string} params.sessionId server session ID
   * @param {string} params.destination BOX destination
   * @param {number} params.maxMessageSize maximum message size that can be
   * transmitted to the server. This is typically the maximum WebSocket message
   * size, determined by the `box.ws.maxTextMessageSize` system property.
   * @param {number} params.maxEnvelopeSize - the maximum total allowable size
   * of this envelope
   * @param {Uint8Array} params.payload
   * @returns {module:baja/env/mux/BoxEnvelope}
   */
  BoxEnvelope.make = function (params) {
    var maxMessageSize = params.maxMessageSize;
    var payload = params.payload;

    var envelope = new BoxEnvelope({
      envelopeId: params.envelopeId,
      sessionId: params.sessionId,
      destination: params.destination,
      maxMessageSize: maxMessageSize,
      maxEnvelopeSize: params.maxEnvelopeSize
    });

    envelope.append(payload);
    return envelope;
  };

  /**
   * @returns {number} envelope ID
   */
  BoxEnvelope.prototype.getEnvelopeId = function () {
    return this.$envelopeId;
  };

  /**
   * @returns {number} the number of fragments this envelope should contain
   */
  BoxEnvelope.prototype.getFragmentCount = function () {
    return this.$fragmentCount;
  };

  /**
   * @returns {number} envelope birth date as a long timestamp
   */
  BoxEnvelope.prototype.getBirthDate = function () {
    return this.$birthDate;
  };

  /**
   * @returns {boolean} true if this envelope was received from the server
   * unsolicited
   */
  BoxEnvelope.prototype.isUnsolicited = function () {
    return this.$unsolicited;
  };

  /**
   * @returns {Array.<Uint8Array>} the fragments
   * currently contained in this envelope (may be sparse)
   */
  BoxEnvelope.prototype.getFragments = function () {
    return this.$fragments.slice();
  };

  /**
   * Adds a newly-received fragment to an envelope in progress.
   * @param {Uint8Array} fragment
   * @param {number} index fragment index
   * @throws {Error} if the fragment has an invalid fragment index
   */
  BoxEnvelope.prototype.receiveFragment = function (fragment, index) {
    var fragments = this.$fragments;

    if (index < 0 || index >= this.$fragmentCount) {
      throw new Error('invalid fragment index');
    }

    fragments[index] = fragment;
  };

  /**
   * Appends a new fragment to a pending outgoing BoxEnvelope and updates
   * fragment count.
   *
   * @param {string|Uint8Array} fragment
   */
  BoxEnvelope.prototype.append = function (fragment) {
    if (!fragment) { return; }

    if (typeof fragment === 'string') { fragment = toBytes(fragment); }

    var fragments = this.$fragments;
    var maxFragmentLength = this.$maxMessageSize - this.$prefixOverhead;

    var lastFragment = fragments[fragments.length - 1];
    if (lastFragment) {
      var freeSpace = maxFragmentLength - lastFragment.length;
      fragments[fragments.length - 1] = mergeArrays([ lastFragment,
        fragment.slice(0, freeSpace) ]);
      fragment = fragment.slice(freeSpace);
    }

    while (fragment.length) {
      fragments.push(fragment.slice(0, maxFragmentLength));
      this.$fragmentCount++;
      fragment = fragment.slice(maxFragmentLength);
    }
  };

  /**
   * @param {number} payloadSize the size of the payload you wish to append to
   * the envelope
   * @returns {boolean} true if the payload can fit
   */
  BoxEnvelope.prototype.willFit = function (payloadSize) {
    var that = this;
    var fragments = that.$fragments;

    var newData = fragments
      .map(BoxEnvelope.byteLength)
      .reduce(sum, 0) + payloadSize;
    return newData <= this.maxPayloadSize();
  };

  /**
   * @returns {boolean} true if the number of fragments currently in this
   * envelope is equal to the expected fragment count
   */
  BoxEnvelope.prototype.isComplete = function () {
    var count = this.$fragmentCount;
    var fragments = this.$fragments;
    for (var i = 0; i < count; i++) {
      if (!fragments[i]) { return false; }
    }
    return true;
  };

  /**
   * @returns {boolean} true if there is nothing in the envelope to send
   */
  BoxEnvelope.prototype.isEmpty = function () {
    return !this.$fragments[0];
  };

  BoxEnvelope.prototype.expectedMessageSizeForPayload = function (payload) {
    return BoxEnvelope.byteLength(payload) + this.$prefixOverhead;
  };

  BoxEnvelope.prototype.maxPayloadSize = function () {
    var maxFragments = Math.ceil(this.$maxEnvelopeSize / this.$maxMessageSize);

    return this.$maxEnvelopeSize - maxFragments * this.$prefixOverhead;
  };

  /**
   * @returns {Uint8Array} the complete payload reconstituted from all fragments
   * @throws {Error} if the envelope is not yet complete
   */
  BoxEnvelope.prototype.getPayload = function () {
    if (!this.isComplete()) {
      throw new Error('envelope not complete');
    }

    return mergeArrays(this.$fragments);
  };

  /**
   * @returns {Uint8Array[]}
   */
  BoxEnvelope.prototype.toBoxFragments = function () {
    var that = this;
    return that.$fragments.map(function (fragment, i) {
      return toBoxFragment(that, i);
    });
  };

  /**
   * @param {Uint8Array} arr
   * @returns {number} size of the payload in bytes
   */
  BoxEnvelope.byteLength = function (arr) {
    return arr.length;
  };

  BoxEnvelope.toBytes = function (str) {
    return toBytes(str);
  };

  function sum(a, b) { return a + b; }

  /**
   * @param {module:baja/env/mux/BoxEnvelope} env
   * @param {number} fragmentIndex
   * @returns {Uint8Array}
   */
  function toBoxFragment(env, fragmentIndex) {
    return mergeArrays([
      toPrefix(env, fragmentIndex),
      env.$fragments[fragmentIndex]
    ]);
  }

  /**
   * @param {Uint8Array[]} arrays
   */
  function mergeArrays(arrays) {
    return arrays.reduce(function (memo, array) {
      var result = new Uint8Array(memo.length + array.length);
      result.set(memo);
      result.set(array, memo.length);
      return result;
    }, new Uint8Array());
  }

  function toPrefix(env, fragmentIndex) {
    return toBytes([
      'F', '2.2', env.$sessionId, env.$envelopeId, env.$fragmentCount,
      fragmentIndex, env.$unsolicited ? 'u' : 's', ''
    ].join(';'));
  }

  /**
   * @typedef module:baja/env/mux/BoxEnvelope~BoxFragment
   * @property {string} sessionId the server session id
   * @property {number} envelopeId the envelope id
   * @property {number} fragmentIndex fragment index within the envelope
   * @property {number} fragmentCount total fragment count of the envelope
   * @property {boolean} unsolicited true if this is an unsolicited envelope
   * sent down from the server
   * @property {Uint8Array} payload the fragment payload
   */

  return BoxEnvelope;
});
