/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/*jshint browser: true, devel: true */

/**
 * Reused WebSocket Connection.
 *
 * API Status: **Private**
 * @module baja/env/ReusedWebSocketConnection
 */
define([ "bajaScript/comm",
        "bajaScript/baja/comm/Callback",
        "bajaScript/env/Connection",
        "bajaScript/env/WebSocketConnection" ], function (
        baja,
        Callback,
        Connection,
        WebSocketConnection) {

  "use strict";

  /**
   * A connection that reuses an existing WebSocket for a connection.
   *
   * @class
   * @extends module:baja/env/Connection
   * @alias module:baja/env/ReusedWebSocketConnection
   * @param {module:baja/env/Connection~ConnectionEvents} handlers handlers
   * for connection events.
   * @param {module:baja/env/ConnectionManager} connectionManager the connection
   * manager responsible for processing BoxFrames via incoming websocket
   * messages
   * @param {WebSocket} internalConnection the existing parent connection to
   * reuse
   */
  var ReusedWebSocketConnection = function (handlers, connectionManager, internalConnection) {
    baja.callSuper(ReusedWebSocketConnection, this, arguments);

    if (!connectionManager) { throw new Error('ConnectionManager required'); }
    if (!internalConnection) { throw new Error('reused WebSocket required'); }

    this.$internalConnection = internalConnection;
  };

  baja.subclass(ReusedWebSocketConnection, WebSocketConnection);

  /**
   * Authenticates by creating a new server session.
   * @returns {Promise}
   */
  ReusedWebSocketConnection.prototype.doAuthenticate = function () {
    var cb = new baja.comm.Callback();
    baja.comm.makeServerSession(cb);
    return cb.promise();
  };

  /**
   * Piggyback onto all the event handlers from the parent connection.
   */
  ReusedWebSocketConnection.prototype.start = function () {
    var that = this,
        connection = that.$internalConnection,
        handlers = that.$handlers,
        name;
     
    for (name in handlers) {
      if (handlers.hasOwnProperty(name)) {
        connection.addEventListener(name, handlers[name], false);  
      }
    }

    // Check current state to see if the Socket is already open.
    if (connection.readyState === Connection.WS_OPEN) {
      handlers.open();
    }
  };

  ReusedWebSocketConnection.prototype.close = function () {
    var that = this,
        connection = that.$internalConnection,
        handlers = that.$handlers,
        name;

    for (name in handlers) {
      if (handlers.hasOwnProperty(name)) {
        connection.removeEventListener(name, handlers[name], false);  
      }
    }

    that.$handlers.close();
  };

  return ReusedWebSocketConnection;
});  
