/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

define('bajaPromises', [ 'Promise' ], function (Promise) {
  'use strict';

  function monitorState(promise) {
    var prom = promise
      .then(function (result) {
        markResolved(prom);
        return result;
      }, function (err) {
        markRejected(prom);
        throw err;
      });
    return prom;
  }

  function markResolved(prom) {
    prom.$rs = prom.$c = true;
    return prom;
  }

  function markRejected(prom) {
    prom.$rj = prom.$c = true;
    return prom;
  }

  // noinspection UnnecessaryLocalVariableJS
  /**
   * Promise wrapper that exports promise functions needed by BajaScript.
   * You can swap in your own implementation if you need to use a different
   * Promise library. It must implement this exact API.
   * 
   * Specify it in your RequireJS config as a RequireJS ID at
   * `baja.config.promises`.
   *
   * API Status: **Private**
   * 
   * @exports baja/env/Promises
   */
  var exports = {
    /**
     * Return a promise that has been fulfilled with the given value.
     * @param {*} obj
     * @returns {Promise}
     */
    resolve: function (obj) {
      return markResolved(Promise.resolve(obj));
    },

    /**
     * Return a promise that has been rejected with the given error.
     * @param {Error} err
     * @returns {Promise}
     */
    reject: function (err) {
      // eslint-disable-next-line promise/no-promise-in-callback
      return markRejected(Promise.reject(err));
    },

    /**
     * Return a Deferred object. It should have three methods:
     * `resolve` that resolves the promise, `reject` that rejects the promise,
     * and `promise` that returns the Promise itself.
     * @returns {Object}
     */
    deferred: function () {
      var df = {};
      // eslint-disable-next-line promise/avoid-new
      var promise = monitorState(new Promise(function (resolve, reject) {
        df.resolve = resolve; df.reject = reject;
      }));
      df.promise = function () { return promise; };
      return df;
    },

    /**
     * Return a promise that is resolved when all the elements of the array
     * (values or Promises) are resolved.
     * @param {Array.<*|Promise>} array
     * @returns {Promise}
     */
    all: function (array) {
      return Promise.all(array);
    },

    /**
     * Return true if the given promise has been resolved.
     * 
     * @param {Promise} prom
     * @returns {Boolean}
     */
    isResolved: function (prom) {
      return !!prom.$rs;
    },

    /**
     * Return true if the given promise has been rejected.
     * 
     * @param {Promise} prom
     * @returns {Boolean}
     */
    isRejected: function (prom) {
      return !!prom.$rj;
    },

    /**
     * Return true if the given promise is still pending.
     * 
     * @param {Promise} prom
     * @returns {Boolean}
     */
    isPending: function (prom) {
      return !prom.$c;
    }
  };

  return exports;
});
