/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * Defines {@link baja.VirtualComponent}.
 * @module baja/virt/VirtualComponent
 */
define(["bajaScript/sys",
        "bajaScript/baja/comp/Component",
        "bajaScript/baja/ord/Ord",
        "bajaScript/baja/ord/SlotPath",
        "bajaScript/baja/ord/VirtualPath"],
        function (baja, Component, Ord, SlotPath, VirtualPath) {

  "use strict";
  
  var subclass = baja.subclass,
      callSuper = baja.callSuper;

  /**
   * Represents a `baja:VirtualComponent` in BajaScript.
   *
   * @class
   * @alias baja.VirtualComponent
   * @extends baja.Component
   */  
  var VirtualComponent = function VirtualComponent() {
    callSuper(VirtualComponent, this, arguments);
  };
  
  subclass(VirtualComponent, Component);
  
  /**
   * Return the Nav ORD for the Virtual Component.
   *
   * @private
   *
   * @returns {baja.Ord} the Nav ORD or null if it's not mounted.
   */
  VirtualComponent.prototype.getNavOrd = function () {
    if (!this.isMounted()) {
      return null;
    }
    
    var spaceOrd = this.$space.getAbsoluteOrd(),
        body = "/",
        slotPath = this.getSlotPath(),
        i;
    
    for (i = 0; i < slotPath.depth(); ++i) {
      if (i > 0) {
        body += "/";
      }
      body += VirtualPath.toVirtualPathName(slotPath.nameAt(i));
    }
    
    return Ord.make(spaceOrd.toString() + body);
  };

  /**
   * Return the ORD in session for the Virtual Component.
   *
   * @private
   *
   * @returns {baja.Ord} the ORD in session or null if it's not mounted.
   */
  VirtualComponent.prototype.getOrdInSession = function () {
    if (!this.isMounted()) {
      return null;
    }

    return this.getNavOrd().relativizeToSession();
  };

  /**
   * Return the Nav Display Name for the Virtual Component.
   *
   * @private
   *
   * @returns {String}
   */
  VirtualComponent.prototype.getNavDisplayName = function () {
    var space = this.getComponentSpace();

    // If this is the root component of the virtual space, use the virtual gateway's display name
    if (space && space.$gateway && space.getRootComponent() === this) {
      return space.$gateway.getNavDisplayName();
    }

    return callSuper("getNavDisplayName", VirtualComponent, this, arguments);
  };


  /**
   * Return the Nav Icon for this Virtual Component.
   *
   * @private
   *
   * @returns {baja.Icon}
   */
  VirtualComponent.prototype.getNavIcon = function () {
    var space = this.getComponentSpace();

    // If this is the root component of the virtual space, use the virtual gateway's nav icon
    if (space && space.$gateway && space.getRootComponent() === this) {
      return space.$gateway.getNavIcon();
    }

    return callSuper("getNavIcon", VirtualComponent, this, arguments);
  };

  return VirtualComponent;
});