/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * @module baja/tag/SmartTags
 */
define(["bajaScript/sys",
        "bajaScript/baja/tag/TagSet"], function (
        baja,
        TagSet) {

  "use strict";

  /**
   * SmartTags holds a collection of both direct and implemented
   * {@link module:baja/tag/Tag} objects.
   *
   * @class
   * @alias module:baja/tag/SmartTags
   *
   * @param directTags A direct Tags implementation.
   * @param impliedTags An implied Tags implementation.
   */
  var SmartTags = function SmartTags(directTags, impliedTags) {
    this.$directTags = directTags || new TagSet();
    this.$impliedTags = impliedTags || new TagSet();
  };

  /**
   * @returns {Boolean} Returns true if there are no Tag objects.
   */
  SmartTags.prototype.isEmpty = function() {
    return this.$directTags.isEmpty() && this.$impliedTags.isEmpty();
  };

  /**
   * Get the value for the specified tag.
   *
   * @param  {String|module:baja/tag/Id} id The Id
   * used for the search. This can be an Id or a qname for an Id.
   * @returns The value for the tag or null if nothing can be found.
   */
  SmartTags.prototype.get = function(id) {
    var v = null;
    try {
      v = this.$directTags.get(id);
    } catch (e) {
      baja.outln(e);
    }
  
    try {
      if (v === null) {
        v = this.$impliedTags.get(id);
      }
    } catch (e) {
      baja.outln(e);
    }

    return v;
  };

  /**
   * Returns true a Tag with the specified Id (or qname)
   * is found.
   *
   * @param  {String|module:baja/tag/Id} id The Id
   * used for the search. This can be an Id or a qname for an Id.
   * @returns {Boolean} Returns true if found.
   */
  SmartTags.prototype.contains = function(id) {
    var directContains = false;
    var impliedContains = false;
    try {
      directContains = this.$directTags.contains(id);
    } catch(e) {
      baja.outln(e);
    }
    try {
      impliedContains = this.$impliedTags.contains(id);
    } catch(e) {
      baja.outln(e);
    }
    return directContains || impliedContains;
  };

  /**
   * Returns a copy of the contained Tags array.
   *
   * Any duplicate tags are filtered out.
   *
   * @returns {Array<module:baja/tag/Tag>} An array of Tag objects.
   */
  SmartTags.prototype.getAll = function() {
    var tags = this.$impliedTags.getAll(),
        directTags = this.$directTags.getAll(),
        i;

    for (i = 0; i < directTags.length; ++i) {
      if (!this.$impliedTags.get(directTags[i].getId())) {
        tags.push(directTags[i]);
      }
    }

    return tags;
  };

  /**
   * Return the direct tags.
   *
   * @returns The direct Tags.
   */
  SmartTags.prototype.getDirectTags = function () {
    return this.$directTags;
  };

  /**
   * Return the implied tags.
   *
   * @returns the implied tags.
   */
  SmartTags.prototype.getImpliedTags = function () {
    return this.$impliedTags;
  };

  return SmartTags;
});
