/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * Defines {@link baja.VirtualPath}.
 * @module baja/ord/VirtualPath
 */
define(["bajaScript/sys",
        "bajaScript/baja/ord/SlotPath"], function (
          baja,
          SlotPath) {

  "use strict";
  
  var subclass = baja.subclass,
      callSuper = baja.callSuper;
  
  /**
   * Resolves Virtual Slot Paths.
   *
   * @class
   * @alias baja.VirtualPath
   * @extends baja.SlotPath
   */  
  var VirtualPath = function VirtualPath() {
    callSuper(VirtualPath, this, arguments);
  };
  
  subclass(VirtualPath, SlotPath);
  
  /**
   * Make a Slot Path.
   *
   * @private
   *
   * @param {String} query  the ORD Query body used in resolving the ORD.
   * @returns {baja.VirtualPath} the new Slot Path.
   */
  VirtualPath.prototype.makeSlotPath = function (query) {
    return new VirtualPath(query);
  };
  
  /**
   * Return whether the specified path name is valid.
   * 
   * @param {String} pathName the path name to validate.
   *
   * @returns {Boolean} true if the slot name is valid.
   */ 
  VirtualPath.prototype.isValidPathName = function (pathName) {
    return VirtualPath.isValidName(pathName);
  };

  /**
   * Return whether the slot name is valid.
   *
   * @param {String} nm the name to validate.
   *
   * @returns {Boolean} true if the slot name is valid.
   */
  VirtualPath.isValidName = function (nm) {
    return (/^[^\/\|\$\:]+$/).test(nm);
  };

  /**
   * Converts the given SlotPath name to a valid VirtualPath name.  The SlotPath name argument
   * must be in its proper slot escaped form - don't call SlotPath.unescape() on the SlotPath name
   * argument before calling this method.  This method will unescape the given SlotPath name and
   * look for any unsupported virtual characters ('/', '|', '$', ':', and the '~' escape character
   * itself) and escape those using the '~' escape character.
   *
   * @see baja.VirtualPath.toSlotPathName
   *
   * @param {String} slotPathName The SlotPath name to be converted to a valid VirtualPath name.
   * This argument must be in its proper slot escaped form - don't call SlotPath.unescape() on this
   * SlotPath name argument before passing it to this method.
   *
   * @returns {String} A valid VirtualPath name where any unsupported virtual characters have been
   * escaped using the '~' escape character.
   *
   * @since Niagara 4.6
   */
  VirtualPath.toVirtualPathName = function (slotPathName) {
    // While this code is very similar to SlotPath.unescape(), it has some significant differences,
    // so it didn't make sense to attempt to reuse common code versus just having a little duplicate
    // code in both places.
    if (!slotPathName) {
      return slotPathName;
    }

    // Convert from $xx
    slotPathName = slotPathName.replace(/\$[0-9a-fA-F]{2}/g, function (s) {
      var charCodeStr = s.substring(1, s.length),
        charStr = String.fromCharCode(parseInt(charCodeStr, 16));

      // Check for unsupported virtual characters and escape them
      switch(charStr) {
        case '/':
        case '|':
        case '$':
        case ':':
        case '~':
          return '~' + charCodeStr;
        default:
          return charStr;
      }
    });

    // Convert from $uxxxx (don't need to worry about unsupported virtual characters here)
    slotPathName = slotPathName.replace(/\$u[0-9a-fA-F]{4}/g, function (s) {
      return String.fromCharCode(parseInt(s.substring(2, s.length), 16));
    });

    return slotPathName;
  };

  /**
   * Converts the given VirtualPath name to a valid SlotPath name. This method will unescape any
   * unsupported virtual characters ('/', '|', '$', ':', and the '~' escape character itself) in the
   * given VirtualPath name that were previously escaped using the '~' escape character.  It will
   * also ensure that the result is in proper SlotPath escaped form before returning it.
   *
   * In order to support legacy uses of the '~' escape character in pre-4.6 VirtualPath names,
   * this method will only consider '~' as an escape character if it is followed by the code for
   * one of the unsupported virtual characters ('/', '|', '$', ':', and the '~' escape character
   * itself) in the given VirtualPath name.
   *
   * @see baja.VirtualPath.toVirtualPathName
   *
   * @param {String} virtualPathName The VirtualPath name to be converted to a valid SlotPath name.
   *
   * @returns {String} A valid SlotPath name.
   *
   * @since Niagara 4.6
   */
  VirtualPath.toSlotPathName = function (virtualPathName) {
    if (!virtualPathName) {
      return virtualPathName;
    }

    // Convert from ~xx, but only for the unsupported char codes!
    return SlotPath.escape(virtualPathName.replace(/\~3a|\~2f|\~24|\~7e|\~7c/g, function (s) {
      return String.fromCharCode(parseInt(s.substring(1, s.length), 16));
    }));
  };

  /**
   * Return the scheme name for the SlotPath.
   *
   * @returns {String}
   */
  VirtualPath.prototype.getSchemeName = function () {
    return "virtual";
  };

  /**
   * Return the scheme used with this SlotPath.
   *
   * @returns {baja.VirtualScheme}
   */
  VirtualPath.prototype.getScheme = function () {
    return baja.VirtualScheme.DEFAULT;
  };

  return VirtualPath;
});