/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * Defines {@link baja.UnknownScheme}.
 * @module baja/ord/UnknownScheme
 */
define([
  "bajaScript/sys",
  "bajaScript/baja/ord/OrdScheme",
  "bajaScript/baja/ord/OrdTarget" ], function (
  baja,
  OrdScheme,
  OrdTarget) {
  
  "use strict";
  
  var subclass = baja.subclass,
      callSuper = baja.callSuper;
  
  /**
   * Unknown ORD Scheme. 
   * 
   * Some ORD Schemes are represented in BajaScript and some are not. When an 
   * ORD is resolved, the BajaScript Registry is used to see if we locally 
   * have an ORD scheme representation. If all of the ORD Schemes in an ORD 
   * do have a local representation (i.e. they have JS Constructors), the ORD 
   * is resolved locally. If any unknown ORD schemes are found then the entire 
   * ORD is resolved on the server and the corresponding results are 
   * serialized and sent back down to the client.
   *
   * @class
   * @alias baja.UnknownScheme
   * @extends baja.OrdScheme
   * @private
   */
  var UnknownScheme = function UnknownScheme() {
    callSuper(UnknownScheme, this, arguments);
  };
  
  subclass(UnknownScheme, OrdScheme);
  
  /**
   * Default Unknown ORD Scheme instance.
   * @private
   * @type {baja.UnknownScheme}
   */
  UnknownScheme.DEFAULT = new UnknownScheme();

  /**
   * @returns {boolean} false - an unknown scheme cannot be resolved in the
   * client
   */
  UnknownScheme.prototype.isClientResolvable = function () {
    return false;
  };
  
  /**
   * Called when an ORD is resolved.
   *
   * @private
   *
   * @see baja.OrdScheme#resolve
   *
   * @param {module:baja/ord/OrdTarget} target  the current ORD Target.
   * @param {baja.OrdQuery} query  the ORD Query used in resolving the ORD.
   * @param {module:baja/ord/OrdQueryListCursor} cursor  the ORD Query List 
   * cursor used for helping to asynchronously resolve the ORD.
   * @param {Object} options  options used for resolving an ORD.
   */
  UnknownScheme.prototype.resolve = function (target, query, cursor, options) {
    
    // Fail since this should always be resolved Server side
    new OrdTarget(target);
    options.callback.fail("Unknown BajaScript ORD Scheme: " + query.getSchemeName());
  };
  
  return UnknownScheme;
});
