/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * Defines {@link baja.HandleScheme}.
 * @module baja/ord/HandleScheme
 */
define(["bajaScript/sys",
        "bajaScript/baja/ord/OrdScheme",
        "bajaScript/baja/ord/OrdTarget"], 
    function (baja, OrdScheme, OrdTarget) {
  
  "use strict";
  
  var subclass = baja.subclass,
      callSuper = baja.callSuper;
  
  /**
   * Handle ORD Scheme.
   * 
   * This scheme resolves a `SlotPath` to a handle. Each `Component` in a 
   * `ComponentSpace` has a unique handle. This is a great way to keep track 
   * of a `Component` regardless of whether its `SlotPath` changes.
   *
   * @class
   * @alias baja.HandleScheme
   * @extends baja.OrdScheme
   * @private
   */  
  var HandleScheme = function HandleScheme() {
    callSuper(HandleScheme, this, arguments);
  };
  
  subclass(HandleScheme, OrdScheme);
  
  /**
   * Default Handle ORD Scheme instance
   * @private
   * @type {baja.HandleScheme}
   */
  HandleScheme.DEFAULT = new HandleScheme();
     
  /**
   * Called when an ORD is resolved.
   *
   * @private
   *
   * @see baja.OrdScheme#resolve
   *
   * @param {module:baja/ord/OrdTarget} target  the current ORD Target.
   * @param {baja.OrdQuery} query  the ORD Query used in resolving the ORD.
   * @param {module:baja/ord/OrdQueryListCursor} cursor  the ORD Query List 
   * cursor used for helping to asynchronously resolve the ORD.
   * @param {Object} options  options used for resolving an ORD.
   */ 
  HandleScheme.prototype.resolve = function (target, query, cursor, options) {
    
    var object = target.object,
        handle = query.getBody(),
        space;
        
    if (!baja.hasType(object)) {
      throw new Error("Not based via ComponentSpace. Invalid Object");
    }
    else if (object.getType().isComponent()) {
      space = object.getComponentSpace();
    }
    else if (object.getType().is("baja:ComponentSpace")) {
      space = object;
    }
    
    // Pick up whether the Space is null
    if (!baja.hasType(space)) {
      throw new Error("Not based via ComponentSpace");
    }
    
    // Resolve the handle in the Space (may make network calls if necessary)
    space.resolveByHandle({
      "handle": handle, 
      "ok": function ok(comp) {
        var newTarget = new OrdTarget(target);
        newTarget.object = comp;
        cursor.resolveNext(newTarget, options);
      }, 
      "fail": function fail(err) {
        options.callback.fail(err);
      }
    });
  };

  return HandleScheme;
});