/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * @private
 * @module baja/obj/WsAnnotation
 */
define(['bajaScript/sys',
        'bajaScript/baja/obj/Simple',
        'bajaScript/baja/obj/objUtil'], function (
         baja,
         Simple,
         objUtil) {

  'use strict';

  var bajaDef = baja.def,
      cacheDecode = objUtil.cacheDecode,
      cacheEncode = objUtil.cacheEncode;

  /**
   * BajaScript representation of a `baja:WsAnnotation` value.
   *
   * @private
   * @class
   * @extends {baja.Simple}
   * @alias {module:baja/obj/WsAnnotation}
   */
  var WsAnnotation = function WsAnnotation(x, y, width, height) {
    Simple.apply(this, arguments);
    this.$x = bajaDef(x, 0);
    this.$y = bajaDef(y, 0);
    this.$width = bajaDef(width, 8);
    this.$height = bajaDef(height, 0);
  };
  WsAnnotation.prototype = Object.create(Simple.prototype);
  WsAnnotation.prototype.constructor = WsAnnotation;

  /**
   * Create a new `baja:WsAnnotation` instance from the arguments.
   * @param {Number} x x in wixels
   * @param {Number} y y in wixels
   * @param {Number} width width in wixels
   * @param {Number} height height in wixels
   * @returns {module:baja/obj/WsAnnotation}
   */
  WsAnnotation.make = function (x, y, width, height) {
    if (arguments.length === 1 && typeof x === 'string') {
      return WsAnnotation.make.apply(null, x.split(',').map(parseFloat));
    }
    return new WsAnnotation(x, y, width, height);
  };

  /**
   * @see .make
   * @returns {module:baja/obj/WsAnnotation}
   */
  WsAnnotation.prototype.make = function () {
    return WsAnnotation.make.apply(WsAnnotation, arguments);
  };

  /**
   * Merge two annotations together.
   *
   * @param {module:baja/obj/WsAnnotation} a1
   * @param {module:baja/obj/WsAnnotation} a2
   * @returns {module:baja/obj/WsAnnotation}
   */
  WsAnnotation.merge = function (a1, a2) {
    if (!a1 && !a2) { return WsAnnotation.DEFAULT; }
    if (!a1) { return a2; }
    if (!a2) { return a1; }
    return a2.merge(a1);
  };

  /**
   * Merge another annotation in with this one.
   *
   * @param {module:baja/obj/WsAnnotation} other
   * @returns {module:baja/obj/WsAnnotation} a new annotation, same X and Y as
   * this one, but expanded to be large enough to contain the other's width
   * and height
   */
  WsAnnotation.prototype.merge = function (other) {
    return WsAnnotation.make(
      this.getX(),
      this.getY(),
      Math.max(this.getWidth(), other.getWidth()),
      Math.max(this.getHeight(), other.getHeight())
    );
  };

  function die(str) {
    throw new Error('invalid WsAnnotation format "' + str + '"');
  }

  /**
   * Decode an `WsAnnotation` from a string.
   *
   * @function
   * @param {String} str
   * @returns {module:baja/obj/WsAnnotation}
   */
  WsAnnotation.prototype.decodeFromString = cacheDecode(function (str) {
    var split = str.split(',');

    if (split.length < 3) { die(str); }

    return WsAnnotation.make.apply(null, split.map(function (s) {
      var num = parseInt(s, 10);
      if (isNaN(num)) { die(str); }
      return num;
    }));
  });

  /**
   * Encode this `WsAnnotation` to a string.
   *
   * @function
   * @returns {String}
   */
  WsAnnotation.prototype.encodeToString = cacheEncode(function () {
    var s = this.$x + ',' + this.$y + ',' + this.$width,
      height = this.$height;
    if (height) { s += ',' + height; }
    return s;
  });

  /**
   * @returns {Number} x in wixels
   */
  WsAnnotation.prototype.getX = function () { return this.$x; };

  /**
   * @returns {Number} y in wixels
   */
  WsAnnotation.prototype.getY = function () { return this.$y; };

  /**
   * @returns {Number} width in wixels
   */
  WsAnnotation.prototype.getWidth = function () { return this.$width; };

  /**
   * @returns {Number} height in wixels
   */
  WsAnnotation.prototype.getHeight = function () { return this.$height; };


  WsAnnotation.DEFAULT = WsAnnotation.make();

  return WsAnnotation;
});
